<?php
session_start();
header('Content-Type: application/json');

require_once 'db_connect.php';

// Initialize session cart if not exists
if (!isset($_SESSION['cart'])) {
    $_SESSION['cart'] = [];
}

$action = $_GET['action'] ?? $_POST['action'] ?? '';
$response = [];

try {
    // ===== 1. FETCH CART FROM SESSION =====
    if ($action == 'fetch') {
        $response = [
            'status' => 'success',
            'items' => $_SESSION['cart'],
            'count' => count($_SESSION['cart']),
            'subtotal' => calculateSubtotal($_SESSION['cart'])
        ];
    }

    // ===== 2. ADD ITEM TO CART =====
    elseif ($action == 'add') {
        $item_id = $_POST['id'] ?? '';
        $name = $_POST['name'] ?? '';
        $price = floatval($_POST['price'] ?? 0);
        $image = $_POST['image'] ?? 'assets/images/placeholder.png';
        $quantity = intval($_POST['quantity'] ?? 1);

        if (empty($name) || $price <= 0) {
            throw new Exception('Invalid item data');
        }

        // Check if item already exists in cart
        $found = false;
        foreach ($_SESSION['cart'] as &$item) {
            if ($item['name'] === $name) {
                $item['quantity'] += $quantity;
                $found = true;
                break;
            }
        }

        // Add new item if not found
        if (!$found) {
            $_SESSION['cart'][] = [
                'id' => $item_id,
                'name' => $name,
                'price' => $price,
                'image' => $image,
                'quantity' => $quantity,
                'added_at' => date('Y-m-d H:i:s')
            ];
        }

        $response = [
            'status' => 'success',
            'message' => 'Item added to cart',
            'items' => $_SESSION['cart'],
            'count' => count($_SESSION['cart'])
        ];
    }

    // ===== 3. UPDATE ITEM QUANTITY =====
    elseif ($action == 'update_quantity') {
        $name = $_POST['name'] ?? '';
        $quantity = intval($_POST['quantity'] ?? 0);

        if (empty($name)) {
            throw new Exception('Item name is required');
        }

        foreach ($_SESSION['cart'] as &$item) {
            if ($item['name'] === $name) {
                if ($quantity <= 0) {
                    // Remove item if quantity is 0 or less
                    $_SESSION['cart'] = array_values(array_filter($_SESSION['cart'], function($i) use ($name) {
                        return $i['name'] !== $name;
                    }));
                } else {
                    $item['quantity'] = $quantity;
                }
                break;
            }
        }

        $response = [
            'status' => 'success',
            'message' => 'Quantity updated',
            'items' => $_SESSION['cart'],
            'subtotal' => calculateSubtotal($_SESSION['cart'])
        ];
    }

    // ===== 4. REMOVE ITEM FROM CART =====
    elseif ($action == 'remove') {
        $name = $_POST['name'] ?? '';

        if (empty($name)) {
            throw new Exception('Item name is required');
        }

        $_SESSION['cart'] = array_values(array_filter($_SESSION['cart'], function($item) use ($name) {
            return $item['name'] !== $name;
        }));

        $response = [
            'status' => 'success',
            'message' => 'Item removed from cart',
            'items' => $_SESSION['cart'],
            'count' => count($_SESSION['cart']),
            'subtotal' => calculateSubtotal($_SESSION['cart'])
        ];
    }

    // ===== 5. CLEAR ENTIRE CART =====
    elseif ($action == 'clear') {
        $_SESSION['cart'] = [];
        $response = [
            'status' => 'success',
            'message' => 'Cart cleared',
            'items' => [],
            'count' => 0
        ];
    }

    // ===== 6. GET CART COUNT =====
    elseif ($action == 'count') {
        $response = [
            'status' => 'success',
            'count' => count($_SESSION['cart']),
            'subtotal' => calculateSubtotal($_SESSION['cart'])
        ];
    }

    // ===== 7. EXPORT CART TO LOCALSTORAGE =====
    elseif ($action == 'export') {
        $response = [
            'status' => 'success',
            'items' => $_SESSION['cart'],
            'count' => count($_SESSION['cart']),
            'subtotal' => calculateSubtotal($_SESSION['cart']),
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }

    else {
        throw new Exception('Invalid action: ' . $action);
    }

} catch (Exception $e) {
    $response = [
        'status' => 'error',
        'message' => $e->getMessage()
    ];
    http_response_code(400);
}

echo json_encode($response);

// ===== HELPER FUNCTION =====
function calculateSubtotal($cart) {
    $subtotal = 0;
    foreach ($cart as $item) {
        $subtotal += ($item['price'] * $item['quantity']);
    }
    return round($subtotal, 2);
}
?>
