<?php
// =====================================================
// ADMIN WEEKLY SPECIALS MANAGEMENT PAGE
// =====================================================
require_once 'check_admin.php'; // Ensure admin is logged in

// =====================================================
// DATABASE TABLE CREATION (if not exists)
// =====================================================
$create_table_sql = "CREATE TABLE IF NOT EXISTS `weekly_specials` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `title` VARCHAR(255) NOT NULL,
  `description` TEXT NOT NULL,
  `price` VARCHAR(100) NOT NULL,
  `category` VARCHAR(50) NOT NULL,
  `image_path` VARCHAR(255) NOT NULL,
  `badge_type` ENUM('hot', 'new', 'offer', 'none') DEFAULT 'none',
  `created_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";

if (!$conn->query($create_table_sql)) {
    error_log("Table creation error: " . $conn->error);
}

$message = "";
$error = "";

// =====================================================
// PROCESS FORM SUBMISSIONS
// =====================================================

// ADD WEEKLY SPECIAL
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add') {
    $title = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $price = trim($_POST['price'] ?? '');
    $category = trim($_POST['category'] ?? '');
    $badge_type = trim($_POST['badge_type'] ?? 'none');
    
    if (empty($title) || empty($description) || empty($price) || empty($category)) {
        $error = "All fields are required!";
    } elseif (!isset($_FILES['image']) || $_FILES['image']['error'] !== UPLOAD_ERR_OK) {
        $error = "Please select a valid image file!";
    } else {
        $file = $_FILES['image'];
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        $max_size = 5 * 1024 * 1024; // 5MB
        
        if (!in_array($file['type'], $allowed_types) || $file['size'] > $max_size) {
            $error = "Invalid file type or size (Max 5MB, JPG/PNG/WEBP only).";
        } else {
            $upload_dir = '../assets/images/weekly/';
            if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);
            
            $file_ext = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = time() . '_' . bin2hex(random_bytes(5)) . '.' . $file_ext;
            $filepath = $upload_dir . $filename;
            $image_path = 'assets/images/weekly/' . $filename; 
            
            if (move_uploaded_file($file['tmp_name'], $filepath)) {
                $stmt = $conn->prepare("INSERT INTO weekly_specials (title, description, price, category, image_path, badge_type) VALUES (?, ?, ?, ?, ?, ?)");
                $stmt->bind_param("ssssss", $title, $description, $price, $category, $image_path, $badge_type);
                if ($stmt->execute()) {
                    $message = "Weekly special added successfully!";
                } else {
                    $error = "Database error: " . $stmt->error;
                    unlink($filepath);
                }
                $stmt->close();
            } else {
                $error = "Failed to upload image.";
            }
        }
    }
}

// UPDATE WEEKLY SPECIAL
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update') {
    $id = intval($_POST['id'] ?? 0);
    $title = trim($_POST['title'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $price = trim($_POST['price'] ?? '');
    $category = trim($_POST['category'] ?? '');
    $badge_type = trim($_POST['badge_type'] ?? 'none');
    
    if ($id > 0 && !empty($title)) {
        if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
            // New Image Upload Logic
            $file = $_FILES['image'];
            $upload_dir = '../assets/images/weekly/';
            $file_ext = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = time() . '_' . bin2hex(random_bytes(5)) . '.' . $file_ext;
            $filepath = $upload_dir . $filename;
            $image_path = 'assets/images/weekly/' . $filename;
            
            if (move_uploaded_file($file['tmp_name'], $filepath)) {
                $stmt = $conn->prepare("UPDATE weekly_specials SET title=?, description=?, price=?, category=?, image_path=?, badge_type=? WHERE id=?");
                $stmt->bind_param("ssssssi", $title, $description, $price, $category, $image_path, $badge_type, $id);
                $stmt->execute();
                $message = "Special updated successfully!";
            }
        } else {
            // Update without image
            $stmt = $conn->prepare("UPDATE weekly_specials SET title=?, description=?, price=?, category=?, badge_type=? WHERE id=?");
            $stmt->bind_param("sssssi", $title, $description, $price, $category, $badge_type, $id);
            if ($stmt->execute()) $message = "Special updated successfully!";
            else $error = "Update failed.";
        }
    }
}

// DELETE WEEKLY SPECIAL
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {
    $id = intval($_POST['id'] ?? 0);
    if ($id > 0) {
        $stmt = $conn->prepare("SELECT image_path FROM weekly_specials WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
        $res = $stmt->get_result();
        if ($row = $res->fetch_assoc()) {
            $file_path = '../' . $row['image_path'];
            $conn->query("DELETE FROM weekly_specials WHERE id=$id");
            if (file_exists($file_path)) unlink($file_path);
            $message = "Special deleted successfully!";
        }
    }
}

// FETCH DATA
$weekly_specials = [];
$result = $conn->query("SELECT * FROM weekly_specials ORDER BY created_date DESC");
while ($row = $result->fetch_assoc()) $weekly_specials[] = $row;

// STATISTICS
$stats = [
    'total' => count($weekly_specials),
    'hot' => 0, 'new' => 0, 'offer' => 0
];
foreach($weekly_specials as $s) {
    if($s['badge_type'] == 'hot') $stats['hot']++;
    if($s['badge_type'] == 'new') $stats['new']++;
    if($s['badge_type'] == 'offer') $stats['offer']++;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Weekly Specials - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="assets/css/admin-style.css">
    <style>
        .weekly-thumb { width: 60px; height: 60px; object-fit: cover; border-radius: 8px; border: 2px solid #eee; }
        .file-upload-wrapper { border: 2px dashed #ccc; padding: 20px; text-align: center; background: #f9f9f9; border-radius: 8px; cursor: pointer; transition: 0.3s; }
        .file-upload-wrapper:hover { border-color: #931A2A; background: #fff5f5; }
        .file-upload-wrapper input { display: none; }
    </style>
</head>
<body>

<div class="wrapper">
    
    <?php include 'sidebar.php'; ?>

    <div class="main-content">
        <nav class="top-nav">
            <div class="nav-left">
                <h2><i class="fas fa-star"></i> Weekly Specials</h2>
            </div>
            <div class="nav-right">
                <span class="admin-name">Welcome, Admin</span>
            </div>
        </nav>

        <div class="content">
            
            <?php if ($message): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
            <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <div class="row mb-4">
                <div class="col-md-3">
                    <div class="stat-card total" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;">
                        <div class="stat-icon mb-2"><i class="fas fa-list fa-2x opacity-50"></i></div>
                        <h6>Total Specials</h6>
                        <p class="count mb-0 fw-bold fs-3"><?php echo $stats['total']; ?></p>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-card" style="background: linear-gradient(135deg, #ff5858 0%, #f09819 100%); color: white;">
                        <div class="stat-icon mb-2"><i class="fas fa-fire fa-2x opacity-50"></i></div>
                        <h6>Hot Items</h6>
                        <p class="count mb-0 fw-bold fs-3"><?php echo $stats['hot']; ?></p>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-card" style="background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%); color: white;">
                        <div class="stat-icon mb-2"><i class="fas fa-star fa-2x opacity-50"></i></div>
                        <h6>New Arrivals</h6>
                        <p class="count mb-0 fw-bold fs-3"><?php echo $stats['new']; ?></p>
                    </div>
                </div>
                <div class="col-md-3">
                    <div class="stat-card" style="background: linear-gradient(135deg, #fa709a 0%, #fee140 100%); color: white;">
                        <div class="stat-icon mb-2"><i class="fas fa-tags fa-2x opacity-50"></i></div>
                        <h6>Offers</h6>
                        <p class="count mb-0 fw-bold fs-3"><?php echo $stats['offer']; ?></p>
                    </div>
                </div>
            </div>

            <div class="admin-form-section d-flex justify-content-between align-items-center mb-4 p-3">
                <h5 class="mb-0 text-muted">Manage Specials</h5>
                <button class="btn btn-primary-custom" data-bs-toggle="modal" data-bs-target="#addModal">
                    <i class="fas fa-plus"></i> Add New Special
                </button>
            </div>

            <div class="table-responsive">
                <table class="table table-modern align-middle table-hover">
                    <thead>
                        <tr>
                            <th>Image</th>
                            <th>Title</th>
                            <th>Description</th>
                            <th>Price</th>
                            <th>Badge</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($weekly_specials as $special): ?>
                        <tr>
                            <td>
                                <img src="../<?php echo htmlspecialchars($special['image_path']); ?>" class="weekly-thumb">
                            </td>
                            <td class="fw-bold"><?php echo htmlspecialchars($special['title']); ?></td>
                            <td class="text-muted small"><?php echo htmlspecialchars(substr($special['description'], 0, 50)) . '...'; ?></td>
                            <td class="fw-bold text-success"><?php echo htmlspecialchars($special['price']); ?></td>
                            <td>
                                <?php 
                                    $badgeClass = 'bg-secondary';
                                    if($special['badge_type']=='hot') $badgeClass='bg-danger';
                                    if($special['badge_type']=='new') $badgeClass='bg-primary';
                                    if($special['badge_type']=='offer') $badgeClass='bg-success';
                                ?>
                                <span class="badge <?php echo $badgeClass; ?> text-uppercase"><?php echo $special['badge_type']; ?></span>
                            </td>
                            <td>
                                <div class="action-buttons">
                                    <button class="btn btn-sm btn-info text-white" 
                                            onclick="loadEditForm(
                                                <?php echo $special['id']; ?>, 
                                                '<?php echo addslashes($special['title']); ?>', 
                                                '<?php echo addslashes($special['description']); ?>', 
                                                '<?php echo addslashes($special['price']); ?>', 
                                                '<?php echo $special['category']; ?>', 
                                                '<?php echo $special['badge_type']; ?>'
                                            )" 
                                            data-bs-toggle="modal" data-bs-target="#editModal">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    
                                    <form method="POST" style="display:inline;" onsubmit="return confirm('Delete this item?');">
                                        <input type="hidden" name="action" value="delete">
                                        <input type="hidden" name="id" value="<?php echo $special['id']; ?>">
                                        <button class="btn btn-sm btn-danger"><i class="fas fa-trash"></i></button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>

        </div> </div> </div> <div class="modal fade" id="addModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header" style="background: #931A2A; color: white;">
                <h5 class="modal-title">Add New Special</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" enctype="multipart/form-data">
                <div class="modal-body">
                    <input type="hidden" name="action" value="add">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Title</label>
                            <input type="text" name="title" class="form-control" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Price</label>
                            <input type="text" name="price" class="form-control" placeholder="e.g. ₹299" required>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Description</label>
                            <textarea name="description" class="form-control" rows="2" required></textarea>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Category</label>
                            <select name="category" class="form-select" required>
                                <option value="breakfast">Breakfast</option>
                                <option value="lunch">Lunch</option>
                                <option value="dinner">Dinner</option>
                                <option value="special">Special</option>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Badge</label>
                            <select name="badge_type" class="form-select">
                                <option value="none">None</option>
                                <option value="hot">HOT 🔥</option>
                                <option value="new">NEW ⭐</option>
                                <option value="offer">OFFER 💰</option>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Upload Image</label>
                            <div class="file-upload-wrapper" onclick="document.getElementById('addImage').click()">
                                <input type="file" id="addImage" name="image" accept="image/*" required>
                                <i class="fas fa-cloud-upload-alt fa-2x text-muted"></i>
                                <p class="mb-0 text-muted">Click to upload image</p>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary-custom">Save Special</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="editModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header" style="background: #931A2A; color: white;">
                <h5 class="modal-title">Edit Special</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" enctype="multipart/form-data">
                <div class="modal-body">
                    <input type="hidden" name="action" value="update">
                    <input type="hidden" name="id" id="editId">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Title</label>
                            <input type="text" name="title" id="editTitle" class="form-control" required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Price</label>
                            <input type="text" name="price" id="editPrice" class="form-control" required>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Description</label>
                            <textarea name="description" id="editDescription" class="form-control" rows="2" required></textarea>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Category</label>
                            <select name="category" id="editCategory" class="form-select" required>
                                <option value="breakfast">Breakfast</option>
                                <option value="lunch">Lunch</option>
                                <option value="dinner">Dinner</option>
                                <option value="special">Special</option>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Badge</label>
                            <select name="badge_type" id="editBadge" class="form-select">
                                <option value="none">None</option>
                                <option value="hot">HOT 🔥</option>
                                <option value="new">NEW ⭐</option>
                                <option value="offer">OFFER 💰</option>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">Update Image (Optional)</label>
                            <div class="file-upload-wrapper" onclick="document.getElementById('editImage').click()">
                                <input type="file" id="editImage" name="image" accept="image/*">
                                <i class="fas fa-cloud-upload-alt fa-2x text-muted"></i>
                                <p class="mb-0 text-muted">Click to change image</p>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="submit" class="btn btn-primary-custom">Update Special</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
<script>
    function loadEditForm(id, title, desc, price, cat, badge) {
        document.getElementById('editId').value = id;
        document.getElementById('editTitle').value = title;
        document.getElementById('editDescription').value = desc;
        document.getElementById('editPrice').value = price;
        document.getElementById('editCategory').value = cat;
        document.getElementById('editBadge').value = badge;
    }
</script>
</body>
</html>