<?php
require_once 'check_admin.php';

// Ensure the 'image' column exists in your database table 'menu_items'
// Run this SQL once if needed: ALTER TABLE menu_items ADD COLUMN image VARCHAR(255);

$message = "";
$error = "";

// --- HANDLE DELETE ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action']) && $_POST['action'] == 'delete') {
    $menu_id = intval($_POST['menu_id']);
    
    // Get image path to delete file
    $stmt = $conn->prepare("SELECT image FROM menu_items WHERE id = ?");
    $stmt->bind_param("i", $menu_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $file_path = '../' . $row['image']; // Adjust path for admin folder
        if (file_exists($file_path) && !empty($row['image'])) {
            unlink($file_path);
        }
    }
    $stmt->close();

    $stmt = $conn->prepare("DELETE FROM menu_items WHERE id = ?");
    $stmt->bind_param("i", $menu_id);
    if ($stmt->execute()) {
        $message = "Menu item deleted successfully!";
    } else {
        $error = "Failed to delete menu item.";
    }
}

// --- HANDLE ADD / EDIT ---
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['action']) && ($_POST['action'] == 'add' || $_POST['action'] == 'edit')) {
    $name = trim($_POST['name']);
    $main_category = trim($_POST['main_category']);
    $sub_category = trim($_POST['sub_category']);
    $description = trim($_POST['description']);
    $price = floatval($_POST['price']);
    $image_db_path = '';

    if (empty($name) || empty($main_category) || empty($sub_category) || empty($price)) {
        $error = "Please fill all required fields.";
    } else {
        // --- IMAGE UPLOAD LOGIC ---
        if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
            $file = $_FILES['image'];
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            $max_size = 5 * 1024 * 1024; // 5MB
            
            if (!in_array($file['type'], $allowed_types)) {
                $error = "Only JPG, PNG, GIF, and WebP images are allowed!";
            } elseif ($file['size'] > $max_size) {
                $error = "Image size must be less than 5MB!";
            } else {
                // Upload directory (Relative to admin folder)
                $upload_dir = '../assets/images/menu/';
                if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);
                
                $file_ext = pathinfo($file['name'], PATHINFO_EXTENSION);
                $filename = time() . '_' . bin2hex(random_bytes(5)) . '.' . $file_ext;
                $target_path = $upload_dir . $filename;
                
                // Path to store in Database (Relative to root folder)
                $image_db_path = 'assets/images/menu/' . $filename;
                
                if (!move_uploaded_file($file['tmp_name'], $target_path)) {
                    $error = "Failed to upload image.";
                }
            }
        }
        
        if (empty($error)) {
            if ($_POST['action'] == 'add') {
                if ($image_db_path) {
                    $stmt = $conn->prepare("INSERT INTO menu_items (name, main_category, sub_category, description, price, image) VALUES (?, ?, ?, ?, ?, ?)");
                    // FIXED: Changed "sssds" to "ssssds" (6 variables)
                    $stmt->bind_param("ssssds", $name, $main_category, $sub_category, $description, $price, $image_db_path);
                } else {
                    $stmt = $conn->prepare("INSERT INTO menu_items (name, main_category, sub_category, description, price) VALUES (?, ?, ?, ?, ?)");
                    // FIXED: "ssssd" (5 variables) - This was already correct but verifying
                    $stmt->bind_param("ssssd", $name, $main_category, $sub_category, $description, $price);
                }
            } else {
                $menu_id = intval($_POST['menu_id']);
                if ($image_db_path) {
                    // Update with new image
                    $stmt = $conn->prepare("UPDATE menu_items SET name=?, main_category=?, sub_category=?, description=?, price=?, image=? WHERE id=?");
                    // FIXED: Changed "sssdsi" to "ssssdsi" (7 variables)
                    $stmt->bind_param("ssssdsi", $name, $main_category, $sub_category, $description, $price, $image_db_path, $menu_id);
                } else {
                    // Update without changing image
                    $stmt = $conn->prepare("UPDATE menu_items SET name=?, main_category=?, sub_category=?, description=?, price=? WHERE id=?");
                    // FIXED: Changed "ssssd" to "ssssdi" (6 variables)
                    $stmt->bind_param("ssssdi", $name, $main_category, $sub_category, $description, $price, $menu_id);
                }
            }

            if ($stmt->execute()) {
                $message = $_POST['action'] == 'add' ? "Menu item added successfully!" : "Menu item updated successfully!";
            } else {
                $error = "Failed to save menu item: " . $stmt->error;
            }
        }
    }
}

// --- FETCH DATA ---
$result = $conn->query("SELECT * FROM menu_items ORDER BY main_category, sub_category, name");
$menu_items = [];
while ($row = $result->fetch_assoc()) {
    $menu_items[] = $row;
}

// --- STATISTICS ---
$total_items = count($menu_items);
$category_counts = [];
foreach ($menu_items as $item) {
    $cat = $item['main_category'];
    if (!isset($category_counts[$cat])) $category_counts[$cat] = 0;
    $category_counts[$cat]++;
}
$total_categories = count($category_counts);

// Categories Data
$main_categories = ['Veg', 'Non-Veg', 'Drinks'];
$sub_categories = [
    'Veg' => ['Breakfast', 'Lunch', 'Dinner'],
    'Non-Veg' => ['Breakfast', 'Lunch', 'Dinner'],
    'Drinks' => ['Hot Drinks', 'Cool Drinks', 'Fresh Juices', 'Ice Cream']
];
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Menu Management - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="assets/css/admin-style.css">
    <style>
        .menu-thumb { width: 60px; height: 60px; object-fit: cover; border-radius: 8px; border: 1px solid #ddd; }
        .description-text { font-size: 0.85em; color: #777; max-width: 200px; white-space: nowrap; overflow: hidden; text-overflow: ellipsis; }
        
        .file-upload-wrapper {
            border: 2px dashed #ccc; padding: 20px; text-align: center;
            background: #f9f9f9; border-radius: 8px; cursor: pointer; transition: 0.3s;
        }
        .file-upload-wrapper:hover { border-color: #931A2A; background: #fff5f5; }
        .file-upload-wrapper input { display: none; }
    </style>
</head>
<body>

<div class="wrapper">
    <?php include 'sidebar.php'; ?>

    <div class="main-content">
        <nav class="top-nav">
            <div class="nav-left"><h2><i class="fas fa-utensils"></i> Menu Management</h2></div>
            <div class="nav-right"><span class="admin-name">Welcome, Admin</span></div>
        </nav>

        <div class="content">
            <?php if ($message): ?>
                <div class="alert alert-success alert-dismissible fade show">
                    <?php echo htmlspecialchars($message); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
            <?php if ($error): ?>
                <div class="alert alert-danger alert-dismissible fade show">
                    <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>

            <div class="row mb-4">
                <div class="col-md-4">
                    <div class="stat-card" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;">
                        <div class="stat-icon mb-2"><i class="fas fa-list fa-2x opacity-50"></i></div>
                        <h3><?php echo $total_items; ?></h3>
                        <p class="mb-0">Total Items</p>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="stat-card" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); color: white;">
                        <div class="stat-icon mb-2"><i class="fas fa-tags fa-2x opacity-50"></i></div>
                        <h3><?php echo $total_categories; ?></h3>
                        <p class="mb-0">Categories</p>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="stat-card" style="background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%); color: white;">
                        <div class="stat-icon mb-2"><i class="fas fa-check-circle fa-2x opacity-50"></i></div>
                        <h3><?php echo count($menu_items); ?></h3>
                        <p class="mb-0">Active Items</p>
                    </div>
                </div>
            </div>

            <div class="admin-form-section d-flex justify-content-between align-items-center mb-3 p-3">
                <h5 class="mb-0 text-muted"><i class="fas fa-list"></i> Full Menu List</h5>
                <button class="btn btn-primary-custom" onclick="openAddModal()">
                    <i class="fas fa-plus"></i> Add New Item
                </button>
            </div>

            <div class="table-responsive">
                <table class="table table-modern align-middle table-hover">
                    <thead>
                        <tr>
                            <th>Image</th>
                            <th>Name</th>
                            <th>Main Category</th>
                            <th>Sub Category</th>
                            <th>Description</th>
                            <th>Price</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if(empty($menu_items)): ?>
                            <tr><td colspan="7" class="text-center text-muted py-4">No items found.</td></tr>
                        <?php else: ?>
                            <?php foreach ($menu_items as $item): ?>
                            <tr>
                                <td>
                                    <?php if (!empty($item['image'])): ?>
                                        <img src="../<?php echo htmlspecialchars($item['image']); ?>" class="menu-thumb" alt="Item">
                                    <?php else: ?>
                                        <div class="menu-thumb d-flex align-items-center justify-content-center bg-light text-muted">
                                            <i class="fas fa-image"></i>
                                        </div>
                                    <?php endif; ?>
                                </td>
                                <td class="fw-bold"><?php echo htmlspecialchars($item['name']); ?></td>
                                <td><span class="badge bg-primary"><?php echo htmlspecialchars($item['main_category']); ?></span></td>
                                <td><span class="badge bg-info"><?php echo htmlspecialchars($item['sub_category']); ?></span></td>
                                <td><div class="description-text" title="<?php echo htmlspecialchars($item['description']); ?>"><?php echo htmlspecialchars($item['description']); ?></div></td>
                                <td class="fw-bold text-success">₹<?php echo number_format($item['price'], 2); ?></td>
                                <td>
                                    <div class="action-buttons">
                                        <button class="btn btn-info btn-sm text-white" 
                                                onclick='openEditModal(<?php echo json_encode($item, JSON_HEX_APOS | JSON_HEX_QUOT); ?>)'>
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <form method="POST" style="display:inline;" onsubmit="return confirm('Delete this item?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="menu_id" value="<?php echo $item['id']; ?>">
                                            <button class="btn btn-danger btn-sm"><i class="fas fa-trash"></i></button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="menuModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header" style="background: #931A2A; color: white;">
                <h5 class="modal-title" id="modalTitle">Add New Item</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" enctype="multipart/form-data">
                <div class="modal-body">
                    <input type="hidden" name="action" id="formAction" value="add">
                    <input type="hidden" name="menu_id" id="menuId" value="">

                    <div class="mb-3">
                        <label class="form-label">Item Name *</label>
                        <input type="text" class="form-control" id="itemName" name="name" required>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Main Category *</label>
                            <select class="form-select" id="itemMainCategory" name="main_category" onchange="updateSubCategories()" required>
                                <option value="">Select</option>
                                <?php foreach ($main_categories as $cat): ?>
                                <option value="<?php echo htmlspecialchars($cat); ?>"><?php echo htmlspecialchars($cat); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Sub Category *</label>
                            <select class="form-select" id="itemSubCategory" name="sub_category" required>
                                <option value="">Select Main First</option>
                            </select>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Price (₹) *</label>
                        <input type="number" class="form-control" id="itemPrice" name="price" step="0.01" required>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Description</label>
                        <textarea class="form-control" id="itemDescription" name="description" rows="2"></textarea>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Image</label>
                        <div class="file-upload-wrapper" onclick="document.getElementById('itemImage').click()">
                            <input type="file" id="itemImage" name="image" accept="image/*">
                            <i class="fas fa-cloud-upload-alt fa-2x text-muted"></i>
                            <p class="mb-0 text-muted" id="fileName">Click to upload image</p>
                        </div>
                        <div id="imagePreview" class="mt-2 text-center"></div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary-custom" id="saveButton">Save</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
<script>
    const menuModal = new bootstrap.Modal(document.getElementById('menuModal'));
    const subCategoriesData = <?php echo json_encode($sub_categories); ?>;

    // Image Preview Logic
    document.getElementById('itemImage').addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            document.getElementById('fileName').innerText = file.name;
            const reader = new FileReader();
            reader.onload = function(ev) {
                document.getElementById('imagePreview').innerHTML = `<img src="${ev.target.result}" style="max-height: 100px; border-radius: 5px;">`;
            }
            reader.readAsDataURL(file);
        }
    });

    function updateSubCategories(selectedSub = null) {
        const mainCat = document.getElementById('itemMainCategory').value;
        const subCatSelect = document.getElementById('itemSubCategory');
        subCatSelect.innerHTML = '<option value="">Select Sub Category</option>';
        
        if (mainCat && subCategoriesData[mainCat]) {
            subCategoriesData[mainCat].forEach(sub => {
                const option = document.createElement('option');
                option.value = sub;
                option.textContent = sub;
                if(selectedSub === sub) option.selected = true;
                subCatSelect.appendChild(option);
            });
        }
    }

    function openAddModal() {
        document.getElementById('modalTitle').innerText = "Add New Item";
        document.getElementById('formAction').value = "add";
        document.getElementById('menuId').value = "";
        document.getElementById('itemName').value = "";
        document.getElementById('itemMainCategory').value = "";
        document.getElementById('itemSubCategory').innerHTML = '<option value="">Select Main First</option>';
        document.getElementById('itemDescription').value = "";
        document.getElementById('itemPrice').value = "";
        document.getElementById('itemImage').value = "";
        document.getElementById('imagePreview').innerHTML = "";
        document.getElementById('fileName').innerText = "Click to upload image";
        document.getElementById('saveButton').innerText = "Add Item";
        menuModal.show();
    }

    function openEditModal(data) {
        document.getElementById('modalTitle').innerText = "Edit Item";
        document.getElementById('formAction').value = "edit";
        document.getElementById('menuId').value = data.id;
        document.getElementById('itemName').value = data.name;
        document.getElementById('itemMainCategory').value = data.main_category;
        document.getElementById('itemDescription').value = data.description;
        document.getElementById('itemPrice').value = data.price;
        
        updateSubCategories(data.sub_category);
        
        if(data.image) {
            document.getElementById('imagePreview').innerHTML = `<img src="../${data.image}" style="max-height: 100px; border-radius: 5px;">`;
            document.getElementById('fileName').innerText = "Change Image (Optional)";
        } else {
            document.getElementById('imagePreview').innerHTML = "";
            document.getElementById('fileName').innerText = "Upload Image";
        }
        
        document.getElementById('saveButton').innerText = "Update Item";
        menuModal.show();
    }
</script>
</body>
</html>