# Weekly Specials Management System

## Overview
A complete admin management system for weekly restaurant specials with full CRUD operations (Create, Read, Update, Delete) while keeping the user-facing website read-only.

## Features

### Admin Panel (`admin/weekly.php`)
- ✅ **Add Weekly Special**: Upload image, set title, description, price, category, and badge type
- ✅ **Edit Weekly Special**: Modify all fields including replacing the image
- ✅ **Delete Weekly Special**: Remove specials with automatic image deletion
- ✅ **View All Specials**: Table view with all details and quick action buttons
- ✅ **Image Management**: Automatic file upload with validation (JPG, PNG, GIF, WebP, max 5MB)
- ✅ **Badge Types**: Hot 🔥, New ⭐, Offer 💰, or None

### Website Display (`weekly.php`)
- ✅ **Read-Only View**: Users can only view, not modify
- ✅ **Database-Driven**: Pulls content from database instead of hardcoded HTML
- ✅ **Category Display**: Shows different categories (Breakfast, Lunch, Dinner, Dessert, Drinks, Special)
- ✅ **Responsive Grid**: 2 columns on desktop, 1 on mobile
- ✅ **Badge Display**: Shows HOT, NEW, OFFER badges automatically

## Database Schema

```sql
CREATE TABLE `weekly_specials` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `title` VARCHAR(255) NOT NULL,
  `description` TEXT NOT NULL,
  `price` VARCHAR(100) NOT NULL,
  `category` VARCHAR(50) NOT NULL,
  `image_path` VARCHAR(255) NOT NULL,
  `badge_type` ENUM('hot', 'new', 'offer', 'none') DEFAULT 'none',
  `created_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
```

## Fields Explanation

| Field | Type | Description | Example |
|-------|------|-------------|---------|
| `id` | INT | Auto-increment primary key | 1 |
| `title` | VARCHAR(255) | Name of the special | "Sunday Feast" |
| `description` | TEXT | Full description | "Royal Biryani + Chicken 65 + Sweet + Drink" |
| `price` | VARCHAR(100) | Price or offer text | "₹399 Family Offer" |
| `category` | VARCHAR(50) | Category type | breakfast, lunch, dinner, dessert, drinks, special |
| `image_path` | VARCHAR(255) | Image file path | "assets/images/weekly/filename.jpg" |
| `badge_type` | ENUM | Badge display type | hot, new, offer, none |
| `created_date` | TIMESTAMP | Creation timestamp | Auto-set |
| `updated_date` | TIMESTAMP | Last update timestamp | Auto-updated |

## Setup Instructions

### 1. Create Database Table
Run the SQL from `admin/weekly_setup.sql`:
```bash
mysql -u root -p bermiz_db < admin/weekly_setup.sql
```

Or manually run in phpMyAdmin:
```sql
-- Copy the entire SQL from admin/weekly_setup.sql
```

### 2. Create Images Directory
```bash
mkdir -p assets/images/weekly/
chmod 755 assets/images/weekly/
```

### 3. Access Admin Panel
- URL: `http://localhost/hotel/admin/weekly.php`
- Requires admin login
- Login with your admin credentials from `admins` table

### 4. View Website Display
- URL: `http://localhost/hotel/weekly.php`
- Read-only for regular users
- Automatically pulls from database

## How to Use

### Adding a Weekly Special (Admin)
1. Go to `/admin/weekly.php`
2. Click "Add New Special" button
3. Fill in the form:
   - **Title**: Name of the special (e.g., "Sunday Feast")
   - **Description**: Details (e.g., "Royal Biryani + Chicken 65 + Sweet + Drink")
   - **Price**: Price or offer text (e.g., "₹399 Family Offer")
   - **Category**: Select from dropdown
   - **Badge Type**: Choose hot 🔥, new ⭐, offer 💰, or none
   - **Image**: Upload JPG/PNG/GIF/WebP (max 5MB)
4. Click "Add Special"

### Editing a Weekly Special (Admin)
1. Go to `/admin/weekly.php`
2. Click the ✏️ (Edit) button on any special
3. Modify the fields
4. Optionally upload a new image
5. Click "Update Special"

### Deleting a Weekly Special (Admin)
1. Go to `/admin/weekly.php`
2. Click the 🗑️ (Delete) button
3. Confirm deletion
4. Image automatically deleted from server

### Viewing Weekly Specials (Users)
1. Navigate to `/weekly.php` on the website
2. View all specials in a responsive grid
3. See badges (HOT 🔥, NEW ⭐, OFFER 💰) automatically
4. No editing possible - read-only interface

## Key Files Modified

### New Files Created:
- ✅ `admin/weekly.php` - Admin management panel
- ✅ `admin/weekly_setup.sql` - Database setup script

### Files Modified:
- ✅ `weekly.php` - Updated to fetch from database instead of hardcoded content

### Image Upload Directory:
- ✅ `assets/images/weekly/` - Storage for weekly special images

## Security Features

- ✅ Admin authentication required (`check_admin.php`)
- ✅ File type validation (only image files)
- ✅ File size limit (5MB maximum)
- ✅ SQL injection prevention (prepared statements)
- ✅ XSS prevention (htmlspecialchars encoding)
- ✅ Automatic old image deletion on updates
- ✅ Session-based access control

## Technical Details

### Image Handling
- Images uploaded to: `assets/images/weekly/`
- Filename format: `{timestamp}_{random}.{extension}`
- Prevents filename collisions and overwriting
- Automatic cleanup when deleting or updating

### Database Operations
- Uses prepared statements for security
- Automatic timestamps on create/update
- Cascading deletes with file cleanup
- Proper error handling and logging

### Frontend
- Bootstrap 5.3.0 for responsive design
- Font Awesome 6.0 for icons
- Drag-and-drop file upload
- Real-time form validation
- Mobile-friendly interface

## Categories Available
- **breakfast** - Morning specials
- **lunch** - Lunch time dishes
- **dinner** - Dinner specials
- **dessert** - Sweet items
- **drinks** - Beverages
- **special** - General specials

## Badge Types
- **hot** 🔥 - Hot/Popular items
- **new** ⭐ - New menu items
- **offer** 💰 - Special offers/discounts
- **none** - No badge

## Responsive Design

### Desktop (>768px)
- 2 columns grid
- 280px × 280px images
- Full details displayed

### Tablet (480px - 768px)
- 1 column grid
- 200px × 200px images
- Optimized for touch

### Mobile (<480px)
- 1 column grid
- 150px × 150px images
- Compact layout

## Troubleshooting

### Issue: "Table already exists"
**Solution**: Table auto-creates. If error persists, check database connection.

### Issue: Images not uploading
**Solution**: 
- Ensure `assets/images/weekly/` directory exists
- Check folder permissions (755)
- Verify PHP max upload size

### Issue: Admin can't access page
**Solution**: 
- Check admin login status
- Verify session in `check_admin.php`
- Clear browser cookies

### Issue: Weekly specials not showing on website
**Solution**:
- Verify database table created
- Check data exists in `weekly_specials` table
- Verify image paths are correct

## Future Enhancements

Possible additions:
- Category-based filtering on website
- Featured special with larger display
- Scheduling (auto-show/hide by date)
- Stock management
- Nutrition information
- Customer reviews

## Support

For issues or questions:
1. Check error logs in admin panel
2. Verify database connection in `db_connect.php`
3. Ensure all image files are present
4. Check folder permissions

---

**Version**: 1.0  
**Last Updated**: December 23, 2025  
**Status**: Production Ready
