# Gallery Management System Documentation

## Overview
The Gallery Management System allows administrators to dynamically manage restaurant gallery images with support for multiple categories. Images added through the admin panel automatically appear on the public gallery page.

## Features

### Admin Features (admin/gallery.php)
1. **Add Images**
   - Upload image files (JPG, PNG, GIF, WebP)
   - Set image title
   - Select category (Food, Ambience & Events, Staff)
   - Maximum file size: 5MB
   - Drag-and-drop file upload support

2. **View Images**
   - Table view of all gallery images
   - Thumbnail preview with click-to-enlarge functionality
   - Display upload date
   - Show category badge with color coding

3. **Edit Images**
   - Update image title
   - Change image category
   - Modal interface for editing

4. **Delete Images**
   - Remove images from database and server
   - Confirmation dialog to prevent accidental deletion
   - Physical file deletion from assets/images/gallery/

### User Features (gallery.php)
- **Category Filtering**: View all images or filter by category
- **Lightbox Preview**: Enlarged image viewing with PhotoSwipe
- **Responsive Design**: Works on all device sizes
- **Dynamic Content**: Displays images from database instead of hardcoded HTML

## Database Schema

### Gallery Table
```sql
CREATE TABLE `gallery` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `title` VARCHAR(255) NOT NULL,
  `category` VARCHAR(50) NOT NULL,
  `image_path` VARCHAR(255) NOT NULL,
  `created_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_date` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)
```

### Valid Categories
- `food` - Food items and dishes
- `events` - Ambience, events, and special occasions
- `staff` - Staff photos, chefs, team members

## File Structure

```
hotel/
├── admin/
│   ├── gallery.php (NEW - Admin gallery management)
│   ├── gallery_setup.sql (NEW - Database table creation)
│   ├── sidebar.php (Updated - Added gallery link)
│   └── index.php (Updated - Added gallery quick action)
├── gallery.php (Updated - Now uses database)
└── assets/
    └── images/
        └── gallery/ (NEW - Auto-created for uploaded images)
```

## Setup Instructions

### Step 1: Create Gallery Table
Run one of these options:

**Option A: Using Admin Panel**
- Gallery table is automatically created when admin/gallery.php is first accessed
- No manual setup required

**Option B: Manual SQL**
- Import `admin/gallery_setup.sql` in phpMyAdmin
- Or run the CREATE TABLE statement directly

### Step 2: Access Admin Gallery
1. Log in to admin panel
2. Click "Gallery" in sidebar OR "Manage Gallery" in Dashboard quick actions
3. Start adding images!

## Usage Guide

### Adding an Image
1. Fill in the "Image Title" field
2. Select a "Category" from dropdown
3. Click the upload area or drag an image
4. Click "Add Image" button
5. Image appears in the gallery list and public website

### Editing an Image
1. Click "Edit" button on the image row
2. Update Title and/or Category
3. Click "Save Changes"
4. Changes appear immediately on website

### Deleting an Image
1. Click "Delete" button on the image row
2. Confirm deletion in dialog
3. Image is removed from database and server

### Viewing Images
1. Click "View" button on image row
2. Or click the thumbnail to open lightbox
3. Click "×" or outside image to close

## Image Upload Details

### Supported Formats
- JPG/JPEG
- PNG
- GIF
- WebP

### File Size Limits
- Maximum: 5MB
- Recommended: 2MB or less for web performance

### Image Naming
- Uploaded files are renamed to: `timestamp_randomstring.extension`
- Example: `1704067200_a3f4b2c1.jpg`
- This prevents filename conflicts and security issues

### Upload Directory
- Images stored in: `assets/images/gallery/`
- Directory is automatically created if not exists
- Files are publicly accessible for website display

## Integration with Website

### How It Works
The user-facing `gallery.php` queries the database and displays all images:

```php
// Fetches all images from database
$result = $conn->query("SELECT * FROM gallery ORDER BY created_date DESC");

// Displays with category class for filtering
echo '<div class="gallery-item ' . $category . '">
      <img src="' . $image_path . '" alt="' . $title . '">
      </div>';
```

### Fallback Display
If no database images exist, website shows hardcoded legacy images from:
- `assets/images/menupage-6/`
- `assets/images/chefspage2/`

### Category Filtering
Filter buttons on website automatically filter displayed images by category:
- All (displays all images)
- Food (category = 'food')
- Ambience & Events (category = 'events')
- Staff (category = 'staff')

## Styling & Design

### Color Scheme
- Primary Red: #931A2A
- Gold Accent: #ffecb0
- Dark Background: #111A1E
- Light Text: #F5E6D3

### Category Badges
- Food: Red (#ff6b6b)
- Events: Blue (#4a90e2)
- Staff: Green (#84cc16)

### Responsive Breakpoints
- Desktop: Full layout
- Tablet (768px): Adjusted padding and font sizes
- Mobile (480px): Single column layout, compact buttons

## Troubleshooting

### Images Not Uploading
1. Check file format (must be JPG, PNG, GIF, or WebP)
2. Verify file size is under 5MB
3. Ensure directory permissions are correct
4. Check browser console for error messages

### Images Not Showing on Website
1. Verify image path is correct in database
2. Check if `assets/images/gallery/` directory exists
3. Ensure uploaded files have correct permissions (644)
4. Try clearing browser cache

### Database Errors
1. Run `admin/gallery_setup.sql` to ensure table exists
2. Verify database connection in `db_connect.php`
3. Check error logs for SQL errors

### File Upload Permissions
If you get "Failed to upload image" error:
1. Ensure `assets/images/gallery/` directory exists and is writable
2. Set folder permissions to 755
3. Set file permissions to 644 after upload
4. On Windows, check NTFS permissions

## Security Features

1. **File Type Validation**: Only image files are accepted
2. **File Size Limits**: Prevents large file uploads
3. **SQL Injection Prevention**: Uses prepared statements
4. **File Naming**: Random strings prevent directory traversal
5. **Admin Authentication**: Gallery page requires admin login
6. **Confirmation Dialogs**: Prevents accidental data loss

## Performance Optimization

1. **Image Format**: Use WebP for better compression
2. **File Size**: Recommended max 2MB for web
3. **Database Indexing**: ID and created_date indexed automatically
4. **Lazy Loading**: Website implements image lazy loading
5. **Caching**: Browser caching for image files

## Sample Data

To add sample images, uncomment and run in gallery_setup.sql:
```sql
INSERT INTO `gallery` (`title`, `category`, `image_path`) VALUES
('Delicious Biryani', 'food', 'assets/images/menupage-6/menu-1.png'),
('Fresh Salad', 'food', 'assets/images/menupage-6/menu-2.png'),
('Restaurant Ambience', 'events', 'assets/images/menupage-6/1.jpg'),
('Our Chef', 'staff', 'assets/images/chefspage2/chefs-1.png');
```

## Support

For issues or questions:
1. Check this documentation
2. Review error messages in browser console
3. Check PHP error logs
4. Verify database connection
5. Test with a simple image first

## Future Enhancements

Possible improvements:
- Image compression on upload
- Reordering images via drag-drop
- Batch upload multiple images
- Image SEO optimization
- Social media integration
- Image statistics/views tracking
- Approval workflow for images
- Image cropping tool
- Watermark support
