# 📋 OTP Registration - Complete Verification Checklist

## Overview
This is your step-by-step checklist to verify that the complete OTP registration system works correctly before going live.

---

## 🎯 What You'll Verify
✅ OTP is generated correctly when you send it  
✅ OTP is stored in the database  
✅ OTP expires after 10 minutes  
✅ OTP can be verified  
✅ User account is created after verification  
✅ User can login with new credentials  

---

## 📊 Database Tables to Check

### 1. email_otp Table (Temporary - stores OTP codes)
```sql
SELECT * FROM email_otp;
```
**Fields:**
- `email`: User's email address
- `otp`: 6-digit code
- `created_at`: When OTP was generated
- `expires_at`: When OTP expires (10 minutes after creation)
- `verified`: 0 = not verified, 1 = verified

### 2. users Table (Permanent - stores registered users)
```sql
SELECT * FROM users;
```
**Fields:**
- `id`: User ID
- `name`: User's full name
- `email`: User's email
- `password`: Encrypted password
- `created_at`: Registration date

---

## 🚀 STEP-BY-STEP REGISTRATION TEST

### PHASE 1: SEND OTP

**Action 1.1: Go to Login/Sign Up Page**
```
URL: http://localhost/hotel/login.php?show=signup
```

**Action 1.2: Click "Sign Up" Button**
- The page shows three forms, all hidden except "Dine In" form
- Click the "Sign Up" link/button at bottom

**Action 1.3: Enter Test Email**
- Email field appears: `🔐 Step 1: Verify Your Email`
- Enter any test email: `john.doe@example.com`
- Click "Send OTP" button

**Expected Results:**
```
✅ You see message: "OTP sent successfully! Check your email."
✅ Form transitions to Step 2: OTP verification
```

**Verify in Database (Open another tab or use terminal):**
```sql
SELECT * FROM email_otp WHERE email = 'john.doe@example.com';
```

**Expected Output:**
```
id  | email                | otp    | created_at          | expires_at          | verified
1   | john.doe@example.com | 456789 | 2025-12-24 10:30:00 | 2025-12-24 10:40:00 | 0
```

**What This Means:**
- ✅ OTP `456789` was generated
- ✅ Created at exactly this moment
- ✅ Will expire in 10 minutes
- ✅ Not verified yet (0)

---

### PHASE 2: VERIFY OTP

**Action 2.1: Copy OTP from Email**
- Check your email inbox/spam folder
- Look for email with subject: "Your OTP Code"
- Copy the 6-digit code (e.g., `456789`)

**OR if you don't receive email:**
- Open your test page: `http://localhost/hotel/otp-test.php`
- Look at "All OTP Records" table
- Find your email and copy the OTP code
- The code is shown in large text

**Action 2.2: Enter OTP in Form**
- Go back to `http://localhost/hotel/login.php`
- Should see Step 2: "Enter the OTP sent to your email"
- Paste the OTP code (6 digits)
- Click "Verify OTP" button

**Expected Results:**
```
✅ You see message: "OTP verified successfully!"
✅ Form transitions to Step 3: Registration form
```

**Verify in Database:**
```sql
SELECT * FROM email_otp WHERE email = 'john.doe@example.com';
```

**Expected Output:**
```
id  | email                | otp    | created_at          | expires_at          | verified
1   | john.doe@example.com | 456789 | 2025-12-24 10:30:00 | 2025-12-24 10:40:00 | 1
                                                                                        ↑
                                                          Changed from 0 to 1 (verified!)
```

**What This Means:**
- ✅ OTP was verified successfully
- ✅ System confirmed OTP is correct
- ✅ `verified` changed from 0 to 1
- ✅ User can now proceed to registration

---

### PHASE 3: COMPLETE REGISTRATION

**Action 3.1: Fill Registration Form**
Step 3 form should show:
```
Name: _________________ (required)
Email: john.doe@example.com (read-only, auto-filled)
Password: _________________ (min 6 chars)
Confirm Password: _________ (must match)
```

**Fill the form:**
- Name: `John Doe`
- Password: `TestPassword123`
- Confirm: `TestPassword123`
- Click "Join Bermiz" button

**Expected Results:**
```
✅ You see message: "Registration successful! You can now login."
✅ Form resets or redirects to login
```

**Verify in Database - Check users table:**
```sql
SELECT * FROM users WHERE email = 'john.doe@example.com';
```

**Expected Output:**
```
id | name     | email                | password                                            | created_at
1  | John Doe | john.doe@example.com | $2y$10$kG8y7QxP2L4m9Vc5sN8xUe9D6kL2Vc3Nm1Qx5Zw... | 2025-12-24 10:32:00
                                       ↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑↑
                          Encrypted password (bcrypt hash)
```

**What This Means:**
- ✅ User account created successfully
- ✅ Password is encrypted (starts with `$2y$10$`)
- ✅ Email is unique (no duplicates)
- ✅ User can now login

**Verify OTP Cleanup:**
```sql
SELECT * FROM email_otp WHERE email = 'john.doe@example.com';
```

**Expected Output:**
```
(Empty - no results)
```

**What This Means:**
- ✅ OTP record was automatically deleted after successful registration
- ✅ System cleaned up temporary data

---

### PHASE 4: LOGIN TEST

**Action 4.1: Go to Login Page**
```
URL: http://localhost/hotel/login.php
```

**Action 4.2: Enter Credentials**
```
Email: john.doe@example.com
Password: TestPassword123
```
- Click "Dine In" button

**Expected Results:**
```
✅ Login successful
✅ Redirected to homepage (index.php)
✅ Session variables set (name, email, user_id visible in navigation)
```

**Verify Session:**
- Look at top right of homepage
- Should see username or email displayed
- Should see "Logout" button instead of Login button

---

## 🐛 TROUBLESHOOTING

### Problem 1: OTP Not Appearing in Database
**Symptoms:** After clicking "Send OTP", no record in email_otp table

**Solutions:**
1. Check if table exists:
```sql
SHOW TABLES LIKE 'email_otp';
```
✅ Should show: `email_otp`

2. Verify email_otp table structure:
```sql
DESCRIBE email_otp;
```
✅ Should have columns: id, email, otp, created_at, expires_at, verified

3. Check auth.php file:
- Make sure it has the `auto_create_table` code block
- It should create the table automatically if missing

4. Check for PHP errors:
- Open browser console (F12)
- Look for JavaScript errors
- Check XAMPP error log: `c:\xampp\apache\logs\error.log`

---

### Problem 2: Email Not Received
**Symptoms:** Cannot find OTP in inbox or spam folder

**Solutions:**
1. Use Test Page to Get OTP:
```
URL: http://localhost/hotel/otp-test.php
```
- See all OTP records in table
- Copy the code from database directly

2. In Production:
- Configure SMTP in PHP
- Current setup uses mail() which may not send in development

3. For Testing:
- Use the otp-test.php page to verify OTP was stored
- Email sending is secondary (system works without it)

---

### Problem 3: OTP Verification Failed
**Symptoms:** "OTP is incorrect or expired" message

**Solutions:**
1. Make sure OTP hasn't expired (should be within 10 minutes):
```sql
SELECT * FROM email_otp WHERE email = 'your@email.com';
```
- Check if `expires_at` is greater than current time
- If expired, request new OTP

2. Check you're entering correct OTP:
- Must be exactly 6 digits
- Copy from database or email - don't manually type

3. Check 3-attempt limit:
- After 3 wrong attempts, must resend OTP

---

### Problem 4: User Not Created After Verification
**Symptoms:** OTP verified but user not in users table

**Solutions:**
1. Make sure users table exists:
```sql
SHOW TABLES LIKE 'users';
```

2. Check user registration form:
- All fields must be filled
- Password must be minimum 6 characters
- Passwords must match

3. Check for database errors:
- Open browser console (F12)
- Look for error messages from fetch call

---

### Problem 5: Cannot Login After Registration
**Symptoms:** User created but login fails

**Solutions:**
1. Verify user exists:
```sql
SELECT * FROM users WHERE email = 'your@email.com';
```

2. Check password is correct:
- Must match exactly (case-sensitive)

3. Check session is created:
- After login, check top right for username display

---

## ✅ FINAL CHECKLIST

Before declaring the system ready, verify ALL of these:

```
[ ] 1. OTP Page Accessible
    URL: http://localhost/hotel/login.php?show=signup
    Status: Page loads without errors

[ ] 2. OTP Sent to Database
    After clicking "Send OTP":
    [ ] Message: "OTP sent successfully"
    [ ] Database shows record in email_otp table
    [ ] OTP is 6 digits
    [ ] verified = 0
    [ ] expires_at is 10 minutes in future

[ ] 3. OTP Verified
    After entering OTP and clicking "Verify":
    [ ] Message: "OTP verified successfully"
    [ ] Database verified field changed to 1
    [ ] Registration form appears
    [ ] Email field is read-only (pre-filled)

[ ] 4. User Created
    After completing registration:
    [ ] Message: "Registration successful"
    [ ] Database shows new user in users table
    [ ] Password is encrypted (starts with $2y$10$)
    [ ] OTP record deleted from email_otp
    [ ] Can check with: SELECT * FROM users WHERE email = 'your@email.com';

[ ] 5. Login Works
    After logging in with new credentials:
    [ ] Redirected to homepage
    [ ] Username/email visible in navigation
    [ ] "Logout" button appears
    [ ] Can navigate to profile page

[ ] 6. Database Cleanup
    After complete registration:
    [ ] OTP record automatically deleted
    [ ] User record persists in users table
    [ ] No duplicate emails in users table

[ ] 7. Resend OTP Works
    If OTP expired or lost:
    [ ] Click "Resend OTP"
    [ ] New OTP generated with new timestamp
    [ ] Attempt counter reset
    [ ] Can verify with new OTP

[ ] 8. Edge Cases
    [ ] Cannot register without email verification
    [ ] Cannot register with invalid email format
    [ ] Cannot register with mismatched passwords
    [ ] Cannot register with password < 6 chars
    [ ] Cannot login with wrong password
    [ ] Cannot use same email twice
```

---

## 📞 Quick Reference Commands

### View OTP Records
```sql
SELECT email, otp, verified, expires_at FROM email_otp;
```

### View All Users
```sql
SELECT name, email, created_at FROM users;
```

### Check Specific User
```sql
SELECT * FROM users WHERE email = 'your@email.com';
```

### Delete All Test OTP Records
```sql
DELETE FROM email_otp;
```

### Delete All Test Users
```sql
DELETE FROM users;
```

### Check Table Structure
```sql
DESCRIBE email_otp;
DESCRIBE users;
```

---

## 🎉 SUCCESS!

If you checked all boxes above, your OTP registration system is working perfectly!

### Next Steps:
1. ✅ Test with real users
2. ✅ Configure email sending for production
3. ✅ Remove otp-test.php file from production
4. ✅ Keep backup of OTP_SYSTEM_COMPLETE_DETAILS.md for reference
5. ✅ Monitor registration flow in admin panel

---

**Last Updated:** December 24, 2025  
**System Status:** Complete & Ready for Testing
