# OTP Email Verification System - Complete Details & Setup Guide

## 📋 System Overview

### What the System Does
1. **Email Verification**: User enters email → System generates OTP → Stores in database
2. **OTP Storage**: 6-digit code saved in `email_otp` table with 10-minute expiration
3. **OTP Verification**: User enters OTP → System validates against database
4. **Registration**: After OTP verified → User completes registration
5. **Login**: User can login with new account

---

## 🗄️ Database Structure

### Table 1: `email_otp` (NEW - Auto-created)
```sql
CREATE TABLE email_otp (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL UNIQUE,
    otp VARCHAR(6) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    verified TINYINT(1) DEFAULT 0
);
```

**Fields Explanation:**
- `id`: Unique identifier for each OTP record
- `email`: User's email address (must be unique - prevents duplicate requests)
- `otp`: 6-digit numeric code (e.g., "123456")
- `created_at`: When OTP was generated (auto-set to NOW())
- `expires_at`: When OTP expires (auto-set to NOW() + 10 minutes)
- `verified`: Flag (0 = not verified, 1 = verified)

**Example Record:**
```
id: 1
email: john@example.com
otp: 456789
created_at: 2025-12-24 10:30:00
expires_at: 2025-12-24 10:40:00 (10 minutes later)
verified: 0 (not yet verified)
```

### Table 2: `users` (EXISTING)
```sql
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    email VARCHAR(255) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

**User is created ONLY after OTP verification**

---

## 📊 Complete Registration Flow

### STEP 1: User Enters Email
```
Frontend (login.php)
    ↓
User clicks "Sign Up"
    ↓
"Verify Email" form displayed
    ↓
User enters: john@example.com
    ↓
User clicks "Send OTP"
```

### STEP 2: System Generates OTP
```
Backend (auth.php - send_otp action)
    ↓
Validate email format
    ✓ Check: Is email a valid format?
    ✗ Error if: Invalid email
    
Check if email already registered
    ✓ Check: SELECT FROM users WHERE email = ?
    ✗ Error if: Email already exists
    
Generate 6-digit OTP
    Code: str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT)
    Example: 456789
    
Delete old OTP for this email (if exists)
    Code: DELETE FROM email_otp WHERE email = ?
    
Store OTP in database
    INSERT INTO email_otp (email, otp, expires_at)
    VALUES ('john@example.com', '456789', NOW() + 10 minutes)
    
Send email to user
    ✓ Contains: OTP code (456789)
    ✓ Contains: Expiration time (10 minutes)
    ✓ Contains: Note about spam folder
    
Return success to frontend
    Response: "OTP sent successfully to your email"
```

### STEP 3: User Receives OTP
```
Email arrives in user's inbox:
┌─────────────────────────────────────┐
│ From: noreply@bermiz.local          │
│ Subject: Bermiz Registration OTP    │
│                                     │
│ Hello,                              │
│                                     │
│ Your OTP Code: 456789               │
│ Valid for: 10 minutes               │
│                                     │
│ Bermiz Restaurant Team              │
└─────────────────────────────────────┘
```

### STEP 4: User Enters OTP
```
Frontend (login.php)
    ↓
"OTP Verification" form displayed
    ↓
User sees: "OTP sent to: john@example.com"
    ↓
User enters: 456789
    ↓
User clicks "Verify OTP"
```

### STEP 5: System Validates OTP
```
Backend (auth.php - verify_otp action)
    ↓
Query database for OTP
    SELECT otp FROM email_otp 
    WHERE email = 'john@example.com'
    AND expires_at > NOW()
    AND verified = 0
    
Validation Checks:
    ✓ OTP exists?
    ✓ OTP not expired? (expires_at > NOW())
    ✓ OTP not already used? (verified = 0)
    ✓ OTP matches entered value? (456789 == 456789)
    
If all pass:
    UPDATE email_otp SET verified = 1
    Store in SESSION: $_SESSION['verified_email'] = 'john@example.com'
    Return success: "Email verified successfully"
    
If any fails:
    Return error with reason
```

### STEP 6: User Completes Registration
```
Frontend (login.php)
    ↓
"Create Account" form displayed
    ↓
Form auto-fills email: john@example.com (read-only)
    ↓
User enters:
    - Name: John Doe
    - Password: password123
    - Confirm: password123
    ↓
User clicks "Join Bermiz"
```

### STEP 7: System Creates User Account
```
Backend (auth.php - signup action)
    ↓
Check: Was email verified?
    SELECT FROM SESSION: $_SESSION['verified_email']
    ✗ Error if: Not verified or missing
    
Validate all inputs
    ✓ Name not empty?
    ✓ Password not empty?
    ✓ Password >= 6 characters?
    ✓ Passwords match?
    ✓ Email not already in users table?
    
Hash password
    Code: password_hash('password123', PASSWORD_DEFAULT)
    Result: $2y$10$xxxxxxxxxxxxx... (hashed)
    
Insert into users table
    INSERT INTO users (name, email, password)
    VALUES ('John Doe', 'john@example.com', 'hashed_password')
    
Delete OTP record
    DELETE FROM email_otp WHERE email = 'john@example.com'
    
Clear session
    unset($_SESSION['verified_email'])
    
Return success
    Response: "Registration successful! Please login now."
```

### STEP 8: User Logs In
```
Frontend (login.php)
    ↓
Switch to "Sign In" form
    ↓
User enters:
    - Email: john@example.com
    - Password: password123
    ↓
User clicks "Dine In"
```

### STEP 9: System Authenticates User
```
Backend (auth.php - login action)
    ↓
Query users table
    SELECT * FROM users 
    WHERE email = 'john@example.com'
    
Verify password
    password_verify('password123', hashed_password)
    ✓ If true: Passwords match
    ✗ If false: Invalid password
    
Create session
    $_SESSION['user_id'] = 1
    $_SESSION['user_name'] = 'John Doe'
    $_SESSION['user_email'] = 'john@example.com'
    
Redirect to homepage
    Response: Redirect to index.php
```

### STEP 10: User Logged In Successfully
```
Frontend (index.php)
    ↓
Session detected: $_SESSION['user_name'] = 'John Doe'
    ↓
Header shows: "Hello, John Doe"
    ↓
Login buttons replaced with profile dropdown
    ↓
✅ Registration and login complete!
```

---

## 📧 Email Sending Details

### Current Status in XAMPP
XAMPP comes with a basic mail setup. The `mail()` function works but:
- May go to spam folder in some cases
- May not work without proper server configuration
- Works fine for local development and testing

### How Email is Sent

**File**: `auth.php` (lines 61-75)

```php
function sendOtpEmail($email, $otp) {
    $to = $email;                    // Where to send
    $subject = "Bermiz Registration OTP: 456789";  // Email subject
    
    $message = "Hello,\n\n";         // Email body
    $message .= "Your OTP is: " . $otp . "\n";
    // ... more message content ...
    
    $headers = "From: noreply@bermiz.local\r\n";   // Email headers
    $headers .= "Reply-To: support@bermiz.local\r\n";
    
    return @mail($to, $subject, $message, $headers);  // Send email
}
```

### Email Content Sent
```
Subject: Bermiz Registration OTP: 456789
From: noreply@bermiz.local

---

Hello,

Thank you for registering with Bermiz Restaurant.
Your One-Time Password (OTP) is: 456789

This OTP is valid for 10 minutes only.
If you did not request this, please ignore this email.

Best regards,
Bermiz Restaurant Team
© 2025 Bermiz Restaurant. All rights reserved.
```

---

## 🗂️ Files Involved

### 1. login.php (Frontend)
**Location**: `c:\xampp\htdocs\hotel\login.php`

**What it does:**
- Shows 3-step registration form
- Step 1: Email verification form
- Step 2: OTP verification form
- Step 3: Registration form
- Handles user interactions
- Sends data to auth.php

**Key JavaScript Functions:**
```javascript
sendOtp()          // Sends email, calls auth.php send_otp
verifyOtp()        // Verifies OTP, calls auth.php verify_otp
signupForm submit  // Completes registration, calls auth.php signup
loginForm submit   // Login process, calls auth.php login
```

### 2. auth.php (Backend)
**Location**: `c:\xampp\htdocs\hotel\auth.php`

**What it does:**
- Handles all backend logic
- Creates email_otp table automatically
- Generates and stores OTP in database
- Validates OTP
- Creates user account
- Handles login

**Key Functions:**
```php
sendOtpEmail()     // Sends email with OTP
send_otp action    // Generates OTP and stores in DB
verify_otp action  // Validates OTP from user
signup action      // Creates user after OTP verification
login action       // Standard login
```

---

## 🔍 Database Verification Steps

### Check if email_otp table exists
```sql
SHOW TABLES LIKE 'email_otp';
```
Should return 1 row if table exists.

### View all OTP records
```sql
SELECT * FROM email_otp;
```

### View OTP for specific email
```sql
SELECT * FROM email_otp WHERE email = 'john@example.com';
```

### Check if OTP is expired
```sql
SELECT * FROM email_otp 
WHERE email = 'john@example.com' 
AND expires_at > NOW();
```
Returns OTP if still valid, empty if expired.

### Check if OTP is verified
```sql
SELECT * FROM email_otp 
WHERE email = 'john@example.com' 
AND verified = 1;
```

### Check created user
```sql
SELECT * FROM users WHERE email = 'john@example.com';
```

---

## 🧪 Testing the Complete Flow

### Prerequisite: Enable Debugging
Add this to top of `auth.php` for testing (remove after):
```php
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

### Test Scenario: John registers

**STEP 1: Send OTP**
- Go to: `http://localhost/hotel/login.php`
- Click "Sign Up" button
- Enter email: `john@example.com`
- Click "Send OTP"
- **Check Database:**
  ```sql
  SELECT * FROM email_otp WHERE email = 'john@example.com';
  ```
  You should see:
  ```
  id: 1
  email: john@example.com
  otp: 456789 (example, will be random)
  created_at: 2025-12-24 10:30:00
  expires_at: 2025-12-24 10:40:00
  verified: 0
  ```

**STEP 2: Verify OTP**
- Open above OTP record to see the generated OTP code
- Enter OTP in the form
- Click "Verify OTP"
- **Check Database:**
  ```sql
  SELECT * FROM email_otp WHERE email = 'john@example.com';
  ```
  Should now show:
  ```
  verified: 1 (changed from 0 to 1)
  ```

**STEP 3: Complete Registration**
- Fill in:
  - Name: John Doe
  - Email: john@example.com (auto-filled, read-only)
  - Password: password123
  - Confirm: password123
- Click "Join Bermiz"
- **Check Database:**
  ```sql
  SELECT * FROM users WHERE email = 'john@example.com';
  ```
  Should show:
  ```
  id: 1
  name: John Doe
  email: john@example.com
  password: $2y$10$xxxxxxxxxxxx... (hashed)
  created_at: 2025-12-24 10:35:00
  ```
  
  And OTP record should be deleted:
  ```sql
  SELECT * FROM email_otp WHERE email = 'john@example.com';
  ```
  Should return: (No rows)

**STEP 4: Login**
- Click "Sign In" button
- Enter:
  - Email: john@example.com
  - Password: password123
- Click "Dine In"
- Should redirect to: `index.php`
- Header should show: "Hello, John Doe"

---

## ⚠️ Common Issues & Solutions

### Issue 1: "Failed to send OTP. Check connection"
**Cause**: Email function failing (normal in XAMPP)
**Solution**: OTP is still stored in database - this message is misleading
**Action**: Check database directly using test page (see below)

### Issue 2: OTP not showing in database
**Cause**: Database table not created
**Solution**: Run the SQL setup manually or trigger send_otp once
**Query to run:**
```sql
CREATE TABLE IF NOT EXISTS email_otp (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL UNIQUE,
    otp VARCHAR(6) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    verified TINYINT(1) DEFAULT 0
);
```

### Issue 3: "OTP expired or not found"
**Cause**: OTP was more than 10 minutes old
**Solution**: Generate new OTP by clicking "Resend OTP"

### Issue 4: "Maximum attempts exceeded"
**Cause**: Entered wrong OTP 3 times
**Solution**: Start over with new email

### Issue 5: Email field still has old email after signup
**Cause**: Form not cleared properly
**Solution**: Refresh page or click "Sign Up" again

---

## 📋 Quick Reference

### OTP Rules
- **Length**: 6 digits (0-999999)
- **Expiration**: 10 minutes
- **Storage**: email_otp table
- **Max Attempts**: 3
- **Unique**: One OTP per email at a time (old ones deleted)

### Password Rules
- **Minimum Length**: 6 characters
- **Hashing**: bcrypt (PASSWORD_DEFAULT)
- **Must Match**: Confirm password required

### Email Validation
- **Format**: Must be valid email format
- **Unique**: Cannot be already registered
- **Storage**: users table after verification

### Session Data
After OTP verification:
- `$_SESSION['verified_email']` = user's email

After login:
- `$_SESSION['user_id']` = user ID
- `$_SESSION['user_name']` = user's name
- `$_SESSION['user_email']` = user's email

---

## ✅ Success Indicators

✅ **OTP Sent**: Message appears in frontend
✅ **OTP Stored**: Record visible in email_otp table
✅ **OTP Verified**: verified field changes from 0 to 1
✅ **User Created**: Record appears in users table
✅ **User Logged In**: Redirected to index.php with session data
✅ **Header Updated**: Shows "Hello, [User Name]"

---

This completes the entire OTP registration and login flow!
