# 🚀 Gallery Management System - Deployment Guide

## Pre-Deployment Checklist

### System Requirements
- [ ] PHP 7.4 or higher
- [ ] MySQL 5.7 or higher
- [ ] Apache with mod_rewrite (if using URL rewriting)
- [ ] At least 1GB free disk space
- [ ] HTTPS enabled (recommended for production)

### File Permissions
```bash
# Set proper permissions
chmod 755 assets/images/gallery/      # Web server can write
chmod 644 admin/gallery.php            # Readable by web server
chmod 644 gallery.php                  # Readable by web server
```

---

## Step-by-Step Deployment

### 1. Database Setup

**Option A: Automatic (Easiest)**
```
1. Log in to admin panel
2. Navigate to admin/gallery.php
3. Table auto-creates on first access
4. Done ✓
```

**Option B: Manual SQL**
```
1. Open phpMyAdmin
2. Select database: bermiz_db
3. Go to SQL tab
4. Copy content from admin/gallery_setup.sql
5. Click "Go" to execute
6. Verify table created ✓
```

**Option C: Command Line**
```bash
mysql -u root -p bermiz_db < admin/gallery_setup.sql
```

### 2. File Verification

Verify all files created:
```bash
✓ admin/gallery.php                    # Admin panel (1015 lines)
✓ admin/gallery_setup.sql              # Database schema
✓ admin/GALLERY_DOCUMENTATION.md      # Full documentation
✓ gallery.php (modified)               # Website gallery (database-driven)
✓ admin/index.php (modified)           # Dashboard with gallery action
✓ admin/sidebar.php (verified)         # Has gallery navigation
```

### 3. Test Database Connection

```php
<?php
// test_connection.php (create temporarily)
include 'admin/db_connect.php';

if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}

// Check gallery table exists
$result = $conn->query("DESC gallery");
if (!$result) {
    die("Gallery table missing. Run admin/gallery_setup.sql");
}

echo "✓ Database connection OK";
echo "✓ Gallery table exists";
echo "✓ All columns present";
?>
```

### 4. Test Admin Access

1. Log in to admin: `http://yoursite/admin/`
2. Click "Gallery" in sidebar
3. Should see:
   - [ ] Upload form at top
   - [ ] Empty gallery table (if first time)
   - [ ] No errors in browser console

### 5. Test Image Upload

1. Fill form:
   - Title: "Test Image"
   - Category: "Food"
   - File: Any JPG/PNG image
2. Click "Add Image"
3. Verify:
   - [ ] Success message appears
   - [ ] Image in table
   - [ ] Thumbnail visible
   - [ ] File on server: `assets/images/gallery/`

### 6. Test Website Gallery

1. Open `gallery.php` in browser
2. Verify:
   - [ ] Image appears
   - [ ] Category correct
   - [ ] Filtering works
   - [ ] Lightbox works
   - [ ] Responsive on mobile

### 7. Test Edit/Delete

1. Click "Edit" on image:
   - [ ] Modal opens
   - [ ] Title/category pre-filled
   - [ ] Save works
2. Click "Delete" on image:
   - [ ] Confirmation shows
   - [ ] Image removed from DB
   - [ ] File deleted from server
   - [ ] Website updates

### 8. Security Check

```bash
# Verify admin authentication required
1. Log out of admin
2. Try accessing http://yoursite/admin/gallery.php directly
3. Should redirect to login ✓

# Verify file upload security
1. Try uploading .php file → rejected ✓
2. Try uploading >5MB file → rejected ✓
3. Try uploading .zip file → rejected ✓
```

---

## Production Deployment

### Environment Configuration

```php
// Define in config.php
define('MAX_IMAGE_SIZE', 5 * 1024 * 1024);
define('UPLOAD_DIR', 'assets/images/gallery/');
define('ALLOWED_TYPES', ['image/jpeg', 'image/png', 'image/gif', 'image/webp']);
```

### Performance Optimization

```sql
-- Add database indexes for speed
CREATE INDEX idx_category ON gallery(category);
CREATE INDEX idx_created_date ON gallery(created_date DESC);
```

### Backup Strategy

```bash
# Before deployment
mysqldump -u root -p bermiz_db > backup_$(date +%Y%m%d).sql
cp -r assets/images/gallery assets/images/gallery.backup_$(date +%Y%m%d)
```

### Monitoring

Add logging for uploads:
```php
// In admin/gallery.php after successful upload
error_log("Image uploaded: " . $filename . " by admin_" . $_SESSION['admin_id']);
```

---

## Rollback Plan

If issues occur:

### Database Rollback
```bash
# Restore from backup
mysql -u root -p bermiz_db < backup_20240101.sql
```

### File Rollback
```bash
# Delete recent uploads
rm -rf assets/images/gallery/*.jpg
rm -rf assets/images/gallery/*.png

# Restore from backup
cp -r assets/images/gallery.backup/* assets/images/gallery/
```

### Code Rollback
```bash
# If code has issues
git revert HEAD
git push
```

---

## Go-Live Checklist

### 1 Week Before
- [ ] Database backed up
- [ ] Code reviewed
- [ ] Documentation updated
- [ ] Team trained

### 1 Day Before
- [ ] Test full deployment on staging
- [ ] Database integrity check
- [ ] File permissions verified
- [ ] Backup strategies tested

### Go-Live Day
- [ ] Announce to users (if needed)
- [ ] Monitor error logs
- [ ] Test admin panel
- [ ] Test website gallery
- [ ] Monitor performance

### 24 Hours After
- [ ] Check error logs
- [ ] Verify images displayed correctly
- [ ] Check admin functionality
- [ ] Get user feedback

---

## Hosting Provider Setup

### If Using Shared Hosting

1. **Create Database**
   - Name: `hotel_gallery` or use existing
   - User: Created by host
   - Password: Secure password
   - Host: Usually `localhost` or provider-specific

2. **Set File Permissions**
   ```
   admin/gallery.php: 644
   assets/images/gallery/: 755
   admin/gallery_setup.sql: 644
   ```

3. **Upload Files**
   - Via FTP or File Manager
   - admin/gallery.php → /public_html/admin/
   - Gallery setup files → /public_html/admin/
   - Updated gallery.php → /public_html/

4. **Test on Host**
   - Open admin/gallery.php
   - Try uploading image
   - Verify on website

### If Using cPanel

1. Create database via cPanel → MySQL Databases
2. Create user with all privileges
3. Import gallery_setup.sql via phpMyAdmin
4. Upload files via FTP
5. Test functionality

### If Using Docker

```dockerfile
FROM php:7.4-apache

# Install MySQL extension
RUN docker-php-ext-install mysqli

# Enable mod_rewrite
RUN a2enmod rewrite

# Set working directory
WORKDIR /var/www/html

# Copy files
COPY . .

# Set permissions
RUN chmod 755 assets/images/gallery/
```

---

## Maintenance Schedule

### Daily
- [ ] Check error logs
- [ ] Monitor disk space
- [ ] Verify images display correctly

### Weekly
- [ ] Check database size
- [ ] Review admin usage
- [ ] Backup database

### Monthly
- [ ] Update PHP/MySQL
- [ ] Review security logs
- [ ] Clean up orphaned files
- [ ] Database optimization

### Quarterly
- [ ] Full system backup
- [ ] Security audit
- [ ] Performance review
- [ ] Update documentation

---

## Troubleshooting Deployment

### Issue: Gallery table not created
**Solution**:
1. Manual run: `admin/gallery_setup.sql` in phpMyAdmin
2. Verify table: `SHOW TABLES LIKE 'gallery'`
3. Check columns: `DESCRIBE gallery`

### Issue: Images not uploading
**Solution**:
1. Check directory permissions: `ls -la assets/images/gallery/`
2. Change if needed: `chmod 755 assets/images/gallery/`
3. Verify PHP settings: `php.ini` upload_max_filesize ≥ 5MB

### Issue: Images not showing on website
**Solution**:
1. Check database query: `SELECT COUNT(*) FROM gallery`
2. Verify image paths: `SELECT image_path FROM gallery`
3. Test path exists: Check file in `assets/images/gallery/`

### Issue: Admin authentication failing
**Solution**:
1. Verify `check_admin.php` included
2. Check `$_SESSION` variable set
3. Verify login cookie enabled

---

## Performance Tuning

### Database Optimization
```sql
-- Analyze table for better query planning
ANALYZE TABLE gallery;

-- Optimize table storage
OPTIMIZE TABLE gallery;

-- Check table for errors
CHECK TABLE gallery;
```

### Apache Configuration
```apache
# Enable gzip compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml
</IfModule>

# Enable browser caching
<IfModule mod_expires.c>
    ExpiresActive On
    ExpiresByType image/jpeg "access plus 30 days"
    ExpiresByType image/gif "access plus 30 days"
    ExpiresByType image/png "access plus 30 days"
</IfModule>
```

### PHP Configuration
```php
// In admin/gallery.php
ini_set('max_execution_time', 300); // 5 minutes for large uploads
ini_set('upload_max_filesize', '5M');
ini_set('post_max_size', '10M');
```

---

## Monitoring & Alerts

### Set Up Error Logging
```php
// In admin/gallery.php
ini_set('log_errors', 1);
ini_set('error_log', '/var/log/php_gallery_errors.log');

// Log important events
error_log("Image uploaded: " . date('Y-m-d H:i:s') . " - " . $filename);
error_log("Image deleted: " . date('Y-m-d H:i:s') . " - " . $id);
```

### Monitor Disk Space
```bash
# Check disk usage
du -sh assets/images/gallery/

# Set up cron alert if > 1GB
0 */4 * * * if [ $(du -sb /path/to/gallery | cut -f1) -gt 1073741824 ]; then mail -s "Gallery over 1GB" admin@site.com; fi
```

### Database Monitoring
```sql
-- Check table size
SELECT 
    TABLE_NAME,
    ROUND(((data_length + index_length) / 1024 / 1024), 2) AS 'Size in MB'
FROM information_schema.TABLES
WHERE TABLE_NAME = 'gallery';
```

---

## Documentation for End Users

### Admin Documentation Provided
- ✓ GALLERY_QUICK_START.md
- ✓ GALLERY_DOCUMENTATION.md
- ✓ GALLERY_IMPLEMENTATION_SUMMARY.md
- ✓ GALLERY_ARCHITECTURE.md
- ✓ GALLERY_TESTING_CHECKLIST.md
- ✓ GALLERY_CODE_REFERENCE.md
- ✓ GALLERY_DOCUMENTATION_INDEX.md

### Share with Admins
1. Email GALLERY_QUICK_START.md (5-minute read)
2. Provide link to full documentation
3. Schedule brief training session
4. Offer email support

---

## Support & SLA

### Response Times
- **Critical Issues** (site down): 1 hour
- **High Priority** (feature broken): 4 hours
- **Medium Priority** (slow performance): 24 hours
- **Low Priority** (cosmetic): 48 hours

### Escalation Path
1. Check GALLERY_DOCUMENTATION.md troubleshooting
2. Review error logs
3. Test in staging environment
4. Contact hosting provider if needed
5. Restore from backup if necessary

---

## Final Deployment Checklist

- [ ] Database created and verified
- [ ] All files deployed correctly
- [ ] File permissions set properly
- [ ] Admin can access gallery panel
- [ ] Test image uploads successfully
- [ ] Website displays gallery correctly
- [ ] Filtering and lightbox work
- [ ] Security checks passed
- [ ] Performance acceptable
- [ ] Backups in place
- [ ] Documentation provided
- [ ] Team trained
- [ ] Monitoring configured
- [ ] Support process documented

---

## Success Criteria

✓ **Functionality**
- All CRUD operations work
- Website displays images from database
- Filtering works correctly
- Lightbox preview functional

✓ **Performance**
- Page loads in <3 seconds
- Image upload/delete <5 seconds
- Database queries optimized
- No N+1 query problems

✓ **Security**
- Admin authentication required
- File upload validated
- SQL injection prevented
- XSS prevention enabled
- Proper permissions set

✓ **Usability**
- Admin interface intuitive
- Clear error messages
- Responsive on all devices
- Mobile-friendly

✓ **Reliability**
- Database backups working
- File backups in place
- Error logging active
- System stable

---

## Post-Deployment Review

### 1 Week After Launch
- [ ] Gather user feedback
- [ ] Review error logs
- [ ] Check performance metrics
- [ ] Verify backup integrity

### 1 Month After Launch
- [ ] Performance analysis
- [ ] Security audit
- [ ] User adoption review
- [ ] Document lessons learned

### Quarterly Reviews
- [ ] System health check
- [ ] Database optimization
- [ ] Security updates
- [ ] Feature requests review

---

## Contact & Support

For deployment issues:

1. **Check Documentation**: GALLERY_DOCUMENTATION.md
2. **Review Code Comments**: In admin/gallery.php
3. **Check Error Logs**: PHP error log
4. **Database Verification**: Run DESCRIBE gallery;
5. **Contact Support**: With error log and steps to reproduce

---

## Success! 🎉

You have successfully deployed the Gallery Management System!

**Next Steps**:
1. Train admins on system
2. Populate gallery with images
3. Monitor for first week
4. Gather user feedback
5. Plan future enhancements

---

*Gallery Deployment Guide v1.0*
*Production Ready Checklist*
