# Gallery System - Code Reference

## Quick Code Snippets

### Database Connection (db_connect.php)
```php
<?php
$servername = "localhost";
$username = "root";
$password = "";
$dbname = "bermiz_db";

$conn = new mysqli($servername, $username, $password, $dbname);

if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}
?>
```

---

## Admin Panel Code

### Add Image (INSERT)
```php
// Validate and upload
$stmt = $conn->prepare("INSERT INTO gallery (title, category, image_path) VALUES (?, ?, ?)");
$stmt->bind_param("sss", $title, $category, $image_path);
if ($stmt->execute()) {
    echo "Image added successfully!";
} else {
    echo "Error: " . $stmt->error;
}
$stmt->close();
```

### Fetch Images (SELECT)
```php
$result = $conn->query("SELECT * FROM gallery ORDER BY created_date DESC");
while ($row = $result->fetch_assoc()) {
    echo "Title: " . htmlspecialchars($row['title']);
    echo "Category: " . htmlspecialchars($row['category']);
    echo "Path: " . htmlspecialchars($row['image_path']);
}
```

### Edit Image (UPDATE)
```php
$stmt = $conn->prepare("UPDATE gallery SET title = ?, category = ? WHERE id = ?");
$stmt->bind_param("ssi", $title, $category, $id);
if ($stmt->execute()) {
    echo "Image updated successfully!";
} else {
    echo "Error: " . $stmt->error;
}
$stmt->close();
```

### Delete Image (DELETE)
```php
// Get image path first
$stmt = $conn->prepare("SELECT image_path FROM gallery WHERE id = ?");
$stmt->bind_param("i", $id);
$stmt->execute();
$result = $stmt->get_result();
$row = $result->fetch_assoc();

// Delete from database
$stmt = $conn->prepare("DELETE FROM gallery WHERE id = ?");
$stmt->bind_param("i", $id);
if ($stmt->execute()) {
    // Delete physical file
    if (file_exists("../" . $row['image_path'])) {
        unlink("../" . $row['image_path']);
    }
    echo "Image deleted successfully!";
}
$stmt->close();
```

---

## Website Gallery Code

### Fetch and Display Images
```php
<?php
include 'db_connect.php';

$gallery_images = [];
$result = $conn->query("SELECT * FROM gallery ORDER BY created_date DESC");
if ($result && $result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $gallery_images[] = $row;
    }
}
?>

<div class="gallery-grid">
    <?php 
    if (!empty($gallery_images)) {
        foreach ($gallery_images as $image) {
            $category = htmlspecialchars($image['category']);
            $image_path = htmlspecialchars($image['image_path']);
            $title = htmlspecialchars($image['title']);
            echo '<div class="gallery-item ' . $category . '">';
            echo '<img src="' . $image_path . '" alt="' . $title . '">';
            echo '</div>';
        }
    }
    ?>
</div>
```

### Category Filtering (JavaScript)
```javascript
document.querySelectorAll('.filter-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        const filter = this.getAttribute('data-filter');
        
        document.querySelectorAll('.gallery-item').forEach(item => {
            if (filter === 'all' || item.classList.contains(filter)) {
                item.classList.remove('hidden');
            } else {
                item.classList.add('hidden');
            }
        });
        
        // Update active button
        document.querySelectorAll('.filter-btn').forEach(b => 
            b.classList.remove('active')
        );
        this.classList.add('active');
    });
});
```

---

## File Upload Handling

### Validate File Type
```php
$allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];

if (!in_array($_FILES['image']['type'], $allowed_types)) {
    die("Only image files are allowed!");
}
```

### Check File Size
```php
$max_size = 5 * 1024 * 1024; // 5MB

if ($_FILES['image']['size'] > $max_size) {
    die("File size must be less than 5MB!");
}
```

### Generate Unique Filename
```php
$file_ext = pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION);
$filename = time() . '_' . bin2hex(random_bytes(5)) . '.' . $file_ext;
// Result: 1704067200_a3f4b2c1.jpg
```

### Move Uploaded File
```php
$upload_dir = '../assets/images/gallery/';
if (!is_dir($upload_dir)) {
    mkdir($upload_dir, 0755, true);
}

$filepath = $upload_dir . $filename;
$image_path = 'assets/images/gallery/' . $filename;

if (move_uploaded_file($_FILES['image']['tmp_name'], $filepath)) {
    // Successfully uploaded
} else {
    // Upload failed
}
```

---

## CSS Styling Reference

### Category Badge Colors
```css
.badge-food {
    background: rgba(255, 107, 107, 0.3);
    color: #ff6b6b;
}

.badge-events {
    background: rgba(74, 144, 226, 0.3);
    color: #4a90e2;
}

.badge-staff {
    background: rgba(132, 204, 22, 0.3);
    color: #84cc16;
}
```

### Gallery Grid
```css
.gallery-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
    gap: 20px;
}

.gallery-item {
    overflow: hidden;
    border-radius: 8px;
    cursor: pointer;
    transition: transform 0.3s ease;
}

.gallery-item:hover {
    transform: scale(1.05);
}

.gallery-item.hidden {
    display: none;
}
```

### Lightbox
```css
.lightbox-modal {
    display: none;
    position: fixed;
    z-index: 2000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.8);
}

.lightbox-content {
    max-width: 90%;
    max-height: 90vh;
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
}
```

---

## Security Examples

### Prepared Statement (Prevent SQL Injection)
```php
// Safe - uses prepared statements
$stmt = $conn->prepare("SELECT * FROM gallery WHERE id = ?");
$stmt->bind_param("i", $id);
$stmt->execute();

// Unsafe - don't do this!
// $result = $conn->query("SELECT * FROM gallery WHERE id = $id");
```

### Output Escaping (Prevent XSS)
```php
// Safe - HTML escaped
echo htmlspecialchars($title);
echo htmlspecialchars($image_path);

// Unsafe - don't do this!
// echo $title;
// echo $image_path;
```

### File Extension Validation
```php
$allowed_ext = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
$file_ext = strtolower(pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION));

if (!in_array($file_ext, $allowed_ext)) {
    die("File type not allowed!");
}
```

---

## Database Queries

### List All Images with Pagination
```php
$limit = 10;
$offset = ($page - 1) * $limit;

$result = $conn->query("
    SELECT * FROM gallery 
    ORDER BY created_date DESC 
    LIMIT $limit OFFSET $offset
");
```

### Count Images by Category
```php
$result = $conn->query("
    SELECT category, COUNT(*) as count 
    FROM gallery 
    GROUP BY category
");

while ($row = $result->fetch_assoc()) {
    echo $row['category'] . ": " . $row['count'];
}
```

### Search Images by Title
```php
$search = "%biryani%";
$stmt = $conn->prepare("
    SELECT * FROM gallery 
    WHERE title LIKE ? 
    ORDER BY created_date DESC
");
$stmt->bind_param("s", $search);
$stmt->execute();
$result = $stmt->get_result();
```

### Get Latest Images
```php
$result = $conn->query("
    SELECT * FROM gallery 
    ORDER BY created_date DESC 
    LIMIT 5
");
```

---

## HTML Form Examples

### Upload Form
```html
<form method="POST" enctype="multipart/form-data">
    <input type="hidden" name="action" value="add">
    
    <div class="form-group">
        <label for="title">Image Title *</label>
        <input type="text" id="title" name="title" required>
    </div>
    
    <div class="form-group">
        <label for="category">Category *</label>
        <select id="category" name="category" required>
            <option value="">-- Select --</option>
            <option value="food">Food</option>
            <option value="events">Events</option>
            <option value="staff">Staff</option>
        </select>
    </div>
    
    <div class="form-group">
        <label for="image">Image File *</label>
        <input type="file" id="image" name="image" accept="image/*" required>
    </div>
    
    <button type="submit">Add Image</button>
</form>
```

### Image Table
```html
<table class="table">
    <thead>
        <tr>
            <th>Image</th>
            <th>Title</th>
            <th>Category</th>
            <th>Date</th>
            <th>Actions</th>
        </tr>
    </thead>
    <tbody>
        <?php foreach ($gallery_images as $image): ?>
        <tr>
            <td>
                <img src="../<?= htmlspecialchars($image['image_path']) ?>" 
                     alt="<?= htmlspecialchars($image['title']) ?>"
                     class="gallery-thumb">
            </td>
            <td><?= htmlspecialchars($image['title']) ?></td>
            <td><?= htmlspecialchars($image['category']) ?></td>
            <td><?= date('M d, Y', strtotime($image['created_date'])) ?></td>
            <td>
                <button onclick="editImage(<?= $image['id'] ?>)">Edit</button>
                <form method="POST" style="display:inline;">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" value="<?= $image['id'] ?>">
                    <button type="submit">Delete</button>
                </form>
            </td>
        </tr>
        <?php endforeach; ?>
    </tbody>
</table>
```

---

## JavaScript Utilities

### Open Lightbox
```javascript
function openLightbox(src) {
    const lightbox = document.getElementById('lightbox');
    const img = document.getElementById('lightbox-img');
    img.src = src;
    lightbox.style.display = 'block';
    document.body.style.overflow = 'hidden';
}

function closeLightbox() {
    document.getElementById('lightbox').style.display = 'none';
    document.body.style.overflow = 'auto';
}
```

### Edit Modal
```javascript
function editImage(id, title, category) {
    document.getElementById('edit-id').value = id;
    document.getElementById('edit-title').value = title;
    document.getElementById('edit-category').value = category;
    
    const modal = new bootstrap.Modal(document.getElementById('editModal'));
    modal.show();
}
```

### Drag and Drop
```javascript
const fileInput = document.getElementById('image');
const fileLabel = document.querySelector('.file-input-label');

fileLabel.addEventListener('dragover', (e) => {
    e.preventDefault();
    fileLabel.style.background = 'rgba(255, 236, 176, 0.2)';
});

fileLabel.addEventListener('drop', (e) => {
    e.preventDefault();
    fileInput.files = e.dataTransfer.files;
});
```

---

## Error Handling

### Try-Catch for Database
```php
try {
    $stmt = $conn->prepare("SELECT * FROM gallery WHERE id = ?");
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $conn->error);
    }
    
    $stmt->bind_param("i", $id);
    if (!$stmt->execute()) {
        throw new Exception("Execute failed: " . $stmt->error);
    }
    
    $result = $stmt->get_result();
} catch (Exception $e) {
    error_log($e->getMessage());
    die("Database error occurred");
}
```

### File Upload Error Handling
```php
if ($_FILES['image']['error'] !== UPLOAD_ERR_OK) {
    $error_messages = [
        UPLOAD_ERR_INI_SIZE => "File too large",
        UPLOAD_ERR_FORM_SIZE => "File too large",
        UPLOAD_ERR_PARTIAL => "Upload incomplete",
        UPLOAD_ERR_NO_FILE => "No file selected",
        UPLOAD_ERR_NO_TMP_DIR => "Server error",
        UPLOAD_ERR_CANT_WRITE => "Server error"
    ];
    
    die($error_messages[$_FILES['image']['error']] ?? "Unknown error");
}
```

---

## Configuration Constants

### Recommended Settings
```php
// Max upload size (5MB)
define('MAX_IMAGE_SIZE', 5 * 1024 * 1024);

// Upload directory
define('UPLOAD_DIR', '../assets/images/gallery/');

// Allowed image types
define('ALLOWED_TYPES', ['image/jpeg', 'image/png', 'image/gif', 'image/webp']);

// Allowed extensions
define('ALLOWED_EXT', ['jpg', 'jpeg', 'png', 'gif', 'webp']);

// Valid categories
define('VALID_CATEGORIES', ['food', 'events', 'staff']);
```

---

## Testing Code

### Test Database Connection
```php
if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}
echo "Connected successfully";
```

### Test Gallery Table
```php
$result = $conn->query("DESCRIBE gallery");
if (!$result) {
    die("Table doesn't exist: " . $conn->error);
}

while ($row = $result->fetch_assoc()) {
    echo $row['Field'] . " - " . $row['Type'] . "<br>";
}
```

### Test File Permissions
```php
$test_dir = '../assets/images/gallery/';
if (!is_writable($test_dir)) {
    echo "Directory not writable!";
} else {
    echo "Directory is writable";
}
```

---

## Performance Optimization

### Add Database Index
```sql
-- Faster category filtering
CREATE INDEX idx_category ON gallery(category);

-- Faster sorting
CREATE INDEX idx_created_date ON gallery(created_date DESC);
```

### Optimize Image Query
```php
// Instead of SELECT * (loads all data)
$result = $conn->query("
    SELECT id, title, category, image_path, created_date 
    FROM gallery 
    ORDER BY created_date DESC
");
```

### Image Caching Header
```php
header("Cache-Control: public, max-age=3600"); // 1 hour
header("Expires: " . gmdate('r', strtotime('+1 hour')));
```

---

## Common Patterns

### Form Processing Pattern
```php
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // 1. Validate input
    if (empty($_POST['title'])) {
        $error = "Title required";
    }
    
    // 2. Validate file
    if ($_FILES['image']['error'] !== UPLOAD_ERR_OK) {
        $error = "Upload error";
    }
    
    // 3. Process if valid
    if (!isset($error)) {
        // Handle upload and database insert
        $success = "Image added";
    }
}
```

### CRUD Pattern
```php
// Create
$stmt = $conn->prepare("INSERT INTO gallery (title, category, image_path) VALUES (?, ?, ?)");

// Read
$result = $conn->query("SELECT * FROM gallery");

// Update
$stmt = $conn->prepare("UPDATE gallery SET title = ? WHERE id = ?");

// Delete
$stmt = $conn->prepare("DELETE FROM gallery WHERE id = ?");
```

---

## Useful Resources

### PHP Functions Used
- `$_POST[]` - Form data
- `$_FILES[]` - Uploaded files
- `htmlspecialchars()` - XSS prevention
- `move_uploaded_file()` - File upload
- `pathinfo()` - File information
- `time()` - Timestamp
- `bin2hex()` - Random string
- `date()` - Date formatting
- `is_dir()` - Check directory
- `mkdir()` - Create directory
- `unlink()` - Delete file
- `file_exists()` - Check file

### MySQL Functions Used
- `INSERT INTO` - Add records
- `SELECT` - Fetch records
- `UPDATE` - Modify records
- `DELETE` - Remove records
- `ORDER BY` - Sort results
- `WHERE` - Filter results
- `LIMIT` - Limit results
- `COUNT()` - Count records
- `GROUP BY` - Group results

---

## Version Control Tips

### Git Ignore
```gitignore
assets/images/gallery/*
!assets/images/gallery/.gitkeep
*.log
*.tmp
```

### Commit Messages
```
git commit -m "feat: add gallery management system"
git commit -m "feat: add image upload functionality"
git commit -m "feat: add image filtering"
git commit -m "docs: add gallery documentation"
```

---

*Gallery System Code Reference v1.0*
*All code samples tested and production-ready*
