# Session-Based Cart CRUD System Documentation

## Overview
The restaurant app now uses a **session-based cart system** that replaces the previous localStorage-only approach. All cart operations are managed through server-side sessions with backup localStorage synchronization.

---

## File Structure

### Backend Files
- **`cart-handler.php`** - Main CRUD API handler for all cart operations
- **`cart-actions.php`** - Legacy database operations (still available)
- **`cart-view.php`** - Display cart contents page

### Frontend Files
- **`online-delivery.php`** - Menu items page with add to cart functionality
- **`checkout-page.php`** - Checkout page with cart display and CRUD operations
- **`login.php`** - Login/registration page

---

## API Endpoints

All endpoints use `cart-handler.php`. Requests can be GET or POST:

### 1. **FETCH CART** (Read)
```javascript
fetch('cart-handler.php?action=fetch')
    .then(response => response.json())
    .then(data => console.log(data.items));
```

**Response:**
```json
{
    "status": "success",
    "items": [
        {
            "id": "1",
            "name": "Biryani",
            "price": 250,
            "image": "assets/images/biryani.jpg",
            "quantity": 2,
            "added_at": "2025-12-24 10:30:45"
        }
    ],
    "count": 1,
    "subtotal": 500
}
```

---

### 2. **ADD ITEM TO CART** (Create)
```javascript
const formData = new FormData();
formData.append('action', 'add');
formData.append('name', 'Biryani');
formData.append('price', 250);
formData.append('image', 'assets/images/biryani.jpg');
formData.append('quantity', 1);

fetch('cart-handler.php', {
    method: 'POST',
    body: formData
})
.then(response => response.json())
.then(data => console.log(data));
```

**Parameters:**
| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `action` | string | ✓ | Must be `"add"` |
| `id` | string | ✗ | Item ID (optional) |
| `name` | string | ✓ | Item name |
| `price` | float | ✓ | Item price (e.g., 250.50) |
| `image` | string | ✓ | Image URL/path |
| `quantity` | int | ✓ | Quantity to add (default: 1) |

**Response:**
```json
{
    "status": "success",
    "message": "Item added to cart",
    "items": [...],
    "count": 1
}
```

**Logic:**
- If item name already exists → increment quantity
- If new item → add to session cart
- Session persists across page navigation
- Automatically syncs to localStorage

---

### 3. **UPDATE ITEM QUANTITY** (Update)
```javascript
const formData = new FormData();
formData.append('action', 'update_quantity');
formData.append('name', 'Biryani');
formData.append('quantity', 5);

fetch('cart-handler.php', {
    method: 'POST',
    body: formData
})
.then(response => response.json())
.then(data => console.log(data));
```

**Parameters:**
| Parameter | Type | Description |
|-----------|------|-------------|
| `action` | string | Must be `"update_quantity"` |
| `name` | string | Item name (exact match) |
| `quantity` | int | New quantity (if ≤0, item is removed) |

**Response:**
```json
{
    "status": "success",
    "message": "Quantity updated",
    "items": [...],
    "subtotal": 1250
}
```

**Behavior:**
- If quantity ≤ 0 → removes item from cart
- If quantity > 0 → updates existing quantity
- Recalculates subtotal

---

### 4. **REMOVE ITEM FROM CART** (Delete)
```javascript
const formData = new FormData();
formData.append('action', 'remove');
formData.append('name', 'Biryani');

fetch('cart-handler.php', {
    method: 'POST',
    body: formData
})
.then(response => response.json())
.then(data => console.log(data));
```

**Parameters:**
| Parameter | Type | Description |
|-----------|------|-------------|
| `action` | string | Must be `"remove"` |
| `name` | string | Item name to remove |

**Response:**
```json
{
    "status": "success",
    "message": "Item removed from cart",
    "items": [...],
    "count": 0,
    "subtotal": 0
}
```

---

### 5. **CLEAR ENTIRE CART** (Delete All)
```javascript
const formData = new FormData();
formData.append('action', 'clear');

fetch('cart-handler.php', {
    method: 'POST',
    body: formData
})
.then(response => response.json())
.then(data => console.log(data));
```

**Response:**
```json
{
    "status": "success",
    "message": "Cart cleared",
    "items": [],
    "count": 0
}
```

---

### 6. **GET CART COUNT** (Read)
```javascript
fetch('cart-handler.php?action=count')
    .then(response => response.json())
    .then(data => console.log(data.count));
```

**Response:**
```json
{
    "status": "success",
    "count": 3,
    "subtotal": 1500
}
```

---

### 7. **EXPORT CART DATA** (Read with Timestamp)
```javascript
fetch('cart-handler.php?action=export')
    .then(response => response.json())
    .then(data => console.log(data));
```

**Response:**
```json
{
    "status": "success",
    "items": [...],
    "count": 3,
    "subtotal": 1500,
    "timestamp": "2025-12-24 10:35:20"
}
```

---

## Frontend Implementation

### Online Delivery Page (Adding Items)
```javascript
// Automatically triggered on button click
const formData = new FormData();
formData.append('action', 'add');
formData.append('name', name);
formData.append('price', price);
formData.append('image', imgPath);
formData.append('quantity', 1);

fetch('cart-handler.php', {
    method: 'POST',
    body: formData
})
.then(response => response.json())
.then(data => {
    localStorage.setItem('cart', JSON.stringify(data.items));
    updateCartCount(data.count);
    showCartNotification(name);
});
```

### Checkout Page (CRUD Operations)
```javascript
// Increase Quantity
function increaseQuantity(itemName) {
    fetch('cart-handler.php?action=fetch')
        .then(r => r.json())
        .then(data => {
            const item = data.items.find(i => i.name === itemName);
            if (item) updateQuantity(itemName, item.quantity + 1);
        });
}

// Decrease Quantity
function decreaseQuantity(itemName) {
    fetch('cart-handler.php?action=fetch')
        .then(r => r.json())
        .then(data => {
            const item = data.items.find(i => i.name === itemName);
            if (item) updateQuantity(itemName, item.quantity - 1);
        });
}

// Remove Item
function removeItem(itemName) {
    if (confirm('Remove this item?')) {
        const formData = new FormData();
        formData.append('action', 'remove');
        formData.append('name', itemName);
        
        fetch('cart-handler.php', {method: 'POST', body: formData})
            .then(r => r.json())
            .then(data => {
                loadCartData();
                syncLocalStorage(data.items);
            });
    }
}

// Clear Cart
function clearCart() {
    if (confirm('Clear entire cart?')) {
        const formData = new FormData();
        formData.append('action', 'clear');
        
        fetch('cart-handler.php', {method: 'POST', body: formData})
            .then(r => r.json())
            .then(data => loadCartData());
    }
}
```

---

## Data Flow

```
User Action (Menu Page)
    ↓
Add to Cart Button Clicked
    ↓
cart-handler.php (POST: action=add)
    ↓
Update Session[$_SESSION['cart']]
    ↓
Response with updated items
    ↓
JavaScript syncs to localStorage
    ↓
Notification displayed
    
---

User Action (Checkout Page)
    ↓
Increase/Decrease/Remove Quantity
    ↓
cart-handler.php (POST: action=update_quantity/remove)
    ↓
Update Session
    ↓
Display updated cart
    ↓
Refresh totals
```

---

## Session Cart Structure

```php
$_SESSION['cart'] = [
    [
        'id' => '1',
        'name' => 'Biryani',
        'price' => 250.50,
        'image' => 'assets/images/biryani.jpg',
        'quantity' => 2,
        'added_at' => '2025-12-24 10:30:45'
    ],
    [
        'id' => '2',
        'name' => 'Naan',
        'price' => 50.00,
        'image' => 'assets/images/naan.jpg',
        'quantity' => 3,
        'added_at' => '2025-12-24 10:31:10'
    ]
];
```

---

## Error Handling

All errors return HTTP 400 with error response:

```json
{
    "status": "error",
    "message": "Invalid item data"
}
```

Common errors:
- Missing required parameters
- Invalid action name
- Database connection issues
- Session initialization failures

---

## Key Features

✅ **Session Persistence** - Cart data survives page navigation
✅ **localStorage Sync** - Backup to localStorage for offline access
✅ **Real-time Updates** - Instant quantity changes and total recalculation
✅ **Duplicate Prevention** - Adding same item increases quantity (not duplicated)
✅ **Auto-calculation** - Subtotal, taxes, delivery fees calculated automatically
✅ **CRUD Complete** - Full Create, Read, Update, Delete operations
✅ **Error Handling** - Validation and error responses
✅ **User Notifications** - Visual feedback for all actions

---

## Integration Points

### Online Delivery Page
- **Add Items**: Uses `action=add`
- **Display Count**: Uses `action=count`
- **Sync localStorage**: After each add operation

### Checkout Page
- **Fetch Cart**: Uses `action=fetch` on page load
- **Update Quantity**: Uses `action=update_quantity`
- **Remove Items**: Uses `action=remove`
- **Export for Order**: Uses `action=export`
- **Clear on Success**: Uses `action=clear`

### Future Integration
- **Database Storage**: Can add `user_id` to session to save order to `orders` table
- **Order History**: Query `$_SESSION['cart']` at checkout to create order record
- **Analytics**: Track cart additions/removals for user behavior
- **Abandoned Cart**: Store session cart for recovery emails

---

## Testing Endpoints

Using **curl** or **Postman**:

```bash
# Fetch cart
curl "http://localhost/hotel/cart-handler.php?action=fetch"

# Add item
curl -X POST "http://localhost/hotel/cart-handler.php" \
  -d "action=add&name=Biryani&price=250&image=path/to/image.jpg"

# Update quantity
curl -X POST "http://localhost/hotel/cart-handler.php" \
  -d "action=update_quantity&name=Biryani&quantity=5"

# Remove item
curl -X POST "http://localhost/hotel/cart-handler.php" \
  -d "action=remove&name=Biryani"

# Clear cart
curl -X POST "http://localhost/hotel/cart-handler.php" \
  -d "action=clear"

# Get count
curl "http://localhost/hotel/cart-handler.php?action=count"

# Export
curl "http://localhost/hotel/cart-handler.php?action=export"
```

---

## Troubleshooting

| Issue | Solution |
|-------|----------|
| Cart not persisting | Ensure `session_start()` is called in all files |
| Duplicate items | Check item name comparison logic |
| Quantity not updating | Verify item name matches exactly (case-sensitive) |
| localStorage not syncing | Check if JavaScript errors exist in console |
| Response 400 error | Verify all required parameters are sent |
| Cart shows 0 items | Clear cache and restart session |

---

## Security Considerations

✅ Session-based storage (more secure than localStorage)
✅ Server-side validation of all parameters
✅ Input sanitization for item names and prices
✅ CSRF protection ready (add tokens if needed)
⚠️ TODO: Add user authentication check for persistent orders
⚠️ TODO: Implement rate limiting for API endpoints
⚠️ TODO: Add price validation against database

---

## Future Enhancements

1. **Database Integration**
   - Save cart to `user_carts` table
   - Enable abandoned cart recovery
   - Track cart history

2. **Advanced Features**
   - Quantity limits per item
   - Stock availability checks
   - Combo deals/bundles
   - Gift wrapping options
   - Special instructions per item

3. **Performance**
   - Redis caching for frequently accessed carts
   - Bulk operations API
   - Lazy loading for large carts

4. **Analytics**
   - Track most added items
   - Cart abandonment rates
   - Average cart value
   - Popular combinations

---

## Version History

- **v1.0** (2025-12-24) - Initial session-based CRUD implementation
  - Created `cart-handler.php` with full CRUD operations
  - Integrated with checkout page for quantity management
  - Added localStorage synchronization
  - Implemented real-time notifications

---

## Support

For issues or questions about the cart system, check:
1. Browser console for JavaScript errors
2. Server logs for PHP errors
3. Network tab in DevTools for API responses
4. Session data: Add `echo json_encode($_SESSION);` for debugging
