# Cart & Checkout System - Quick Reference

## System Overview

**Cart Management**: Session-based + Database backup  
**Flow**: Shopping → Cart View → Checkout → Payment  
**Status**: Fully functional frontend, backend order processing pending

---

## Key Features

### 🛒 Cart Management (cart-view.php)
- View all items with images
- **+/-** buttons to adjust quantity
- 🗑️ **Trash icon** to remove items
- Order summary with total calculation
- Delivery/Takeaway toggle
- "Continue Shopping" button
- "Proceed to Checkout" button

### 💳 Checkout Form (checkout-page.php)
- **Delivery Address**: Name, Phone, Street, City, State, Postal
- **5 Payment Methods**:
  1. UPI (Google Pay, PhonePe, Paytm)
  2. Card (Visa, Mastercard, RuPay)
  3. Net Banking (8 banks)
  4. Digital Wallet (Paytm, Amazon Pay, MobiKwik)
  5. Cash on Delivery
- Order Summary (Sticky sidebar)
- Real-time price calculations

### ✅ Validations
```
✓ Phone: 10 digits
✓ Postal: 6 digits  
✓ Cart: Not empty
✓ Address: All fields filled
✓ Terms: Must accept
✓ Payment: Method selected
```

---

## File Structure

```
hotel/
├── cart-actions.php          ← Backend CRUD operations
├── cart-view.php             ← Cart display & management
├── checkout-page.php         ← Checkout form & payment
└── online-delivery.php       ← Browse & add items
```

---

## Database Operations

### Add Item to Cart
```php
// cart-actions.php - Action: 'add'
POST /cart-actions.php
Parameters:
  - action: add
  - name: Product name
  - price: Product price  
  - image: Image path

Response: {"status": "success"}
```

### Update Quantity
```php
// Action: 'update_qty'
POST /cart-actions.php
Parameters:
  - action: update_qty
  - id: Item ID
  - qty: New quantity
  - name: Product name

Response: {"status": "success"}
```

### Remove Item
```php
// Action: 'remove'
POST /cart-actions.php
Parameters:
  - action: remove
  - id: Item ID
  - name: Product name

Response: {"status": "success"}
```

### Fetch Cart
```php
// Action: 'fetch'
POST /cart-actions.php
Parameters:
  - action: fetch

Response: [array of cart items]
```

---

## Session Variables

```php
$_SESSION['user_id']         // User ID (required for checkout)
$_SESSION['user_name']       // Display name
$_SESSION['user_email']      // Email address
$_SESSION['user_phone']      // Phone number
$_SESSION['cart']            // Array of cart items

// Cart item structure:
[
    'id' => 'item_xyz',
    'product_name' => 'Biryani',
    'product_price' => 250,
    'product_image' => 'image.jpg',
    'quantity' => 2
]
```

---

## JavaScript Functions

### cart-view.php
```javascript
loadCart()                      // Load cart from session
renderCart(cartItems)          // Display cart items
updateQty(id, newQty, name)   // Update quantity
removeItem(id, name)           // Remove with confirmation
updateSummary(subtotal)        // Calculate totals
updateCartCount(count)         // Update header count
```

### checkout-page.php
```javascript
loadCartData()                 // Load cart from session
displayCart(cart)              // Display cart items
updateTotals(subtotal)         // Calculate price breakdown
placeOrder()                   // Validate & submit order
```

---

## Pricing Breakdown

### Checkout Summary
```
Subtotal:          (Sum of all items)
+ Delivery:        ₹40 (for delivery), ₹0 (for takeaway)
+ Taxes:           5% of subtotal
- Discount:        ₹10 (hardcoded)
= Grand Total:     Final amount
```

### Example
```
Biryani (₹250) x 2 = ₹500
Butter Chicken (₹200) x 1 = ₹200
─────────────────────────────
Subtotal:         ₹700
Delivery:         ₹40
Taxes (5%):       ₹35
Discount:         -₹10
─────────────────────────────
TOTAL:            ₹765
```

---

## UI Components

### Buttons & Colors
```css
Primary (Gradient):    #667eea → #764ba2 (Checkout, Add, Apply)
Success:              #28a745 (Place Order)
Danger:               #dc3545 (Remove)
Info:                 #17a2b8 (Delivery/Takeaway)
```

### Responsive Breakpoints
```
Mobile:   < 768px    → 1-column layout
Tablet:   768-1024px → 1-2 column
Desktop:  > 1024px   → 2-column with sticky sidebar
```

---

## Common Tasks

### Add Item from Product Page
```javascript
function addToCart(name, price, image) {
    const formData = new FormData();
    formData.append('action', 'add');
    formData.append('name', name);
    formData.append('price', price);
    formData.append('image', image);
    
    fetch('cart-actions.php', {method: 'POST', body: formData})
        .then(r => r.json())
        .then(d => {
            alert('✅ Added to cart!');
            updateCartIcon();
        });
}
```

### Display Cart Count
```javascript
function updateCartIcon() {
    const formData = new FormData();
    formData.append('action', 'fetch');
    
    fetch('cart-actions.php', {method: 'POST', body: formData})
        .then(r => r.json())
        .then(items => {
            document.querySelector('.cart-count').innerText = items.length;
        });
}
```

### Check If User Can Checkout
```php
<?php
if (!isset($_SESSION['user_id'])) {
    echo "User must login first";
    header('Location: login.php');
}
?>
```

---

## Error Handling

### Validation Messages
```javascript
✗ "Please fill all required fields"       → Missing address
✗ "Please enter a valid 10-digit number"  → Invalid phone
✗ "Please enter a valid 6-digit code"     → Invalid postal
✗ "Your cart is empty"                    → No items added
✗ "Please accept terms"                   → T&C not checked
```

### Server Errors
```php
// Database connection failed
{"status": "error", "message": "Connection failed"}

// Invalid action
No response

// Success
{"status": "success"}
```

---

## Payment Methods Implementation

### Currently UI Only
Each payment method has its own form section that displays when selected:

```html
<input type="radio" name="paymentMethod" value="upi">
<div id="upiForm" class="payment-form">
    <!-- UPI ID input -->
</div>

<input type="radio" name="paymentMethod" value="card">
<div id="cardForm" class="payment-form">
    <!-- Card details form -->
</div>
```

### Next Phase: Integration
1. Send payment method to payment gateway API
2. Handle callback/webhook
3. Update order status
4. Send confirmation email

---

## Testing

### Test Cart Flow
1. Go to `online-delivery.php`
2. Click "Add to Cart" on any item
3. Go to `cart-view.php` (click cart icon)
4. Verify item appears with correct price
5. Click +/- to adjust quantity
6. Click trash to remove
7. Click "Proceed to Checkout"
8. Fill address form
9. Select payment method
10. Click "Place Order"
11. Verify confirmation message

### Test Login Requirement
1. Clear cookies/session
2. Try to access `checkout-page.php`
3. Should redirect to `login.php`

### Test Responsive Design
1. Open `cart-view.php`
2. Resize browser window
3. Verify layout adjusts properly
4. Test on mobile device

---

## Troubleshooting

| Issue | Solution |
|-------|----------|
| Items not showing in cart | Check `$_SESSION['cart']` initialization |
| Quantity not updating | Verify `cart-actions.php` is called correctly |
| Checkout redirects to login | User session may have expired, login again |
| Images not loading | Check `product_image` path in database |
| Totals calculated wrong | Verify `product_price` is numeric |

---

## Next Steps (Backend)

```
1. Create orders table
   - Store order details & items
   
2. Implement order creation API
   - POST /order-handler.php
   - Save order to database
   
3. Add payment gateway
   - Integrate Razorpay/PayU
   - Handle payment response
   
4. Send confirmations
   - Email receipt
   - SMS notification
   
5. Create order tracking
   - User can view order status
   - Admin can update status
```

---

## Quick Links

- [Full Documentation](CART_CHECKOUT_INTEGRATION.md)
- [Cart Display](cart-view.php)
- [Checkout Form](checkout-page.php)
- [Backend Operations](cart-actions.php)

---

**Last Updated**: December 24, 2025  
**Version**: 1.0  
**Status**: Production Ready (Frontend)
