<?php
// Protect this page with session check
include 'session_check.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mektron - Production Report</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link rel="stylesheet" href="style.css">
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    
    <style>
        /* --- 1. GLOBAL LAYOUT (Full Width Fix) --- */
        body { 
            background: #f4f6f9; 
            font-family: 'Segoe UI', sans-serif; 
            margin: 0; 
            overflow-x: hidden;
            overflow-y: auto;
        }
        
        /* Main Content takes FULL width by default */
        .main-content { 
            margin: 0; 
            padding: 10px; 
            width: 100%; 
            box-sizing: border-box; 
            min-height: 100vh;
            transition: margin-left 0.3s ease; 
        }

        .container { 
            background: white; 
            padding: 16px; 
            width: 100%;
            max-width: none;
            box-sizing: border-box;
            border-radius: 8px; 
            box-shadow: 0 4px 15px rgba(0,0,0,0.05); 
        }

        /* --- 2. SIDEBAR TOGGLE LOGIC --- */
        .sidebar {
            position: fixed;
            top: 0;
            left: -260px; 
            width: 260px;
            height: 100%;
            background: #013B3B; 
            transition: left 0.3s ease;
            z-index: 1000;
            overflow-y: auto;
        }

        .sidebar.active {
            left: 0;
            box-shadow: 5px 0 15px rgba(0,0,0,0.2);
        }

        .menu-toggle-btn {
            background: none;
            border: none;
            font-size: 20px;
            color: #013B3B;
            cursor: pointer;
            margin-right: 10px;
            padding: 4px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
            transition: 0.2s;
        }
        .menu-toggle-btn:hover { background: #e0e0e0; }

        .overlay {
            display: none;
            position: fixed;
            top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 999;
        }
        .overlay.active { display: block; }


        /* --- 3. HEADER & CONTROLS --- */
        .header-bar { 
            display: flex; 
            justify-content: flex-start; 
            align-items: center; 
            margin-bottom: 12px; 
            border-bottom: 1px solid #eee; 
            padding-bottom: 8px; 
            flex-wrap: wrap; 
            gap: 8px; 
        }

        .header-left { display: flex; align-items: center; }

        .header-title h2 { 
            margin: 0; 
            color: #2c3e50; 
            font-size: 18px; 
            border-left: 4px solid #3498db; 
            padding-left: 8px; 
        }

        .header-meta {
            font-size: 11px; 
            color: #7f8c8d; 
            margin-top: 2px; 
            padding-left: 11px;
        }

        .admin-controls { 
            display: none; 
            gap: 8px; 
            flex-wrap: wrap; 
            align-items: center;
        } 

        /* --- 4. BUTTONS --- */
        .btn { 
            padding: 7px 12px; 
            border-radius: 5px; 
            cursor: pointer; 
            font-weight: 600; 
            border: none; 
            font-size: 12px; 
            display: inline-flex; 
            align-items: center; 
            gap: 6px; 
            white-space: nowrap;
            transition: 0.2s;
        }
        .btn-config { background: #34495e; color: white; }
        .btn-config:hover { background: #2c3e50; transform: translateY(-1px); }
        .btn-danger { background: #e74c3c; color: white; }
        .btn-danger:hover { background: #c0392b; transform: translateY(-1px); }
        .btn-print { background: #2ecc71; color: white; }
        .btn-clear { background: #95a5a6; color: white; }
        .btn-analytics {
            color: #ecfeff;
            background: linear-gradient(130deg, #0b1a28 0%, #0f766e 55%, #22d3ee 100%);
            box-shadow: 0 8px 18px rgba(15, 118, 110, 0.35);
        }
        .btn-analytics:hover {
            transform: translateY(-1px);
            box-shadow: 0 10px 22px rgba(34, 211, 238, 0.35);
        }
        .btn-advanced-analytics {
            color: #e2e8f0;
            background: linear-gradient(130deg, #0a0f1f 0%, #111827 52%, #1f2937 100%);
            box-shadow: 0 8px 18px rgba(15, 23, 42, 0.35);
        }
        .btn-advanced-analytics:hover {
            transform: translateY(-1px);
            box-shadow: 0 10px 22px rgba(15, 23, 42, 0.45);
        }

        /* --- 4A. HYPER DARK ANALYTICS PANEL --- */
        .hyper-analytics-panel {
            margin-bottom: 30px;
            padding: 22px;
            border-radius: 14px;
            border: 1px solid rgba(45, 212, 191, 0.22);
            background:
                radial-gradient(1200px 380px at 8% -22%, rgba(34, 211, 238, 0.28), transparent 45%),
                radial-gradient(1000px 420px at 108% 120%, rgba(16, 185, 129, 0.22), transparent 45%),
                linear-gradient(160deg, #080c16 0%, #0a111c 48%, #07140f 100%);
            box-shadow: 0 16px 38px rgba(2, 6, 23, 0.55), inset 0 1px 0 rgba(255,255,255,0.04);
            color: #e6edf7;
        }

        .hyper-analytics-title {
            margin: 0 0 16px 0;
            color: #ecfeff;
            border-bottom: 1px solid rgba(125, 211, 252, 0.22);
            padding-bottom: 12px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            text-shadow: 0 0 12px rgba(56, 189, 248, 0.25);
        }

        .hyper-close-btn {
            background: rgba(148, 163, 184, 0.14);
            border: 1px solid rgba(148, 163, 184, 0.3);
            color: #e2e8f0;
            width: 34px;
            height: 34px;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        .hyper-close-btn:hover {
            background: rgba(244, 63, 94, 0.2);
            border-color: rgba(244, 63, 94, 0.45);
            color: #ffdce4;
            box-shadow: 0 0 14px rgba(244, 63, 94, 0.35);
        }

        .hyper-analytics-card {
            background: linear-gradient(165deg, #070b12 0%, #0a121e 55%, #07100c 100%);
            border: 1px solid rgba(125, 211, 252, 0.16);
            padding: 18px;
            border-radius: 12px;
            margin-bottom: 18px;
            box-shadow: 0 10px 24px rgba(2, 6, 23, 0.5);
        }

        .hyper-analytics-card h4 {
            margin: 0 0 16px 0;
            color: #e2f8ff;
            font-size: 15px;
            border-bottom: 1px solid rgba(148, 163, 184, 0.24);
            padding-bottom: 10px;
            letter-spacing: 0.2px;
        }

        .hyper-analytics-card.part-card { border-left: 4px solid #22d3ee; }
        .hyper-analytics-card.employee-card { border-left: 4px solid #34d399; }
        .hyper-analytics-card.eff-card { border-left: 4px solid #f472b6; }
        .hyper-analytics-card.machine-card { border-left: 4px solid #60a5fa; }
        .hyper-analytics-card.tool-card { border-left: 4px solid #f59e0b; }

        .analytics-empty {
            color: #95a4bc;
            text-align: center;
            padding: 18px;
            margin: 0;
        }

        .hyper-row {
            margin-bottom: 14px;
        }
        .hyper-row-head {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 6px;
            gap: 10px;
        }
        .hyper-label {
            font-weight: 600;
            color: #dbeafe;
            font-size: 12px;
        }
        .hyper-value {
            color: var(--accent, #5eead4);
            font-weight: 700;
            font-size: 12px;
            text-shadow: 0 0 10px var(--glow, rgba(94, 234, 212, 0.45));
        }
        .hyper-bar-track {
            width: 100%;
            height: 24px;
            background: rgba(148, 163, 184, 0.15);
            border: 1px solid rgba(148, 163, 184, 0.2);
            border-radius: 8px;
            overflow: hidden;
            position: relative;
        }
        .hyper-bar-fill {
            width: 0;
            height: 100%;
            background: linear-gradient(90deg, rgba(15, 23, 42, 0.88) 0%, var(--accent, #34d399) 100%);
            box-shadow: 0 0 16px var(--glow, rgba(52, 211, 153, 0.45));
            border-radius: 8px;
            transition: width 0.45s ease;
            display: flex;
            align-items: center;
            justify-content: flex-end;
            padding-right: 8px;
        }
        .hyper-bar-fill span {
            color: #ecfeff;
            font-weight: 700;
            font-size: 11px;
        }

        .employee-neon-card,
        .eff-neon-card {
            background: linear-gradient(165deg, #050910 0%, #0a111c 55%, #060f0b 100%);
            border: 1px solid rgba(148, 163, 184, 0.2);
            border-radius: 10px;
            padding: 14px;
            box-shadow: inset 0 1px 0 rgba(255,255,255,0.03);
            margin-bottom: 14px;
        }
        .employee-neon-name,
        .eff-name {
            font-weight: 700;
            color: #e2f8ff;
            margin-bottom: 10px;
            font-size: 13px;
        }
        .employee-neon-metric {
            margin-bottom: 10px;
        }
        .employee-neon-metric-head {
            display: flex;
            justify-content: space-between;
            align-items: center;
            font-size: 11px;
            margin-bottom: 4px;
            color: #9fb0c9;
        }
        .employee-neon-divider {
            border-bottom: 1px solid rgba(148, 163, 184, 0.2);
            margin-top: 10px;
        }

        .eff-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 12px;
        }
        .eff-foot-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 10px;
            margin-top: 10px;
        }
        .kpi-chip {
            text-align: center;
            padding: 8px;
            border-radius: 8px;
            background: rgba(148, 163, 184, 0.12);
            border: 1px solid rgba(148, 163, 184, 0.2);
            font-size: 11px;
            color: #c6d4e8;
        }
        .kpi-chip strong {
            display: block;
            font-size: 13px;
            color: #ecfeff;
            margin-bottom: 2px;
        }

        .neon-vertical-wrap {
            display: flex;
            align-items: flex-end;
            justify-content: space-around;
            gap: 10px;
            padding: 18px 8px;
            height: 300px;
            border-bottom: 1px solid rgba(148, 163, 184, 0.24);
            margin-bottom: 15px;
            border-radius: 10px;
            background: linear-gradient(180deg, rgba(15, 23, 42, 0.62) 0%, rgba(2, 6, 23, 0.62) 100%);
        }
        .neon-vertical-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            flex: 1;
            min-width: 54px;
        }
        .neon-vertical-shell {
            width: 46px;
            height: 250px;
            display: flex;
            align-items: flex-end;
            justify-content: center;
            position: relative;
            background: rgba(148, 163, 184, 0.08);
            border: 1px solid rgba(148, 163, 184, 0.15);
            border-radius: 8px;
            overflow: hidden;
        }
        .neon-vertical-bar {
            width: 100%;
            background: linear-gradient(180deg, rgba(15, 23, 42, 0.85) 0%, var(--accent, #22d3ee) 100%);
            box-shadow: 0 -2px 18px var(--glow, rgba(34, 211, 238, 0.45));
            border-radius: 6px 6px 0 0;
        }
        .neon-vertical-top {
            font-weight: 700;
            font-size: 11px;
            margin-bottom: 6px;
            color: var(--accent, #5eead4);
            text-shadow: 0 0 12px var(--glow, rgba(94, 234, 212, 0.45));
        }
        .neon-vertical-label {
            font-size: 10px;
            color: #c8d7eb;
            margin-top: 8px;
            text-align: center;
            word-break: break-word;
            max-width: 58px;
        }
        .neon-tag {
            font-size: 9px;
            padding: 3px 6px;
            border-radius: 999px;
            margin-top: 4px;
            color: #fff;
            font-weight: 700;
            letter-spacing: 0.2px;
        }
        .neon-sub {
            font-size: 9px;
            color: #95a4bc;
            margin-top: 3px;
            text-align: center;
            max-width: 64px;
        }

        .hyper-summary-grid {
            display: grid;
            grid-template-columns: repeat(3, minmax(0, 1fr));
            gap: 10px;
        }
        .hyper-summary-card {
            padding: 10px;
            border-radius: 9px;
            border: 1px solid rgba(148, 163, 184, 0.2);
            background: rgba(15, 23, 42, 0.55);
            text-align: center;
            color: #c9d7eb;
            font-size: 10px;
            text-transform: uppercase;
            letter-spacing: 0.4px;
        }
        .hyper-summary-card strong {
            display: block;
            font-size: 16px;
            margin: 3px 0;
            text-transform: none;
            letter-spacing: normal;
        }
        .hyper-summary-card.warn strong { color: #fbbf24; }
        .hyper-summary-card.good strong { color: #34d399; }
        .hyper-summary-card.crit strong { color: #fb7185; }

        @media (max-width: 900px) {
            .hyper-summary-grid {
                grid-template-columns: 1fr;
            }
            .neon-vertical-wrap {
                overflow-x: auto;
                justify-content: flex-start;
            }
        }

        /* --- 4B. LIGHT ANALYTICS OVERRIDES (VIEW PAGE) --- */
        .btn-analytics {
            color: #ffffff;
            background: linear-gradient(130deg, #0ea5e9 0%, #0284c7 50%, #0369a1 100%);
            box-shadow: 0 8px 18px rgba(14, 165, 233, 0.28);
        }
        .btn-analytics:hover {
            box-shadow: 0 10px 22px rgba(3, 105, 161, 0.3);
        }
        .btn-advanced-analytics {
            color: #ffffff;
            background: linear-gradient(130deg, #0f172a 0%, #1e293b 52%, #334155 100%);
            box-shadow: 0 8px 18px rgba(51, 65, 85, 0.22);
        }

        .hyper-analytics-panel {
            border: 1px solid #dbe8f5;
            background:
                radial-gradient(800px 260px at 8% -25%, rgba(14, 165, 233, 0.14), transparent 48%),
                radial-gradient(900px 300px at 115% 130%, rgba(16, 185, 129, 0.12), transparent 50%),
                #ffffff;
            box-shadow: 0 10px 26px rgba(15, 23, 42, 0.08);
            color: #1f2937;
        }

        .hyper-analytics-title {
            color: #0f172a;
            border-bottom: 1px solid #dbe8f5;
            text-shadow: none;
        }

        .hyper-close-btn {
            background: #f8fafc;
            border: 1px solid #cbd5e1;
            color: #475569;
        }
        .hyper-close-btn:hover {
            background: #fff1f2;
            border-color: #fda4af;
            color: #be123c;
            box-shadow: 0 0 0 3px rgba(244, 63, 94, 0.14);
        }

        .hyper-analytics-card {
            background: #ffffff;
            border: 1px solid #dbe8f5;
            box-shadow: 0 6px 16px rgba(15, 23, 42, 0.05);
        }

        .hyper-analytics-card h4 {
            color: #0f172a;
            border-bottom: 1px solid #e2e8f0;
        }

        .analytics-empty {
            color: #64748b;
        }

        .hyper-label {
            color: #1e293b;
        }
        .hyper-value {
            text-shadow: none;
        }
        .hyper-bar-track {
            background: #f1f5f9;
            border: 1px solid #dbe8f5;
        }
        .hyper-bar-fill {
            background: linear-gradient(90deg, #dbeafe 0%, var(--accent, #0ea5e9) 100%);
            box-shadow: none;
        }
        .hyper-bar-fill span {
            color: #0f172a;
        }

        .employee-neon-card,
        .eff-neon-card {
            background: #ffffff;
            border: 1px solid #dbe8f5;
            box-shadow: 0 4px 10px rgba(15, 23, 42, 0.04);
        }
        .employee-neon-name,
        .eff-name {
            color: #0f172a;
        }
        .employee-neon-metric-head {
            color: #475569;
        }
        .employee-neon-divider {
            border-bottom: 1px solid #e2e8f0;
        }

        .kpi-chip {
            background: #f8fafc;
            border: 1px solid #dbe8f5;
            color: #475569;
        }
        .kpi-chip strong {
            color: #0f172a;
        }

        .neon-vertical-wrap {
            align-items: flex-end;
            justify-content: flex-start;
            gap: 14px;
            padding: 34px 14px 20px;
            min-height: 380px;
            height: auto;
            overflow-x: auto;
            overflow-y: visible;
            border-bottom: 1px solid #dbe8f5;
            background: linear-gradient(180deg, #f8fbff 0%, #f1f7ff 100%);
        }
        .neon-vertical-item {
            min-width: 90px;
            flex: 1 0 90px;
        }
        .neon-vertical-shell {
            width: 56px;
            height: 210px;
            background: #ffffff;
            border: 1px solid #dbe8f5;
        }
        .neon-vertical-bar {
            background: linear-gradient(180deg, #bfdbfe 0%, var(--accent, #0ea5e9) 100%);
            box-shadow: none;
        }
        .neon-vertical-top {
            font-size: 18px;
            line-height: 1.05;
            margin-bottom: 10px;
            white-space: nowrap;
            text-shadow: none;
        }
        .neon-vertical-label {
            margin-top: 10px;
            max-width: 100px;
            min-height: 30px;
            font-size: 12px;
            font-weight: 600;
            line-height: 1.25;
            color: #334155;
        }
        .neon-tag {
            margin-top: 6px;
            padding: 4px 8px;
            font-size: 10px;
        }
        .neon-sub {
            margin-top: 5px;
            font-size: 11px;
            white-space: nowrap;
            color: #64748b;
        }

        .hyper-summary-card {
            background: #ffffff;
            border: 1px solid #dbe8f5;
            color: #475569;
        }
        .hyper-summary-card.warn strong { color: #d97706; }
        .hyper-summary-card.good strong { color: #059669; }
        .hyper-summary-card.crit strong { color: #e11d48; }

        .analytics-chart-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 12px;
            margin-bottom: 14px;
        }
        .analytics-chart-card {
            background: #ffffff;
            border: 1px solid #dbe8f5;
            border-radius: 10px;
            padding: 12px;
            box-shadow: 0 3px 10px rgba(15, 23, 42, 0.04);
        }
        .analytics-chart-card h5 {
            margin: 0 0 10px 0;
            font-size: 12px;
            color: #0f172a;
            letter-spacing: 0.2px;
        }
        .analytics-chart-card canvas {
            width: 100% !important;
            height: 220px !important;
        }

        /* --- 5. FILTER BAR --- */
.filter-bar { 
    background: #f8f9fa; 
    padding: 12px; 
    border-radius: 8px; 
    margin-bottom: 14px; 
    border: 1px solid #e9ecef; 
    display: flex; 
    flex-wrap: wrap; 
    gap: 10px; 
    align-items: flex-end;
    
    /* NEW STICKY STYLES */
    position: sticky;
    top: 0; /* Sticks to the very top */
    z-index: 100; /* High priority */
    box-shadow: 0 4px 6px -1px rgba(0,0,0,0.1); /* Adds a shadow when scrolling */
}

        .filter-group { flex: 0 0 130px; min-width: 130px; display: flex; flex-direction: column; gap: 4px; }
        .filter-group label { font-size: 10px; font-weight: 700; color: #013B3B; text-transform: uppercase; }
        .filter-input { width: 100%; padding: 8px 6px; border: 1px solid #ced4da; border-radius: 5px; font-size: 12px; box-sizing: border-box; height: 36px; }
        
        .filter-actions { display: flex; flex-wrap: wrap; gap: 8px; margin-left: auto; justify-content: flex-end; align-items: center; }

        /* --- 6. TABLE --- */
/* 1. Container - CHANGED overflow-x to auto to allow scrolling */
.table-responsive { 
    overflow: auto; /* Allows both horizontal and vertical scroll */
    border: 1px solid #eee; 
    border-radius: 6px; 
    background: white; 
    max-height: 70vh; 
    width: 100%; 
    min-height: 200px; 
    display: block !important; 
    visibility: visible !important; 
}

/* 2. Table Setup */
.report-table { 
    width: 100%; 
    border-collapse: separate; /* Required for sticky to work correctly in some browsers */
    border-spacing: 0; /* Removes gaps caused by 'separate' */
    font-size: 11px; 
    /* removed table-layout: fixed; to allow table to expand for scrolling */
    white-space: nowrap; /* Prevents wrapping if you want a wide scrolling table */
}

/* 3. Header Styling */
.report-table th { 
    background: #2c3e50; 
    color: white; 
    padding: 14px 6px; 
    text-align: center; 
    border-right: 1px solid #34495e; 
    border-bottom: 1px solid #34495e; /* Added bottom border for header definition */
    font-weight: 600;
    
    /* STICKY TOP HEADER */
    position: sticky;
    top: 0;
    z-index: 10; /* Stays above normal content */
}

/* 4. Body Styling */
.report-table td { 
    padding: 8px 6px; 
    border-bottom: 1px solid #eee; 
    border-right: 1px solid #eee; 
    color: #444; 
    text-align: center; 
    background: #fff; /* Explicit background needed for sticky logic */
}

/* 5. STICKY FIRST COLUMN (Left) */
.report-table th:first-child,
.report-table td:first-child {
    position: sticky;
    left: 0;
    z-index: 11; /* Higher than normal header, lower than corner */
    border-right: 2px solid #ddd; /* Stronger border to clearly separate sticky col */
}

/* 6. THE "CORNER" (Top Left) - Needs highest priority */
.report-table th:first-child {
    z-index: 20; /* Highest layer: stays on top of both scrolls */
}

/* 7. DYNAMIC HOVER FIX */
.report-table tr:hover td {
    background: #f1f1f1;
}

/* Specific fix: When row is hovered, force sticky cell to match the hover color */
.report-table tr:hover td:first-child {
    background-color: #f1f1f1; 
}

        .row-del-btn { color: #e74c3c; background: none; border: none; font-size: 16px; cursor: pointer; padding: 5px; transition: 0.2s; }
        .row-del-btn:hover { transform: scale(1.2); color: #c0392b; }
        .row-del-btn.disabled { color: #ccc; cursor: not-allowed; opacity: 0.6; }

        /* --- 7. MODALS --- */
        .modal { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); justify-content: center; align-items: center; z-index: 2000; }
        .modal-box { background: white; padding: 30px; border-radius: 8px; width: 400px; max-width: 90%; text-align: center; box-shadow: 0 10px 30px rgba(0,0,0,0.2); }
        .modal-input { width: 100%; padding: 12px; margin: 15px 0; border: 2px solid #ddd; border-radius: 5px; box-sizing: border-box; }
        .modal-btns { display: flex; justify-content: center; gap: 15px; margin-top: 20px; }

        /* --- PRINT --- */
        @media print {
            @page { size: landscape; margin: 6mm; }

            html, body {
                width: 100% !important;
                height: auto !important;
                overflow: visible !important;
                -webkit-print-color-adjust: exact;
                print-color-adjust: exact;
            }

            .sidebar, .filter-bar, .admin-controls, .no-print, .btn, .menu-toggle-btn {
                display: none !important;
            }

            .main-content {
                margin: 0 !important;
                padding: 0 !important;
                width: 100% !important;
                min-height: auto !important;
            }

            .container {
                box-shadow: none !important;
                border: none !important;
                border-radius: 0 !important;
                padding: 0 !important;
                width: 100% !important;
            }

            .table-responsive {
                overflow: visible !important;
                max-height: none !important;
                border: none !important;
                border-radius: 0 !important;
            }

            .report-table {
                width: 100% !important;
                table-layout: fixed !important;
                white-space: normal !important;
                border-collapse: collapse !important;
                font-size: 8px !important;
                border: 1px solid #000 !important;
            }

            .report-table thead {
                display: table-header-group !important;
            }

            .report-table tr {
                page-break-inside: avoid !important;
            }

            .report-table th,
            .report-table td {
                border: 1px solid #000 !important;
                padding: 3px 2px !important;
                white-space: normal !important;
                word-break: break-word !important;
                line-height: 1.2 !important;
                color: #000 !important;
                background: #fff !important;
                position: static !important;
                left: auto !important;
                top: auto !important;
            }

            .header-bar {
                margin-bottom: 6px !important;
                padding-bottom: 6px !important;
                border-bottom: 1px solid #000 !important;
            }

            .header-title h2 {
                font-size: 15px !important;
                border-left: none !important;
                padding-left: 0 !important;
            }

            .header-meta {
                font-size: 9px !important;
                padding-left: 0 !important;
            }

            #totalDurationSummary {
                display: block !important;
                margin-top: 8px !important;
                padding: 8px !important;
                border: 1px solid #000 !important;
                background: #fff !important;
            }
        }
    </style>
</head>
<body>

    <div id="overlay" class="overlay" onclick="toggleSidebar()"></div>

    <script src="sidebar.js"></script>

    <div class="main-content">
        <div class="container">
            
            <div class="header-bar">
                <div class="header-left">
                    <button class="menu-toggle-btn" onclick="toggleSidebar()">
                        <i class="fas fa-bars"></i>
                    </button>
                    
                    <div class="header-title">
                        <h2>Production Report</h2>
                        <div class="header-meta">
                            Operator: <span id="dispOperator" style="font-weight:bold; color:#2980b9;">ALL</span> |
                            Month: <span id="dispMonth" style="font-weight:bold;">ALL</span> |
                            Total Duration: <span id="dispTotalDuration" style="font-weight:bold; color:#2980b9;">0.00 hrs</span>
                        </div>
                    </div>
                </div>
            </div>

            <div class="filter-bar no-print">
                <div class="filter-group"><label>Operator</label><select id="operatorFilter" class="filter-input" onchange="filterData()"><option value="all">-- Show All --</option></select></div>
                <div class="filter-group"><label>Month</label><select id="monthFilter" class="filter-input" onchange="filterData()"><option value="all">-- All Months --</option></select></div>
                <div class="filter-group"><label>From</label><input type="date" id="fromDateFilter" class="filter-input" onchange="filterData()"></div>
                <div class="filter-group"><label>To</label><input type="date" id="toDateFilter" class="filter-input" onchange="filterData()"></div>
                <div class="filter-group"><label>MC No</label><select id="mcNoFilter" class="filter-input" onchange="onDynamicFilterChange('mcNoFilter')"><option value="all">-- All MC No --</option></select></div>
                <div class="filter-group"><label>Tool No</label><select id="toolNoFilter" class="filter-input" onchange="onDynamicFilterChange('toolNoFilter')"><option value="all">-- All Tool No --</option></select></div>
                <div class="filter-group"><label>Operation</label><select id="operationFilter" class="filter-input" onchange="onDynamicFilterChange('operationFilter')"><option value="all">-- All Operations --</option></select></div>
                <div class="filter-group"><label>Category</label><select id="categoryFilter" class="filter-input" onchange="onDynamicFilterChange('categoryFilter')"><option value="all">-- All Categories --</option></select></div>
                <div class="filter-group"><label>Part Name</label><select id="partNameFilter" class="filter-input" onchange="onDynamicFilterChange('partNameFilter')"><option value="all">-- All Part Names --</option></select></div>
                
                <div class="filter-actions">
                     <div id="adminControls" class="admin-controls">
                        <button class="btn btn-config" onclick="openSettings()">
                            <i class="fas fa-clock"></i> Limits
                        </button>
                        <button class="btn btn-danger" onclick="confirmDeleteAll()">
                            <i class="fas fa-trash-alt"></i> Delete All
                        </button>
                     </div>
                     <button onclick="clearFilters()" class="btn btn-clear"><i class="fas fa-eraser"></i> Clear</button>
                     <button onclick="printReportView()" class="btn btn-print"><i class="fas fa-print"></i> Print</button>
                     <button onclick="toggleAnalytics()" class="btn btn-analytics"><i class="fas fa-chart-bar"></i> Quick Analytics</button>
                     <button onclick="window.location.href='analytics_advanced.php'" class="btn btn-advanced-analytics"><i class="fas fa-layer-group"></i> Advanced</button>
                </div>
            </div>

            <div id="analyticsSection" class="no-print hyper-analytics-panel" style="display:none;">
                <h3 class="hyper-analytics-title">
                    <span><i class="fas fa-chart-bar"></i> Production Analytics</span>
                    <button onclick="toggleAnalytics()" class="hyper-close-btn" aria-label="Close analytics">
                        <i class="fas fa-times"></i>
                    </button>
                </h3>

                <div class="hyper-analytics-card part-card">
                    <h4><i class="fas fa-cube"></i> Part-Wise Production</h4>
                    <div id="partProducedContainer">
                        <p class="analytics-empty">Loading data...</p>
                    </div>
                </div>

                <div class="hyper-analytics-card employee-card">
                    <h4><i class="fas fa-user"></i> Employee Production Data</h4>
                    <div id="employeeProductionTable">
                        <p class="analytics-empty">Loading data...</p>
                    </div>
                </div>

                <div class="hyper-analytics-card eff-card">
                    <h4><i class="fas fa-percent"></i> Efficiency & Quality Metrics</h4>
                    <div id="efficiencyNCContainer">
                        <p class="analytics-empty">Loading data...</p>
                    </div>
                </div>

                <div class="hyper-analytics-card machine-card">
                    <h4><i class="fas fa-cogs"></i> Machine Hours Analysis</h4>
                    <div id="machineHoursContainer">
                        <p class="analytics-empty">Loading data...</p>
                    </div>
                </div>

                <div class="hyper-analytics-card tool-card">
                    <h4><i class="fas fa-tools"></i> Tool Stroke Analysis</h4>
                    <div id="toolStrokeContainer">
                        <p class="analytics-empty">Loading data...</p>
                    </div>
                </div>
            </div>

            <div class="table-responsive">
                <table class="report-table">
                    <thead id="logTableHead"></thead>
                    <tbody id="logTableBody"><tr><td colspan="10" style="text-align:center; padding:30px;">Loading...</td></tr></tbody>
                </table>
            </div>

            <!-- Total Duration Summary Below Table -->
            <div id="totalDurationSummary" style="display:none; margin-top:20px; padding:15px; background:#f8f9fa; border-radius:8px; border:1px solid #dee2e6; text-align:center;">
                <h4 style="margin:0; color:#013B3B; font-size:16px;">Total Duration Summary</h4>
                <p style="margin:5px 0 0 0; font-size:18px; font-weight:bold; color:#2980b9;" id="summaryTotalDuration">0.00 hrs</p>
            </div>

        </div>
    </div>

    <div id="securityModal" class="modal">
        <div class="modal-box">
            <h3 style="color:#e74c3c; margin-top:0;">Security Check</h3>
            <p>Delete <b>ALL DATA</b>? This cannot be undone.</p>
            <input type="password" id="adminPassInput" class="modal-input" placeholder="Enter Admin Password">
            <div class="modal-btns">
                <button onclick="closeModal('securityModal')" class="btn btn-clear">Cancel</button>
                <button onclick="executeBulkDelete()" class="btn btn-danger">CONFIRM DELETE</button>
            </div>
        </div>
    </div>

    <div id="settingsModal" class="modal">
        <div class="modal-box">
            <h3 style="margin-top:0;">Time Limit Settings</h3>
            <p>Hours allowed for user deletion:</p>
            <input type="number" id="limitInput" class="modal-input" placeholder="Hours (e.g. 24)">
            <div class="modal-btns">
                <button onclick="closeModal('settingsModal')" class="btn btn-clear">Cancel</button>
                <button onclick="saveSettings()" class="btn btn-config">Save</button>
            </div>
        </div>
    </div>

    <script>
        let logsCache = [];       
        let fieldConfig = [];     
        let currentUserRole = ''; 
        let deleteTimeLimit = 24; 
        let currentFilteredLogs = []; // Store currently filtered data for analytics
        let derivedFieldPresence = { efficiency: false, ncQty: false, ncPercent: false };

        const FILTER_IDS_ORDER = ['mcNoFilter', 'toolNoFilter', 'operationFilter', 'categoryFilter', 'partNameFilter'];
        const dropdownFilterMeta = {
            mcNoFilter: {
                defaultText: '-- All MC No --',
                aliases: ['mcno', 'mc_no', 'm/c no', 'machineno', 'machine_no', 'machine'],
                field: null,
                parentFilterId: null
            },
            toolNoFilter: {
                defaultText: '-- All Tool No --',
                aliases: ['toolno', 'tool_no', 'tool number', 'tool'],
                field: null,
                parentFilterId: null
            },
            operationFilter: {
                defaultText: '-- All Operations --',
                aliases: ['operation', 'operation_no', 'operationno', 'process'],
                field: null,
                parentFilterId: null
            },
            categoryFilter: {
                defaultText: '-- All Categories --',
                aliases: ['category', 'category_name'],
                field: null,
                parentFilterId: null
            },
            partNameFilter: {
                defaultText: '-- All Part Names --',
                aliases: ['partname', 'part_name', 'component', 'part'],
                field: null,
                parentFilterId: null
            }
        };

        const metricSourceAliases = {
            norms: ['norms_total_duration', 'norm_total_duration', 'norms', 'norm'],
            produced: ['produced_qty', 'production_qty', 'producedquantity', 'qty_produced', 'produced'],
            accepted: ['accepted_qty', 'accept_qty', 'acceptedquantity', 'accepted']
        };

        const derivedFieldAliases = {
            efficiency: ['production_efficiency', 'production_efficiency_percent', 'efficiency', 'efficiency_percent'],
            ncQty: ['nc_qty', 'ncqty', 'rejected_qty', 'rejection_qty'],
            ncPercent: ['nc_percent', 'ncpercentage', 'rejection_percent']
        };

        const fieldOptionsCache = new Map();

        document.addEventListener("DOMContentLoaded", function() {
            initializePage();
        });

        // --- SIDEBAR TOGGLE ---
        function toggleSidebar() {
            const sidebar = document.querySelector('.sidebar'); 
            const overlay = document.getElementById('overlay');
            
            if(sidebar) {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
            }
        }

        // --- 1. INIT ---
        async function initializePage() {
            try {
                // Role & Settings
                const sessionRes = await fetch('api.php?action=check_session');
                const sessionData = await sessionRes.json();
                currentUserRole = String(sessionData.role || '').trim().toLowerCase(); 

                // Show admin controls for all user types except controller and operator
                if (currentUserRole && currentUserRole !== 'controller' && currentUserRole !== 'operator') {
                    document.getElementById('adminControls').style.display = 'flex';
                }

                const limRes = await fetch('api.php?action=get_delete_limit');
                const limData = await limRes.json();
                deleteTimeLimit = parseInt(limData.hours) || 24;

                // Load Data
                await loadReportFields(); 
                initializeFilterFieldMappings();
                await loadOperatorFilter(); 
                await loadAllLogs();      

            } catch (err) { console.error("Init Error:", err); }
        }

        // --- 2. DATA LOADING ---
        async function loadReportFields() {
            const res = await fetch('api.php?action=get_report_fields');
            fieldConfig = await res.json();
            derivedFieldPresence = detectDerivedFieldPresence();

            let html = `<th>Operator</th>`;
            fieldConfig.forEach(f => html += `<th>${f.field_label}</th>`);

            if (!derivedFieldPresence.efficiency) html += `<th>Production Efficiency %</th>`;
            if (!derivedFieldPresence.ncQty) html += `<th>NC Qty</th>`;
            if (!derivedFieldPresence.ncPercent) html += `<th>NC %</th>`;

            html += `<th class="no-print" style="width:60px;">Action</th>`;
            document.getElementById('logTableHead').innerHTML = `<tr>${html}</tr>`;
        }

        async function loadAllLogs() {
            const res = await fetch('api.php?action=get_logs');
            logsCache = await res.json();
            populateMonthFilter(logsCache);
            await populateDynamicDropdownFilters(logsCache);
            filterData(); 
        }

        function filterData() {
            const opVal = document.getElementById('operatorFilter').value;
            const monthVal = document.getElementById('monthFilter').value;
            const fromVal = document.getElementById('fromDateFilter').value;
            const toVal = document.getElementById('toDateFilter').value;
            const mcNoVal = document.getElementById('mcNoFilter').value.toLowerCase();
            const toolNoVal = document.getElementById('toolNoFilter').value.toLowerCase();
            const operationVal = document.getElementById('operationFilter').value.toLowerCase();
            const categoryVal = document.getElementById('categoryFilter').value.toLowerCase();
            const partNameVal = document.getElementById('partNameFilter').value.toLowerCase();

            let displayData = logsCache.filter(row => {
                let custom = {};
                try { custom = JSON.parse(row.custom_data || '{}'); } catch(e){}
                const full = { ...row, ...custom };

                let opMatch = (opVal === 'all') || (row.operator_name === opVal);
                let dateMatch = true;
                if(fromVal && row.log_date < fromVal) dateMatch = false;
                if(toVal && row.log_date > toVal) dateMatch = false;
                let rowMonth = new Date(row.log_date).getMonth() + 1;
                let monthMatch = (monthVal === 'all') || (rowMonth == monthVal);

                const mcNoMatch = (mcNoVal === 'all') || (getFilterValueFromRow(full, 'mcNoFilter') === mcNoVal);
                const toolNoMatch = (toolNoVal === 'all') || (getFilterValueFromRow(full, 'toolNoFilter') === toolNoVal);
                const operationMatch = (operationVal === 'all') || (getFilterValueFromRow(full, 'operationFilter') === operationVal);
                const categoryMatch = (categoryVal === 'all') || (getFilterValueFromRow(full, 'categoryFilter') === categoryVal);
                const partNameMatch = (partNameVal === 'all') || (getFilterValueFromRow(full, 'partNameFilter') === partNameVal);

                return opMatch && dateMatch && monthMatch && mcNoMatch && toolNoMatch && operationMatch && categoryMatch && partNameMatch;
            });

            // Store filtered data globally for analytics
            currentFilteredLogs = displayData;

            document.getElementById('dispOperator').textContent = opVal === 'all' ? "ALL" : opVal.toUpperCase();
            document.getElementById('dispMonth').textContent = monthVal === 'all' ? "ALL" : getMonthName(monthVal);

            // Calculate total duration for filtered data
            let totalDuration = 0;
            displayData.forEach(row => {
                let custom = {};
                try { custom = JSON.parse(row.custom_data || '{}'); } catch(e){}
                const full = { ...row, ...custom };
                const duration = parseFloat(full.duration || full.start_time && full.end_time ? calculateDuration(full.start_time, full.end_time) : 0);
                if (!isNaN(duration)) totalDuration += duration;
            });
            document.getElementById('dispTotalDuration').textContent = totalDuration.toFixed(2) + ' hrs';

            renderTable(displayData, totalDuration);
        }

        function renderTable(data, totalDuration) {
            const tbody = document.getElementById('logTableBody');
            const tableContainer = document.querySelector('.table-responsive');
            tbody.innerHTML = '';
            if(data.length === 0) {
                tableContainer.style.display = 'none';
                document.getElementById('totalDurationSummary').style.display = 'none';
                return;
            }

            tableContainer.style.display = 'block';

            data.forEach(row => {
                let custom = {};
                try { custom = JSON.parse(row.custom_data || '{}'); } catch(e){}
                let full = { ...row, ...custom };
                const derivedMetrics = calculateDerivedMetrics(full);

                let tr = document.createElement('tr');
                let html = `<td>${full.operator_name}</td>`;

                fieldConfig.forEach(f => {
                    let value = full[f.field_name];

                    if (fieldMatchesAliases(f, derivedFieldAliases.efficiency) && derivedMetrics.efficiency !== null) {
                        value = formatEfficiency(derivedMetrics.efficiency);  // Use color-coded formatter
                    }
                    if (fieldMatchesAliases(f, derivedFieldAliases.ncQty) && derivedMetrics.ncQty !== null) {
                        value = formatNumber(derivedMetrics.ncQty);
                    }
                    if (fieldMatchesAliases(f, derivedFieldAliases.ncPercent) && derivedMetrics.ncPercent !== null) {
                        value = formatPercent(derivedMetrics.ncPercent);
                    }

                    const cellValue = (value === null || value === undefined || value === '') ? '-' : value;
                    html += `<td>${cellValue}</td>`;
                });

                if (!derivedFieldPresence.efficiency) html += `<td>${formatEfficiency(derivedMetrics.efficiency)}</td>`;
                if (!derivedFieldPresence.ncQty) html += `<td>${formatNumber(derivedMetrics.ncQty)}</td>`;
                if (!derivedFieldPresence.ncPercent) html += `<td>${formatPercent(derivedMetrics.ncPercent)}</td>`;

                // DELETE LOGIC (limit-wise for non-admin roles)
                const rowTimestamp = full.created_at
                    || (full.log_date ? `${full.log_date}T${full.log_time || full.start_time || '00:00:00'}` : null);
                const entryTime = rowTimestamp ? new Date(rowTimestamp).getTime() : Number.NaN;
                const now = Date.now();
                const diffHours = Number.isFinite(entryTime) ? (now - entryTime) / (1000 * 60 * 60) : Number.POSITIVE_INFINITY;

                let btnClass = 'row-del-btn';
                let btnAttr = `onclick="deleteRow(${full.id})"`;
                let btnTitle = "Delete Entry";

                const isAdminRole = currentUserRole === 'administrator' || currentUserRole === 'admin';
                if (!isAdminRole) {
                    if (!Number.isFinite(diffHours)) {
                        btnClass += ' disabled';
                        btnAttr = '';
                        btnTitle = 'Delete not available (entry time missing)';
                    } else if (diffHours > deleteTimeLimit) {
                        btnClass += ' disabled';
                        btnAttr = '';
                        btnTitle = `Delete allowed only within ${deleteTimeLimit} hour(s)`;
                    }
                }

                html += `<td class="no-print" style="text-align:center;">
                            <button class="${btnClass}" ${btnAttr} title="${btnTitle}">
                                <i class="fas fa-trash-alt"></i>
                            </button>
                         </td>`;

                tr.innerHTML = html;
                tbody.appendChild(tr);
            });

            // Show total duration summary below table
            if (data.length > 0) {
                document.getElementById('totalDurationSummary').style.display = 'block';
                document.getElementById('summaryTotalDuration').textContent = totalDuration.toFixed(2) + ' hrs';
            } else {
                document.getElementById('totalDurationSummary').style.display = 'none';
            }
        }

        function printReportView() {
            const escapeHtml = (value) => String(value)
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#39;');

            const table = document.querySelector('.report-table');
            const tableHead = document.getElementById('logTableHead');
            const tableBody = document.getElementById('logTableBody');

            if (!table || !tableHead || !tableBody || tableBody.children.length === 0) {
                Swal.fire({
                    title: 'No Data',
                    text: 'No report data available to print.',
                    icon: 'warning',
                    confirmButtonColor: '#f39c12'
                });
                return;
            }

            const tableClone = table.cloneNode(true);
            tableClone.querySelectorAll('.no-print').forEach((el) => el.remove());
            tableClone.querySelectorAll('th, td').forEach((cell) => {
                cell.style.position = 'static';
                cell.style.left = 'auto';
                cell.style.top = 'auto';
                cell.style.zIndex = 'auto';
            });

            const colCount = tableClone.querySelectorAll('thead th').length || 1;
            const tableFontSize = colCount >= 15 ? '7px' : (colCount >= 12 ? '8px' : '9px');
            const cellPadding = colCount >= 15 ? '2px 1px' : '3px 2px';

            const operator = escapeHtml((document.getElementById('dispOperator')?.textContent || 'ALL').trim());
            const month = escapeHtml((document.getElementById('dispMonth')?.textContent || 'ALL').trim());
            const totalDuration = escapeHtml((document.getElementById('dispTotalDuration')?.textContent || '0.00 hrs').trim());
            const summaryDuration = escapeHtml((document.getElementById('summaryTotalDuration')?.textContent || totalDuration).trim());

            const printWindow = window.open('', '_blank', 'width=1400,height=900');
            if (!printWindow) {
                Swal.fire({
                    title: 'Popup Blocked',
                    text: 'Please allow popups to print the report.',
                    icon: 'error',
                    confirmButtonColor: '#e74c3c'
                });
                return;
            }

            const printableHtml = `
                <!DOCTYPE html>
                <html lang="en">
                <head>
                    <meta charset="UTF-8">
                    <meta name="viewport" content="width=device-width, initial-scale=1.0">
                    <title>Production Report Print</title>
                    <style>
                        @page { size: landscape; margin: 6mm; }
                        * { box-sizing: border-box; }
                        body { margin: 0; font-family: Arial, sans-serif; color: #111; }
                        .print-wrap { width: 100%; }
                        .header { border: 1px solid #000; padding: 6px 8px; margin-bottom: 6px; }
                        .header h2 { margin: 0 0 4px 0; font-size: 16px; }
                        .meta { font-size: 10px; }
                        .table-wrap { width: 100%; }
                        table { width: 100%; border-collapse: collapse; table-layout: fixed; font-size: ${tableFontSize}; }
                        thead { display: table-header-group; }
                        tr { page-break-inside: avoid; }
                        th, td {
                            border: 1px solid #000;
                            padding: ${cellPadding};
                            text-align: center;
                            vertical-align: middle;
                            white-space: normal;
                            word-break: break-word;
                            line-height: 1.2;
                            color: #000;
                            background: #fff;
                        }
                        th { font-weight: 700; }
                        .summary {
                            margin-top: 8px;
                            border: 1px solid #000;
                            padding: 8px;
                            text-align: center;
                        }
                        .summary h4 { margin: 0; font-size: 13px; }
                        .summary p { margin: 4px 0 0 0; font-size: 14px; font-weight: 700; }
                    </style>
                </head>
                <body>
                    <div class="print-wrap">
                        <div class="header">
                            <h2>Production Report</h2>
                            <div class="meta">
                                Operator: <b>${operator}</b> |
                                Month: <b>${month}</b> |
                                Total Duration: <b>${totalDuration}</b>
                            </div>
                        </div>
                        <div class="table-wrap">
                            ${tableClone.outerHTML}
                        </div>
                        <div class="summary">
                            <h4>Total Duration Summary</h4>
                            <p>${summaryDuration}</p>
                        </div>
                    </div>
                </body>
                </html>
            `;

            printWindow.document.open();
            printWindow.document.write(printableHtml);
            printWindow.document.close();

            printWindow.focus();
            setTimeout(() => {
                printWindow.print();
                printWindow.close();
            }, 250);
        }

        // --- 3. ACTIONS ---
        async function deleteRow(id) {
            try {
                const result = await Swal.fire({
                    title: 'Delete Entry?',
                    text: 'This action cannot be undone.',
                    icon: 'warning',
                    showCancelButton: true,
                    confirmButtonColor: '#e74c3c',
                    cancelButtonColor: '#95a5a6',
                    confirmButtonText: 'Yes, Delete',
                    cancelButtonText: 'Cancel'
                });
                
                if (!result.isConfirmed) return;
                
                Swal.fire({
                    title: 'Deleting...',
                    text: 'Please wait...',
                    icon: 'info',
                    allowOutsideClick: false,
                    didOpen: async () => {
                        Swal.showLoading();
                        
                        try {
                            const fd = new FormData();
                            fd.append('id', id);
                            
                            const res = await fetch('api.php?action=delete_log_entry', {
                                method: 'POST',
                                body: fd
                            });
                            
                            const data = await res.json();
                            
                            if (data.success) {
                                Swal.fire({
                                    title: 'Success!',
                                    text: data.message || 'Entry deleted successfully',
                                    icon: 'success',
                                    confirmButtonColor: '#27ae60'
                                }).then(() => {
                                    loadAllLogs();
                                });
                            } else {
                                Swal.fire({
                                    title: 'Delete Failed',
                                    text: data.message || 'An error occurred while deleting',
                                    icon: 'error',
                                    confirmButtonColor: '#e74c3c'
                                });
                            }
                        } catch (error) {
                            console.error('Delete error:', error);
                            Swal.fire({
                                title: 'Error',
                                text: 'Network error: ' + error.message,
                                icon: 'error',
                                confirmButtonColor: '#e74c3c'
                            });
                        }
                    }
                });
            } catch (e) {
                console.error('Delete function error:', e);
                Swal.fire({
                    title: 'Error',
                    text: 'An unexpected error occurred',
                    icon: 'error',
                    confirmButtonColor: '#e74c3c'
                });
            }
        }

        function confirmDeleteAll() { document.getElementById('securityModal').style.display = 'flex'; }

        async function executeBulkDelete() {
            const pass = document.getElementById('adminPassInput').value;
            if(!pass) {
                Swal.fire({icon:'warning', title:'Required', text:'Password Required', confirmButtonColor:'#ff9800'});
                return;
            }
            const fd = new FormData(); fd.append('password', pass);
            try {
                const res = await fetch('api.php?action=delete_all_data', { method: 'POST', body: fd });
                const data = await res.json();
                if(data.success) { 
                    Swal.fire({icon:'success', title:'Success', text:'Deleted Successfully', confirmButtonColor:'#28a745'});
                    closeModal('securityModal'); 
                    loadAllLogs(); 
                } else { 
                    Swal.fire({icon:'error', title:'Error', text:data.error, confirmButtonColor:'#dc3545'});
                }
            } catch(e) { console.error(e); }
        }

        function openSettings() {
            document.getElementById('limitInput').value = deleteTimeLimit;
            document.getElementById('settingsModal').style.display = 'flex';
        }

        async function saveSettings() {
            const hours = document.getElementById('limitInput').value;
            const fd = new FormData(); fd.append('hours', hours);
            await fetch('api.php?action=update_delete_limit', { method: 'POST', body: fd });
            Swal.fire({icon:'success', title:'Updated', text:'Settings updated successfully', confirmButtonColor:'#013B3B'}).then(() => {
                closeModal('settingsModal'); 
                location.reload();
            });
        }

        function closeModal(id) { document.getElementById(id).style.display = 'none'; }
        
        async function clearFilters() {
            document.getElementById('operatorFilter').value = 'all';
            document.getElementById('monthFilter').value = 'all';
            document.getElementById('fromDateFilter').value = '';
            document.getElementById('toDateFilter').value = '';
            document.getElementById('mcNoFilter').value = 'all';
            document.getElementById('toolNoFilter').value = 'all';
            document.getElementById('operationFilter').value = 'all';
            document.getElementById('categoryFilter').value = 'all';
            document.getElementById('partNameFilter').value = 'all';
            await reloadAdminFilterDropdowns();
            filterData();
        }

        async function loadOperatorFilter() {
            const sel = document.getElementById('operatorFilter');
            const res = await fetch('api.php?action=get_employees');
            const data = await res.json();
            data.forEach(e => sel.innerHTML += `<option value="${e.name}">${e.name}</option>`);
        }

        function populateMonthFilter(data) {
            const sel = document.getElementById('monthFilter');
            const prev = sel.value || 'all';
            sel.innerHTML = '<option value="all">-- All Months --</option>';
            for (let m = 1; m <= 12; m++) {
                const opt = document.createElement('option');
                opt.value = String(m);
                opt.textContent = getMonthName(m);
                sel.appendChild(opt);
            }
            sel.value = Array.from(sel.options).some(o => o.value === prev) ? prev : 'all';
        }

        async function populateDynamicDropdownFilters(data) {
            fieldOptionsCache.clear();
            initializeFilterFieldMappings();
            await reloadAdminFilterDropdowns();

            const fallbackValues = {
                mcNoFilter: [],
                toolNoFilter: [],
                operationFilter: [],
                categoryFilter: [],
                partNameFilter: []
            };

            data.forEach(row => {
                let custom = {};
                try { custom = JSON.parse(row.custom_data || '{}'); } catch(e){}
                const full = { ...row, ...custom };

                FILTER_IDS_ORDER.forEach(filterId => {
                    fallbackValues[filterId].push(getFilterRawValueFromRow(full, filterId));
                });
            });

            FILTER_IDS_ORDER.forEach(filterId => {
                const select = document.getElementById(filterId);
                if (!select || select.options.length > 1) return;
                setDynamicSelectOptions(filterId, fallbackValues[filterId], dropdownFilterMeta[filterId].defaultText);
            });
        }

        async function onDynamicFilterChange(filterId) {
            await refreshChildFilterDropdowns(filterId);
            filterData();
        }

        async function reloadAdminFilterDropdowns() {
            for (const filterId of FILTER_IDS_ORDER) {
                const meta = dropdownFilterMeta[filterId];
                if (meta.parentFilterId) continue;
                await loadAdminOptionsForFilter(filterId, false);
                await refreshChildFilterDropdowns(filterId);
            }
        }

        async function refreshChildFilterDropdowns(parentFilterId) {
            for (const filterId of FILTER_IDS_ORDER) {
                const meta = dropdownFilterMeta[filterId];
                if (meta.parentFilterId !== parentFilterId) continue;
                await loadAdminOptionsForFilter(filterId, true);
                await refreshChildFilterDropdowns(filterId);
            }
        }

        async function loadAdminOptionsForFilter(filterId, resetSelection = false) {
            const meta = dropdownFilterMeta[filterId];
            const select = document.getElementById(filterId);
            if (!meta || !select || !meta.field || meta.field.field_type !== 'dropdown') return false;

            let options = [];

            if (meta.parentFilterId) {
                const parentMeta = dropdownFilterMeta[meta.parentFilterId];
                const parentSelect = document.getElementById(meta.parentFilterId);
                if (!parentMeta?.field || !parentSelect) return false;

                const parentValue = String(parentSelect.value || 'all');
                if (parentValue === 'all') {
                    options = await getFieldOptions(meta.field.id);
                } else {
                    const parentOptions = await getFieldOptions(parentMeta.field.id);
                    const parentOpt = parentOptions.find(o => String(o.label || '').toLowerCase() === parentValue.toLowerCase());
                    if (parentOpt) {
                        options = await getFieldOptions(meta.field.id, parentOpt.id);
                    }
                }
            } else {
                options = await getFieldOptions(meta.field.id);
            }

            let labels = options.map(opt => String(opt.label || '').trim()).filter(Boolean);
            if (labels.length === 0 && logsCache.length > 0) {
                const fallback = [];
                logsCache.forEach(row => {
                    let custom = {};
                    try { custom = JSON.parse(row.custom_data || '{}'); } catch(e){}
                    const full = { ...row, ...custom };
                    fallback.push(getFilterRawValueFromRow(full, filterId));
                });
                labels = fallback.filter(Boolean);
            }

            setDynamicSelectOptions(filterId, labels, meta.defaultText, resetSelection);
            return labels.length > 0;
        }

        async function getFieldOptions(fieldId, parentOptId = '') {
            const cacheKey = `${fieldId}_${parentOptId || 'all'}`;
            if (fieldOptionsCache.has(cacheKey)) return fieldOptionsCache.get(cacheKey);

            let url = `api.php?action=get_options&field_id=${fieldId}`;
            if (parentOptId !== '' && parentOptId !== null) url += `&parent_opt_id=${parentOptId}`;

            try {
                const res = await fetch(url);
                const data = await res.json();
                const list = Array.isArray(data) ? data : [];
                fieldOptionsCache.set(cacheKey, list);
                return list;
            } catch (error) {
                console.error('Failed to load filter options', error);
                return [];
            }
        }

        function initializeFilterFieldMappings() {
            FILTER_IDS_ORDER.forEach(filterId => {
                const meta = dropdownFilterMeta[filterId];
                meta.field = findFieldByAliases(meta.aliases);
                meta.parentFilterId = null;
            });

            FILTER_IDS_ORDER.forEach(filterId => {
                const meta = dropdownFilterMeta[filterId];
                if (!meta.field?.parent_field_id) return;

                const parentFieldId = String(meta.field.parent_field_id);
                const parentFilterId = FILTER_IDS_ORDER.find(id => {
                    const candidate = dropdownFilterMeta[id].field;
                    return candidate && String(candidate.id) === parentFieldId;
                });

                meta.parentFilterId = parentFilterId || null;
            });
        }

        function findFieldByAliases(aliases) {
            if (!Array.isArray(fieldConfig)) return null;

            const normalizedAliases = aliases.map(normalizeKey);
            const exact = fieldConfig.find(field => {
                const keys = getFieldKeyCandidates(field);
                return normalizedAliases.some(alias => keys.includes(alias));
            });
            if (exact) return exact;

            return fieldConfig.find(field => {
                const keys = getFieldKeyCandidates(field);
                return normalizedAliases.some(alias =>
                    alias.length >= 4 &&
                    keys.some(key => key.includes(alias) || (key.length >= 4 && alias.includes(key)))
                );
            }) || null;
        }

        function getFieldKeyCandidates(field) {
            const rawName = String(field?.field_name || '');
            const rawLabel = String(field?.field_label || '');
            const noPrefixName = rawName.replace(/^f_/i, '');
            return [normalizeKey(rawName), normalizeKey(noPrefixName), normalizeKey(rawLabel)];
        }

        function getFilterValueFromRow(rowObj, filterId) {
            return getFilterRawValueFromRow(rowObj, filterId).toLowerCase();
        }

        function getFilterRawValueFromRow(rowObj, filterId) {
            const meta = dropdownFilterMeta[filterId];
            if (meta?.field?.field_name) {
                const value = rowObj?.[meta.field.field_name];
                const normalized = String(value ?? '').trim();
                if (normalized !== '') return normalized;
            }
            return getRowValueByAliasesRaw(rowObj, meta?.aliases || []);
        }

        function getMonthName(m) {
            const names = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];
            return names[m-1] || m;
        }

        function normalizeKey(value) {
            return String(value || '').toLowerCase().replace(/[^a-z0-9]/g, '');
        }

        function getRowValueByAliases(rowObj, aliases) {
            const aliasSet = new Set(aliases.map(normalizeKey));
            for (const [key, value] of Object.entries(rowObj || {})) {
                const normalizedKey = normalizeKey(key);
                const normalizedKeyNoPrefix = normalizeKey(String(key).replace(/^f_/i, ''));
                if (aliasSet.has(normalizedKey) || aliasSet.has(normalizedKeyNoPrefix)) {
                    return String(value ?? '').toLowerCase();
                }
            }
            return '';
        }

        function getRowValueByAliasesRaw(rowObj, aliases) {
            const aliasSet = new Set(aliases.map(normalizeKey));
            for (const [key, value] of Object.entries(rowObj || {})) {
                const normalizedKey = normalizeKey(key);
                const normalizedKeyNoPrefix = normalizeKey(String(key).replace(/^f_/i, ''));
                if (aliasSet.has(normalizedKey) || aliasSet.has(normalizedKeyNoPrefix)) {
                    return String(value ?? '').trim();
                }
            }
            return '';
        }

        function setDynamicSelectOptions(selectId, values, defaultText, resetSelection = false) {
            const select = document.getElementById(selectId);
            const prev = select.value || 'all';
            const normalizedMap = new Map();

            values.forEach(v => {
                const raw = String(v || '').trim();
                if (!raw) return;
                const key = raw.toLowerCase();
                if (!normalizedMap.has(key)) normalizedMap.set(key, raw);
            });

            const sortedValues = Array.from(normalizedMap.values()).sort((a, b) =>
                a.localeCompare(b, undefined, { numeric: true, sensitivity: 'base' })
            );

            select.innerHTML = '';
            const defaultOption = document.createElement('option');
            defaultOption.value = 'all';
            defaultOption.textContent = defaultText;
            select.appendChild(defaultOption);

            sortedValues.forEach(value => {
                const opt = document.createElement('option');
                opt.value = value;
                opt.textContent = value;
                select.appendChild(opt);
            });

            if (resetSelection) {
                select.value = 'all';
            } else {
                select.value = Array.from(select.options).some(o => o.value === prev) ? prev : 'all';
            }
        }

        function fieldMatchesAliases(field, aliases) {
            const keys = getFieldKeyCandidates(field);
            return aliases.map(normalizeKey).some(alias => keys.includes(alias) || keys.some(k => k.includes(alias) || alias.includes(k)));
        }

        function detectDerivedFieldPresence() {
            return {
                efficiency: fieldConfig.some(f => fieldMatchesAliases(f, derivedFieldAliases.efficiency)),
                ncQty: fieldConfig.some(f => fieldMatchesAliases(f, derivedFieldAliases.ncQty)),
                ncPercent: fieldConfig.some(f => fieldMatchesAliases(f, derivedFieldAliases.ncPercent))
            };
        }

        function parseNumeric(value) {
            const cleaned = String(value ?? '').replace(/,/g, '').trim();
            if (!cleaned) return null;
            const num = parseFloat(cleaned);
            return Number.isFinite(num) ? num : null;
        }

        function escapeMetricText(value) {
            return String(value ?? '')
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#39;');
        }

        function withAlpha(hexColor, alpha = '66') {
            return /^#[0-9a-fA-F]{6}$/.test(hexColor) ? `${hexColor}${alpha}` : 'rgba(148, 163, 184, 0.45)';
        }

        function calculateDerivedMetrics(rowObj) {
            // Get plan_qty, produced_qty, accepted_qty from the row
            const planQty = parseNumeric(rowObj['plan_qty'] || rowObj['f_plan_qty'] || 0);
            const produced = parseNumeric(rowObj['produced_qty'] || rowObj['f_produced_qty'] || 0);
            const accepted = parseNumeric(rowObj['accepted_qty'] || rowObj['f_accepted_qty'] || 0);

            let efficiency = null;
            let ncQty = null;
            let ncPercent = null;

            // === CORRECTED FORMULAS ===
            // Production Efficiency (%) = (Produced Qty / Plan Qty) × 100
            if (planQty !== null && planQty > 0 && produced !== null) {
                efficiency = (produced / planQty) * 100;
            }

            // NC Qty = Produced Qty - Accepted Qty
            if (produced !== null && accepted !== null) {
                ncQty = Math.max(produced - accepted, 0);
            }

            // NC % = (NC Qty / Produced Qty) × 100
            if (produced !== null && produced > 0 && ncQty !== null) {
                ncPercent = (ncQty / produced) * 100;
            }

            return { efficiency, ncQty, ncPercent };
        }

        function formatNumber(value) {
            if (value === null || value === undefined || Number.isNaN(value)) return '-';
            const rounded = Math.round(value * 100) / 100;
            return Number.isInteger(rounded) ? String(rounded) : rounded.toFixed(2);
        }

        function formatPercent(value) {
            if (value === null || value === undefined || Number.isNaN(value)) return '-';
            return `${formatNumber(value)}%`;
        }

        // === EFFICIENCY COLOR INDICATOR function ===
        function formatEfficiency(value) {
            if (value === null || value === undefined || Number.isNaN(value)) return '-';
            const numValue = parseFloat(value);
            const formatted = formatPercent(numValue);
            
            // Color indicators based on manufacturing standards
            let color = '#999';  // Default gray
            let bgColor = '#f0f0f0';
            
            if (numValue < 70) {
                color = '#fff';
                bgColor = '#d32f2f';  // RED - Critical
            } else if (numValue < 85) {
                color = '#fff';
                bgColor = '#ff6f00';  // ORANGE - Warning
            } else if (numValue <= 100) {
                color = '#fff';
                bgColor = '#388e3c';  // GREEN - Good
            } else {
                color = '#fff';
                bgColor = '#1976d2';  // BLUE - Over Target (Overproduction)
            }
            
            return `<span style="background:${bgColor}; color:${color}; padding:4px 8px; border-radius:4px; font-weight:bold; white-space:nowrap;">${formatted}</span>`;
        }

        function calculateDuration(startTime, endTime) {
            if (!startTime || !endTime) return 0;
            const start = new Date(`1970-01-01T${startTime}:00`);
            const end = new Date(`1970-01-01T${endTime}:00`);
            const diffMs = end - start;
            return diffMs > 0 ? diffMs / (1000 * 60 * 60) : 0; // Convert to hours
        }

        // ===== ANALYTICS FUNCTIONS =====
        let fieldAnalyticsChart = null;
        let analyticsCache = {}; // Cache for field statistics

        // Hours-Based Analytics Variables
        let hoursOperatorChart = null;
        let hoursMachineChart = null;
        let hoursTimelineChart = null;
        let efficiencyTrendChart = null;
        let qualitySplitChart = null;

        function toggleAnalytics() {
            const section = document.getElementById('analyticsSection');
            if (section.style.display === 'none') {
                section.style.display = 'block';
                loadProductionAnalytics();
            } else {
                section.style.display = 'none';
            }
        }

        function loadProductionAnalytics() {
            try {
                const filteredLogs = currentFilteredLogs && currentFilteredLogs.length > 0 ? currentFilteredLogs : logsCache;
                console.log('Analytics - Filtered Logs Count:', filteredLogs.length);
                if (filteredLogs.length > 0) {
                    console.log('First log sample:', filteredLogs[0]);
                }
                calculateProductionAnalytics(filteredLogs);
            } catch (error) {
                console.error('Error loading production analytics:', error);
                const safeError = escapeMetricText(error.message || 'Unknown error');
                document.getElementById('partProducedContainer').innerHTML = `<p class="analytics-empty">Error: ${safeError}</p>`;
                document.getElementById('employeeProductionTable').innerHTML = `<p class="analytics-empty">Error: ${safeError}</p>`;
                document.getElementById('efficiencyNCContainer').innerHTML = `<p class="analytics-empty">Error: ${safeError}</p>`;
                document.getElementById('machineHoursContainer').innerHTML = `<p class="analytics-empty">Error: ${safeError}</p>`;
                document.getElementById('toolStrokeContainer').innerHTML = `<p class="analytics-empty">Error: ${safeError}</p>`;
                destroyQuickAnalyticsCharts();
            }
        }

        function calculateProductionAnalytics(filteredLogs) {
            if (filteredLogs.length === 0) {
                document.getElementById('partProducedContainer').innerHTML = '<p class="analytics-empty">No data available</p>';
                document.getElementById('employeeProductionTable').innerHTML = '<p class="analytics-empty">No data available</p>';
                document.getElementById('efficiencyNCContainer').innerHTML = '<p class="analytics-empty">No data available</p>';
                document.getElementById('machineHoursContainer').innerHTML = '<p class="analytics-empty">No data available</p>';
                document.getElementById('toolStrokeContainer').innerHTML = '<p class="analytics-empty">No data available</p>';
                destroyQuickAnalyticsCharts();
                return;
            }

            // Data aggregation objects
            const partProduced = {};     // Sum of produced_qty by part
            const employeeData = {};     // { employee: { produced, accepted, nc, efficiency_sum, count } }
            const machineHours = {};     // { machine: total_hours }
            const toolStrokes = {};      // { tool: { total: strokes, remaining: remaining_strokes, count } }

            filteredLogs.forEach(log => {
                let logData = { ...log };
                try {
                    const custom = JSON.parse(log.custom_data || '{}');
                    logData = { ...logData, ...custom };
                } catch (e) {}

                // Extract quantities with better field detection
                const produced = parseNumeric(logData.produced_qty || logData.production_qty || logData.qty || 0);
                const accepted = parseNumeric(logData.accepted_qty || logData.accept_qty || logData.accepted_quantity || 0);
                const planned = parseNumeric(logData.plan_qty || logData.planned_qty || produced || 1);
                
                // Calculate NC as produced - accepted
                const ncQty = Math.max(0, produced - accepted);
                
                // CALCULATE Efficiency = (produced / planned) * 100
                const efficiency = produced > 0 ? (produced / planned) * 100 : 0;
                
                // CALCULATE NC% = (nc / produced) * 100
                const ncPercent = produced > 0 ? (ncQty / produced) * 100 : 0;
                
                // Better part name detection
                let partName = logData.part_no || logData.part_name || logData.part || 'Unknown';
                if (partName === 'Unknown') {
                    for (let key in logData) {
                        if (key.toLowerCase().includes('part') && logData[key] && logData[key] !== '') {
                            partName = logData[key];
                            break;
                        }
                    }
                }
                
                // Skip if no production data
                if (produced > 0) {
                    if (!partProduced[partName]) partProduced[partName] = 0;
                    partProduced[partName] += produced;
                }

                // 2. Employee-wise production breakdown with efficiency
                const employeeName = log.employee_name || log.operator_name || 'Unknown';
                if (!employeeData[employeeName]) {
                    employeeData[employeeName] = { produced: 0, accepted: 0, nc: 0, efficiency_sum: 0, count: 0 };
                }
                employeeData[employeeName].produced += produced;
                employeeData[employeeName].accepted += accepted;
                employeeData[employeeName].nc += ncQty;
                employeeData[employeeName].efficiency_sum += efficiency;
                employeeData[employeeName].count += 1;

                // 3. Machine hours tracking
                const machineName = logData.machine_no || logData.m_c_no || logData.mc_no || logData.machine || 'Unknown Machine';
                const machineHour = parseNumeric(logData.duration || logData.hours || logData.working_hours || logData.norms || 0);
                if (machineHour > 0) {
                    if (!machineHours[machineName]) machineHours[machineName] = 0;
                    machineHours[machineName] += machineHour;
                }

                // 4. Tool stroke tracking with remaining strokes
                const toolName = logData.tool_no || logData.tool || logData.tool_name || 'Unknown Tool';
                const stroke = parseNumeric(logData.stroke || logData.strokes || logData.tool_stroke || logData.current_usage || 0);
                const maxStrokes = parseNumeric(logData.max_life_strokes || logData.life_limit || logData.max_strokes || 10000);
                const remainingStrokes = Math.max(0, maxStrokes - stroke);
                
                if (stroke > 0) {
                    if (!toolStrokes[toolName]) {
                        toolStrokes[toolName] = { total: 0, remaining: 0, count: 0, maxLife: maxStrokes };
                    }
                    toolStrokes[toolName].total += stroke;
                    toolStrokes[toolName].remaining = remainingStrokes;
                    toolStrokes[toolName].count += 1;
                }
            });

            console.log('Part Produced:', partProduced);
            console.log('Employee Data:', employeeData);
            console.log('Machine Hours:', machineHours);
            console.log('Tool Strokes:', toolStrokes);

            buildPartProducedBars(partProduced);
            buildEmployeeProductionTable(employeeData);
            buildEfficiencyMetrics(employeeData);
            buildMachineHoursAnalysis(machineHours);
            buildToolStrokeAnalysis(toolStrokes);
            renderTrendAndQualityCharts(filteredLogs, employeeData);
        }

        function destroyQuickAnalyticsCharts() {
            if (efficiencyTrendChart) {
                efficiencyTrendChart.destroy();
                efficiencyTrendChart = null;
            }
            if (qualitySplitChart) {
                qualitySplitChart.destroy();
                qualitySplitChart = null;
            }
        }

        function extractLogDateLabel(log) {
            const candidates = [
                log.log_date,
                log.date,
                log.created_at,
                log.entry_date,
                log.timestamp
            ];
            for (const raw of candidates) {
                if (!raw) continue;
                const dateObj = new Date(raw);
                if (!Number.isNaN(dateObj.getTime())) {
                    return dateObj.toISOString().slice(0, 10);
                }
                const text = String(raw);
                const m = text.match(/\d{4}-\d{2}-\d{2}/);
                if (m) return m[0];
            }
            return null;
        }

        function renderTrendAndQualityCharts(filteredLogs, employeeData) {
            const trendCanvas = document.getElementById('effTrendChart');
            const pieCanvas = document.getElementById('qualityPieChart');
            if (!trendCanvas || !pieCanvas) return;

            destroyQuickAnalyticsCharts();

            const dateBuckets = {};
            filteredLogs.forEach((log) => {
                let merged = { ...log };
                try {
                    merged = { ...merged, ...(JSON.parse(log.custom_data || '{}')) };
                } catch (e) {}

                const produced = parseNumeric(merged.produced_qty || merged.production_qty || merged.qty || 0) || 0;
                const accepted = parseNumeric(merged.accepted_qty || merged.accept_qty || merged.accepted_quantity || 0) || 0;
                const planned = parseNumeric(merged.plan_qty || merged.planned_qty || produced || 1) || 1;
                const efficiency = planned > 0 ? (produced / planned) * 100 : 0;
                const dateKey = extractLogDateLabel(merged) || 'Unknown';

                if (!dateBuckets[dateKey]) {
                    dateBuckets[dateKey] = { sum: 0, count: 0 };
                }
                dateBuckets[dateKey].sum += efficiency;
                dateBuckets[dateKey].count += 1;
            });

            const trendEntries = Object.entries(dateBuckets)
                .filter(([key]) => key !== 'Unknown')
                .sort((a, b) => a[0].localeCompare(b[0]));

            const trendLabels = trendEntries.map(([date]) => date.slice(5));
            const trendValues = trendEntries.map(([, val]) => val.count > 0 ? Number((val.sum / val.count).toFixed(2)) : 0);

            efficiencyTrendChart = new Chart(trendCanvas.getContext('2d'), {
                type: 'line',
                data: {
                    labels: trendLabels.length > 0 ? trendLabels : ['No Data'],
                    datasets: [{
                        label: 'Avg Efficiency %',
                        data: trendValues.length > 0 ? trendValues : [0],
                        borderColor: '#0ea5e9',
                        backgroundColor: 'rgba(14,165,233,0.18)',
                        borderWidth: 2.5,
                        tension: 0.3,
                        pointRadius: 3,
                        pointBackgroundColor: '#0369a1',
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: true, labels: { color: '#334155' } }
                    },
                    scales: {
                        x: { ticks: { color: '#64748b' }, grid: { color: '#e2e8f0' } },
                        y: { ticks: { color: '#64748b' }, grid: { color: '#e2e8f0' }, beginAtZero: true }
                    }
                }
            });

            let totalAccepted = 0;
            let totalNc = 0;
            Object.values(employeeData).forEach((entry) => {
                totalAccepted += entry.accepted || 0;
                totalNc += entry.nc || 0;
            });

            qualitySplitChart = new Chart(pieCanvas.getContext('2d'), {
                type: 'pie',
                data: {
                    labels: ['Accepted Qty', 'NC Qty'],
                    datasets: [{
                        data: [Math.max(0, totalAccepted), Math.max(0, totalNc)],
                        backgroundColor: ['#10b981', '#ef4444'],
                        borderColor: '#ffffff',
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: { color: '#334155', boxWidth: 14 }
                        }
                    }
                }
            });
        }

        function buildPartProducedBars(partProduced) {
            const container = document.getElementById('partProducedContainer');
            if (!container) {
                console.error('partProducedContainer not found');
                return;
            }

            const parts = Object.entries(partProduced)
                .sort((a, b) => b[1] - a[1])
                .slice(0, 10);

            console.log('Parts to display:', parts);

            if (parts.length === 0) {
                container.innerHTML = '<p class="analytics-empty">No part data available</p>';
                return;
            }

            const maxQty = Math.max(...parts.map(p => p[1]));
            const colors = ['#22d3ee', '#34d399', '#f59e0b', '#f472b6', '#60a5fa', '#a78bfa', '#f43f5e', '#06b6d4', '#84cc16', '#f97316'];
            const totalQty = parts.reduce((sum, [, qty]) => sum + qty, 0);
            const topPart = parts[0];

            let html = '<div class="neon-vertical-wrap">';
            parts.forEach(([partName, quantity], idx) => {
                const percentage = (quantity / maxQty) * 100;
                const barHeight = (percentage / 100) * 250;
                const color = colors[idx % colors.length];
                const safePart = escapeMetricText(partName);
                const isTop = partName === topPart[0];

                html += `
                    <div class="neon-vertical-item">
                        <span class="neon-vertical-top" style="--accent:${color};">${Math.round(quantity).toLocaleString()}</span>
                        <div class="neon-vertical-shell">
                            <div class="neon-vertical-bar" style="--accent:${color}; height:${Math.max(12, barHeight).toFixed(1)}px;"></div>
                        </div>
                        <span class="neon-vertical-label">${safePart}</span>
                        ${isTop ? '<span class="neon-tag" style="background:#0ea5e9;">TOP</span>' : ''}
                        <span class="neon-sub">${percentage.toFixed(0)}% of max</span>
                    </div>
                `;
            });

            html += '</div>';
            html += `
                <div class="hyper-summary-grid">
                    <div class="hyper-summary-card warn">
                        Top Part
                        <strong>${escapeMetricText(topPart[0])}</strong>
                        ${Math.round(topPart[1]).toLocaleString()} units
                    </div>
                    <div class="hyper-summary-card good">
                        Total Produced
                        <strong>${Math.round(totalQty).toLocaleString()}</strong>
                        units
                    </div>
                    <div class="hyper-summary-card">
                        Parts
                        <strong>${parts.length}</strong>
                        analyzed
                    </div>
                </div>
            `;

            container.innerHTML = html;
        }

        function buildEmployeeProductionTable(employeeData) {
            const container = document.getElementById('employeeProductionTable');
            if (!container) {
                console.error('employeeProductionTable not found');
                return;
            }

            const employees = Object.entries(employeeData)
                .sort((a, b) => b[1].produced - a[1].produced)
                .slice(0, 10);

            console.log('Employees to display:', employees);

            if (employees.length === 0) {
                container.innerHTML = '<p class="analytics-empty">No employee data available</p>';
                return;
            }

            let html = '';
            employees.forEach(([empName, data]) => {
                const produced = data.produced;
                const accepted = data.accepted;
                const nc = data.nc;
                const maxQty = Math.max(produced, accepted, nc, 1);
                const safeName = escapeMetricText(empName);
                
                const producedPercent = (produced / maxQty) * 100;
                const acceptedPercent = (accepted / maxQty) * 100;
                const ncPercent = (nc / maxQty) * 100;
                const producedColor = '#34d399';
                const acceptedColor = '#38bdf8';
                const ncColor = '#fb7185';
                
                html += `
                    <div class="employee-neon-card">
                        <div class="employee-neon-name">${safeName}</div>

                        <div class="employee-neon-metric">
                            <div class="employee-neon-metric-head">
                                <span>Produced</span>
                                <span class="hyper-value" style="--accent:${producedColor}; --glow:${withAlpha(producedColor)};">${Math.round(produced)} units</span>
                            </div>
                            <div class="hyper-bar-track">
                                <div class="hyper-bar-fill" style="--accent:${producedColor}; --glow:${withAlpha(producedColor)}; width:${producedPercent.toFixed(2)}%;">
                                    <span>${producedPercent.toFixed(0)}%</span>
                                </div>
                            </div>
                        </div>

                        <div class="employee-neon-metric">
                            <div class="employee-neon-metric-head">
                                <span>Accepted</span>
                                <span class="hyper-value" style="--accent:${acceptedColor}; --glow:${withAlpha(acceptedColor)};">${Math.round(accepted)} units</span>
                            </div>
                            <div class="hyper-bar-track">
                                <div class="hyper-bar-fill" style="--accent:${acceptedColor}; --glow:${withAlpha(acceptedColor)}; width:${acceptedPercent.toFixed(2)}%;">
                                    <span>${acceptedPercent.toFixed(0)}%</span>
                                </div>
                            </div>
                        </div>

                        <div class="employee-neon-metric">
                            <div class="employee-neon-metric-head">
                                <span>NC (Rejected)</span>
                                <span class="hyper-value" style="--accent:${ncColor}; --glow:${withAlpha(ncColor)};">${Math.round(nc)} units</span>
                            </div>
                            <div class="hyper-bar-track">
                                <div class="hyper-bar-fill" style="--accent:${ncColor}; --glow:${withAlpha(ncColor)}; width:${ncPercent.toFixed(2)}%;">
                                    <span>${ncPercent.toFixed(0)}%</span>
                                </div>
                            </div>
                        </div>

                        <div class="employee-neon-divider"></div>
                    </div>
                `;
            });
            
            container.innerHTML = html;
        }

        function buildEfficiencyMetrics(employeeData) {
            const container = document.getElementById('efficiencyNCContainer');
            if (!container) {
                console.error('efficiencyNCContainer not found');
                return;
            }

            const employees = Object.entries(employeeData)
                .map(([name, data]) => ({
                    name,
                    avgEfficiency: data.count > 0 ? (data.efficiency_sum / data.count).toFixed(2) : 0,
                    ncPercent: data.produced > 0 ? ((data.nc / data.produced) * 100).toFixed(2) : 0,
                    qualityScore: data.produced > 0 ? (100 - ((data.nc / data.produced) * 100)).toFixed(2) : 100
                }))
                .sort((a, b) => b.avgEfficiency - a.avgEfficiency);

            console.log('Efficiency metrics to display:', employees);

            if (employees.length === 0) {
                container.innerHTML = '<p class="analytics-empty">No efficiency data available</p>';
                return;
            }

            let html = `
                <div class="analytics-chart-grid">
                    <div class="analytics-chart-card">
                        <h5><i class="fas fa-chart-line"></i> Efficiency Trend</h5>
                        <canvas id="effTrendChart"></canvas>
                    </div>
                    <div class="analytics-chart-card">
                        <h5><i class="fas fa-chart-pie"></i> Quality Split</h5>
                        <canvas id="qualityPieChart"></canvas>
                    </div>
                </div>
                <div class="eff-grid">
            `;

            employees.slice(0, 10).forEach((emp) => {
                const efficiency = parseFloat(emp.avgEfficiency);
                const ncPercent = parseFloat(emp.ncPercent);
                const qualityScore = parseFloat(emp.qualityScore);
                const safeName = escapeMetricText(emp.name);
                const safeNc = Math.max(0, ncPercent).toFixed(2);
                const safeQuality = Math.max(0, qualityScore).toFixed(2);

                let efficiencyColor = '#34d399';
                if (efficiency < 70) efficiencyColor = '#fb7185';
                else if (efficiency < 85) efficiencyColor = '#fbbf24';
                else if (efficiency > 100) efficiencyColor = '#22d3ee';

                const qualityColor = '#60a5fa';
                const efficiencyBar = Math.min(Math.max(efficiency, 0), 100);
                const qualityBar = Math.min(Math.max(qualityScore, 0), 100);

                html += `
                    <div class="eff-neon-card" style="border-left:3px solid ${efficiencyColor};">
                        <div class="eff-name">${safeName}</div>

                        <div class="employee-neon-metric">
                            <div class="employee-neon-metric-head">
                                <span>Production Efficiency</span>
                                <span class="hyper-value" style="--accent:${efficiencyColor}; --glow:${withAlpha(efficiencyColor)};">${efficiency.toFixed(2)}%</span>
                            </div>
                            <div class="hyper-bar-track">
                                <div class="hyper-bar-fill" style="--accent:${efficiencyColor}; --glow:${withAlpha(efficiencyColor)}; width:${efficiencyBar.toFixed(2)}%;">
                                    <span>${efficiencyBar.toFixed(0)}%</span>
                                </div>
                            </div>
                        </div>

                        <div class="employee-neon-metric">
                            <div class="employee-neon-metric-head">
                                <span>Quality Score</span>
                                <span class="hyper-value" style="--accent:${qualityColor}; --glow:${withAlpha(qualityColor)};">${qualityScore.toFixed(2)}%</span>
                            </div>
                            <div class="hyper-bar-track">
                                <div class="hyper-bar-fill" style="--accent:${qualityColor}; --glow:${withAlpha(qualityColor)}; width:${qualityBar.toFixed(2)}%;">
                                    <span>${qualityBar.toFixed(0)}%</span>
                                </div>
                            </div>
                        </div>

                        <div class="eff-foot-grid">
                            <div class="kpi-chip">
                                <strong>${safeNc}%</strong>
                                NC Rate
                            </div>
                            <div class="kpi-chip">
                                <strong>${safeQuality}%</strong>
                                Acceptance
                            </div>
                        </div>
                    </div>
                `;
            });

            html += '</div>';
            container.innerHTML = html;
        }

        function buildMachineHoursAnalysis(machineHours) {
            const container = document.getElementById('machineHoursContainer');
            if (!container) {
                console.error('machineHoursContainer not found');
                return;
            }

            const machines = Object.entries(machineHours)
                .sort((a, b) => b[1] - a[1])
                .slice(0, 10);

            console.log('Machines to display:', machines);

            if (machines.length === 0) {
                container.innerHTML = '<p class="analytics-empty">No machine hours data available</p>';
                return;
            }

            const maxHours = Math.max(...machines.map(m => m[1]));
            const topMachine = machines[0];
            const colors = ['#60a5fa', '#22d3ee', '#34d399', '#a78bfa', '#f59e0b', '#fb7185', '#84cc16', '#06b6d4', '#f97316', '#38bdf8'];
            const totalHours = Object.values(machineHours).reduce((sum, value) => sum + value, 0);

            let html = '<div class="neon-vertical-wrap">';
            
            machines.forEach(([machineName, machineHour], idx) => {
                const percentage = (machineHour / maxHours) * 100;
                const barHeight = (percentage / 100) * 250;
                const isTop = machineName === topMachine[0];
                const color = isTop ? '#fb7185' : colors[idx % colors.length];
                const glow = withAlpha(color);
                const safeName = escapeMetricText(machineName);

                html += `
                    <div class="neon-vertical-item">
                        <span class="neon-vertical-top" style="--accent:${color}; --glow:${glow};">${machineHour.toFixed(1)}h</span>
                        <div class="neon-vertical-shell">
                            <div class="neon-vertical-bar" style="--accent:${color}; --glow:${glow}; height:${Math.max(12, barHeight).toFixed(1)}px;"></div>
                        </div>
                        <span class="neon-vertical-label">${safeName}</span>
                        ${isTop ? `<span class="neon-tag" style="background:${color}; box-shadow:0 0 10px ${glow};">TOP</span>` : ''}
                        <span class="neon-sub">${percentage.toFixed(0)}% of max</span>
                    </div>
                `;
            });

            html += '</div>';
            
            html += `
                <div class="hyper-summary-grid">
                    <div class="hyper-summary-card warn">
                        Most Used
                        <strong>${escapeMetricText(topMachine[0])}</strong>
                        ${topMachine[1].toFixed(2)} hours
                    </div>
                    <div class="hyper-summary-card good">
                        Total Hours
                        <strong>${totalHours.toFixed(2)}</strong>
                        All Machines
                    </div>
                    <div class="hyper-summary-card">
                        Machines
                        <strong>${machines.length}</strong>
                        Active
                    </div>
                </div>
            `;
            
            container.innerHTML = html;
        }

        function buildToolStrokeAnalysis(toolStrokes) {
            const container = document.getElementById('toolStrokeContainer');
            if (!container) {
                console.error('toolStrokeContainer not found');
                return;
            }

            const tools = Object.entries(toolStrokes)
                .map(([name, data]) => ({
                    name,
                    total: data.total,
                    remaining: data.remaining,
                    maxLife: data.maxLife,
                    usage: ((data.total / data.maxLife) * 100).toFixed(1)
                }))
                .sort((a, b) => b.total - a.total)
                .slice(0, 10);

            console.log('Tools to display:', tools);

            if (tools.length === 0) {
                container.innerHTML = '<p class="analytics-empty">No tool stroke data available</p>';
                return;
            }

            const maxStroke = Math.max(...tools.map(t => t.total));
            const colors = ['#f59e0b', '#fb7185', '#a78bfa', '#60a5fa', '#22d3ee', '#34d399', '#84cc16', '#f97316', '#38bdf8', '#f43f5e'];

            let html = '<div class="neon-vertical-wrap">';
            
            tools.forEach((tool, idx) => {
                const percentage = (tool.total / maxStroke) * 100;
                const barHeight = (percentage / 100) * 250;
                let statusLabel = 'GOOD';
                let statusColor = '#34d399';
                if (tool.remaining <= 0) {
                    statusLabel = 'REPLACE';
                    statusColor = '#fb7185';
                } else if (tool.remaining < (tool.maxLife * 0.1)) {
                    statusLabel = 'URGENT';
                    statusColor = '#f59e0b';
                } else if (tool.remaining < (tool.maxLife * 0.25)) {
                    statusLabel = 'SOON';
                    statusColor = '#fbbf24';
                }

                const color = statusLabel === 'GOOD' ? colors[idx % colors.length] : statusColor;
                const glow = withAlpha(color);
                const safeName = escapeMetricText(tool.name);
                const remainingLabel = Math.max(0, Math.round(tool.remaining)).toLocaleString();

                html += `
                    <div class="neon-vertical-item">
                        <span class="neon-vertical-top" style="--accent:${color}; --glow:${glow};">${Math.round(tool.total).toLocaleString()}</span>
                        <div class="neon-vertical-shell">
                            <div class="neon-vertical-bar" style="--accent:${color}; --glow:${glow}; height:${Math.max(12, barHeight).toFixed(1)}px;"></div>
                        </div>
                        <span class="neon-vertical-label">${safeName}</span>
                        <span class="neon-tag" style="background:${statusColor}; box-shadow:0 0 10px ${withAlpha(statusColor)};">${statusLabel}</span>
                        <span class="neon-sub">${remainingLabel} strokes left</span>
                    </div>
                `;
            });

            html += '</div>';
            
            const criticalTools = tools.filter(t => t.remaining <= 0).length;
            const urgentTools = tools.filter(t => t.remaining > 0 && t.remaining < (t.maxLife * 0.1)).length;
            const soonTools = tools.filter(t => t.remaining >= (t.maxLife * 0.1) && t.remaining < (t.maxLife * 0.25)).length;
            const healthyTools = tools.length - criticalTools - urgentTools - soonTools;
            
            html += `
                <div class="hyper-summary-grid">
                    <div class="hyper-summary-card crit">
                        Replace Now
                        <strong>${criticalTools}</strong>
                        Tools
                    </div>
                    <div class="hyper-summary-card warn">
                        Urgent / Soon
                        <strong>${urgentTools + soonTools}</strong>
                        Tools
                    </div>
                    <div class="hyper-summary-card good">
                        Healthy
                        <strong>${healthyTools}</strong>
                        Tools
                    </div>
                </div>
            `;
            
            container.innerHTML = html;
        }
    </script>
</body>
</html>
