# Notification UI Integration Guide

## Overview
This guide explains how to integrate the notification system UI (bell icon, sidebar popup, settings modal) into your admin dashboard.

## Files Created
- `notification_ui.html` - Complete notification UI component with HTML, CSS, and JavaScript

## Integration Steps

### Step 1: Add Notification Button to Navbar
Add this code to your admin dashboard header/navbar (typically in `index.php` or navbar section):

```html
<!-- Add this to your navbar, typically in the top-right corner -->
<button class="notification-btn" id="notificationBtn" title="View Notifications" onclick="toggleNotificationSidebar()">
    <i class="fas fa-bell"></i>
</button>
```

### Step 2: Include the Notification UI Component
Add this code to the bottom of your admin dashboard HTML (before closing `</body>` tag):

```html
<!-- Include Notification UI Component -->
<?php 
    // Only show notifications for admin users
    if (isset($_SESSION['admin']) && $_SESSION['admin'] == true) {
        include 'notification_ui.html';
    }
?>
```

### Step 3: Ensure FontAwesome Icons Are Loaded
Make sure your admin dashboard includes FontAwesome for icons:

```html
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
```

## File Structure After Integration

```
admin_dashboard/
├── index.php (navbar contains notification button)
├── notification_ui.html (included at bottom)
├── notifications.php (backend API - already created)
├── css/
│   └── style.css
└── js/
    └── navbar_alerts.js
```

## UI Components Explained

### 1. Notification Button (Top-Right Corner)
- Bell icon with red badge showing notification count
- Click to open/close sidebar
- Badge pulses when active notifications exist

### 2. Notification Sidebar (Right Slide-Out)
- Slides in from right side when bell is clicked
- Shows list of active notifications
- Each notification displays:
  - Type (Machine or Tool) with icon
  - Title and message
  - Severity badge (critical/warning/info)
  - Item name and timestamp
  - "Go to Maintenance" action button
- Footer with Settings and Clear All buttons

### 3. Notification Settings Modal
- Appears when "Settings" button is clicked
- Allows adjusting thresholds:
  - Machine Maintenance Alert (days before due)
  - Tool Stroke Alert (remaining strokes)
- Default values: 15 days for machines, 1000 strokes for tools
- Save button persists settings to database

## Features

### Auto-Refresh
- Notifications auto-refresh every 30 seconds
- Change interval by modifying `this.refreshInterval` value in NotificationManager class

### Severity Indicators
- **Critical** (Red): Less than 5 days or 100 strokes remaining
- **Warning** (Orange): Within configured threshold
- **Info** (Blue): General alerts

### Admin-Only Access
- All notification UI components are wrapped in admin check
- Backend API requires admin authentication
- Settings panel only accessible to admin users

### Responsive Design
- Works on desktop and mobile
- Sidebar adjusts width on smaller screens
- Modal is fully responsive

## Customization Options

### Change Alert Thresholds
Edit in `NotificationManager.loadSettings()`:
```javascript
document.getElementById('machineThreshold').value = data.settings.machine_days_threshold;
document.getElementById('toolThreshold').value = data.settings.tool_strokes_threshold;
```

### Change Auto-Refresh Interval
```javascript
this.refreshInterval = 30000; // 30 seconds (in milliseconds)
```

### Change Sidebar Width
Edit CSS:
```css
.notification-sidebar {
    width: 420px; /* Change this value */
}
```

### Change Notification Colors
Edit CSS severity classes:
```css
.notification-item.critical {
    border-left-color: #e74c3c; /* Critical color */
}
```

## Testing the System

### 1. Test Notification Generation
- Go to maintenance.php
- Check if notifications appear in the bell icon
- Click bell to see sidebar

### 2. Test Settings Update
- Click Settings button in sidebar
- Modify threshold values
- Click Save
- Verify values are updated in database (notification_settings table)

### 3. Test Admin-Only Access
- Log in as admin and verify notification UI appears
- Log in as non-admin and verify notification UI doesn't appear
- Try accessing notifications.php directly as non-admin (should return error)

### 4. Test Real-Time Updates
- Open maintenance.php in one window
- Open entry_log.php in another window
- Enter a production log for a tool approaching stroke limit
- Watch notification appear in bell icon (within 30 seconds)

## Styling Reference

### CSS Classes Available
- `.notification-btn` - Bell button styling
- `.notification-sidebar` - Main sidebar container
- `.notification-item` - Individual notification item
- `.notification-severity` - Severity badge
- `.notification-settings-modal` - Settings dialog

### Color Scheme
- Primary: #667eea (purple/blue gradient)
- Critical: #e74c3c (red)
- Warning: #f39c12 (orange)
- Info: #3498db (blue)
- Background: #ffffff (white)
- Text: #333333 (dark gray)

## Troubleshooting

### Notifications Not Appearing
1. Check browser console for JavaScript errors
2. Verify notifications.php is accessible
3. Check database for notification_settings table
4. Verify admin user is logged in

### Settings Not Saving
1. Check database write permissions
2. Verify POST request is sent to notifications.php
3. Check database for updated values in notification_settings table
4. Verify admin authentication is passing

### Sidebar Not Sliding
1. Check if the `open` class is being applied
2. Verify CSS transform transitions are working
3. Check for conflicting CSS z-index issues

## Database Requirements

Ensure the notification_settings table exists:
```sql
CREATE TABLE notification_settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(50) UNIQUE,
    setting_value INT,
    description TEXT,
    last_updated TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

INSERT INTO notification_settings VALUES
(1, 'machine_days_threshold', 15, 'Days before machine maintenance alert'),
(2, 'tool_strokes_threshold', 1000, 'Remaining strokes before tool alert');
```

Run this if not already created:
```bash
mysql -u root < NOTIFICATION_SYSTEM_SETUP.sql
```

## API Endpoints Used

1. **GET** `notifications.php?action=get_notifications`
   - Returns array of active notifications
   - Requires admin authentication

2. **GET** `notifications.php?action=get_notification_settings`
   - Returns current threshold settings
   - Requires admin authentication

3. **POST** `notifications.php?action=update_notification_settings`
   - Updates threshold values
   - Parameters: machine_days_threshold, tool_strokes_threshold
   - Requires admin authentication

## Performance Considerations

- Auto-refresh interval set to 30 seconds (adjustable)
- Only makes API calls if admin user is logged in
- Sidebar overlay prevents interaction with page behind it
- Notifications are sorted by severity (critical first)

## Next Steps (Optional Enhancements)

1. **Sound Alerts** - Play sound when critical notification arrives
2. **Desktop Notifications** - Use browser Notification API
3. **Email Alerts** - Send email when critical threshold reached
4. **Notification History** - Keep log of all notifications
5. **Dismissible Notifications** - Allow users to mark as read
6. **Notification Filters** - Filter by type (machines vs tools)
