# Machine & Tools Management - INTEGRATION GUIDE

## New Architecture

### 1. **admin_fields.php** (MANAGEMENT)
- Add/Edit/Create machines
- Add/Edit/Create tools  
- Full CRUD operations
- Configuration management

**Integration:** Include admin_machines_tools_section.php at the bottom of admin_fields.php

```php
<!-- At the end of admin_fields.php, before closing </body> -->
<?php include 'admin_machines_tools_section.php'; ?>
```

### 2. **machines_tools_admin.php** (REPORTING & MONITORING)
- View-only dashboards
- Add maintenance reports/logs
- Track tool strokes
- View all alerts
- Real-time monitoring

### 3. **alerts.php** (ALERT SYSTEM)
- Automatic alert generation
- Machine maintenance reminders
- Tool stroke warnings
- Global notification system

---

## WORKFLOW

### Machine Management
1. **Admin goes to:** admin_fields.php → Machines tab
2. **Admin can:**
   - Add new machines with interval
   - View all configured machines
   - Enable/Disable machines
3. **Then goes to:** machines_tools_admin.php → Machines tab
4. **Staff can:**
   - View machine details
   - Add maintenance reports (auto-resets cycle)
   - See alerts

### Tool Management
1. **Admin goes to:** admin_fields.php → Tools tab
2. **Admin can:**
   - Add new tools with max strokes
   - View all configured tools
3. **Then goes to:** machines_tools_admin.php → Tools tab
4. **Staff can:**
   - View tool details
   - See stroke history
   - Refurbish/reset strokes

---

## FILE SUMMARY

| File | Purpose | Access |
|------|---------|--------|
| **admin_fields.php** | Create machines & tools | Admin only |
| **machines_tools_admin.php** | View details, add reports | All staff |
| **machines_tools_api.php** | Backend API | All authorized |
| **admin_machines_tools_section.php** | UI components for admin_fields | Include file |
| **alerts.php** | Alert checking & API | All authorized |
| **navbar_alerts.js** | Notification bell | All users |

---

## HOW TO IMPLEMENT

### Step 1: Add to admin_fields.php
Add this line at the END of admin_fields.php (before `</body>`):

```php
<?php include 'admin_machines_tools_section.php'; ?>
```

### Step 2: Verify Files Exist
- ✅ admin_machines_tools_section.php
- ✅ machines_tools_admin.php (redesigned)
- ✅ machines_tools_api.php
- ✅ alerts.php
- ✅ navbar_alerts.js

### Step 3: Update Sidebar
Already done! sidebar.js includes the "Machines & Tools" menu item

### Step 4: Access
- **Admin:** 
  - Visit `admin_fields.php` → Click "Machines" or "Tools" tab
  - Or visit `machines_tools_admin.php` for dashboards
  
- **Staff:**
  - Visit `machines_tools_admin.php` to view and add reports
  - Alerts appear in navbar bell icon

---

## API USAGE

All API endpoints are in `machines_tools_api.php`:

### Getting Data
```javascript
// Get all machines
fetch('machines_tools_api.php?action=get_machines')

// Get all tools
fetch('machines_tools_api.php?action=get_tools')

// Get stroke history
fetch('machines_tools_api.php?action=get_tool_stroke_history&tool_id=1')
```

### Posting Data
```javascript
// Add machine
const fd = new FormData();
fd.append('action', 'add_machine');
fd.append('machine_name', 'CNC-001');
fd.append('maintenance_interval_days', 90);
fetch('machines_tools_api.php', {method: 'POST', body: fd})

// Log maintenance
const fd = new FormData();
fd.append('action', 'log_maintenance');
fd.append('machine_id', 1);
fd.append('maintenance_date', '2025-02-10');
fetch('machines_tools_api.php', {method: 'POST', body: fd})

// Refurbish tool
const fd = new FormData();
fd.append('action', 'refurbish_tool');
fd.append('tool_id', 1);
fd.append('refurbish_date', '2025-02-10');
fetch('machines_tools_api.php', {method: 'POST', body: fd})
```

---

## ADMIN_FIELDS.PHP MODIFICATIONS

The admin_machines_tools_section.php file includes tabs for managing machines and tools. When included in admin_fields.php, users will see:

**Original Tabs:**
- Form Fields (existing)

**New Tabs (Added):**
- Machines 
- Tools

The section is self-contained and doesn't interfere with existing functionality.

---

## ALERT SYSTEM

Alerts are auto-generated by `alerts.php`:

### Machine Alerts
- Generated when: maintenance is due within 15 days
- Severity: Medium/High/Critical based on urgency
- Auto-resolves: When maintenance is logged

### Tool Alerts  
- Generated when: remaining strokes ≤ warning threshold
- Severity: High/Critical based on remaining strokes
- Auto-resolves: When tool is refurbished

### Viewing Alerts
- Navbar bell icon shows count
- Click bell to see dropdown
- Visit machines_tools_admin.php → Alerts tab for full list

---

## TESTING CHECKLIST

- [ ] Include admin_machines_tools_section.php in admin_fields.php
- [ ] Verify admin_fields.php shows three tabs (Fields, Machines, Tools)
- [ ] Add a test machine in admin_fields.php
- [ ] Add a test tool in admin_fields.php
- [ ] Visit machines_tools_admin.php
- [ ] Verify machines show in "Machines" tab
- [ ] Click "Add Maintenance Report" button
- [ ] Submit a maintenance report
- [ ] Verify tool details show in "Tools" tab
- [ ] Check stroke history (should be empty initially)
- [ ] Visit "Alerts" tab
- [ ] Add machines/tools again - verify at least 1 alert appears
- [ ] Click navbar alert bell
- [ ] Verify alert dropdown shows notifications

---

## TROUBLESHOOTING

**Q: admin_fields.php doesn't show Machines tab**
A: Make sure you added the include statement at the end of admin_fields.php file

**Q: Machines not showing in machines_tools_admin.php**
A: Clear browser cache, check browser console for errors, verify database has tables

**Q: API errors in console**
A: Make sure machines_tools_api.php exists and is in same directory as admin_fields.php

**Q: Alerts not appearing**
A: Check `system_alerts` table exists, run alerts.php?action=get_alerts manually

---

## SUMMARY OF CHANGES

✅ **New file:** admin_machines_tools_section.php - Machines & Tools management UI for admin_fields.php
✅ **Redesigned:** machines_tools_admin.php - Now read-only dashboard with reporting features
✅ **Updated:** sidebar.js - Includes "Machines & Tools" menu item (already done)
✅ **Ready:** alerts.php, machines_tools_api.php, navbar_alerts.js, MACHINES_TOOLS_SETUP.sql

This creates a clean separation:
- **Admin manages** (admin_fields.php)
- **Staff reports** (machines_tools_admin.php)
- **System alerts** (navbar)
