
# Machine Maintenance & Tool Stroke Life Management System

## COMPLETE IMPLEMENTATION GUIDE

### Overview
This system is a complete solution for tracking machine maintenance schedules and tool stroke life usage in a manufacturing environment. It includes:
- **Automatic maintenance cycle tracking** (time-based)
- **Tool stroke life management** (usage-based)
- **Real-time alert system** with critical notifications
- **Admin dashboard** for management
- **Integration** with existing production entry system

---

## 1. DATABASE SETUP

### Execute SQL Script
Run the SQL commands from `MACHINES_TOOLS_SETUP.sql`:

```bash
mysql -u root zemaraim_mektron_db < MACHINES_TOOLS_SETUP.sql
```

Or paste the SQL directly in phpMyAdmin.

### Tables Created:
1. **machines** - Machine registry with maintenance intervals
2. **maintenance_history** - Log of all maintenance performed
3. **tools_extended** - Tool inventory with stroke life tracking
4. **tool_stroke_history** - Log of tool usage/strokes
5. **tool_refurbish_log** - History of tool refurbishment
6. **system_alerts** - Global alert management

---

## 2. FILE STRUCTURE

### Core Files:
```
/erp.manufacture/
├── MACHINES_TOOLS_SETUP.sql          ✅ Database creation script
├── alerts.php                         ✅ Alert checking & API endpoints
├── machines_tools_api.php             ✅ Backend logic for machines/tools
├── machines_tools_admin.php           ✅ Admin management interface
├── navbar_alerts.js                   ✅ Navbar notification system
├── api.php                           ✅ Updated with tool stroke tracking
├── entry_log.php                     ✅ Production entry form (integrate tool selector)
└── style.css                         ✅ Add navbar alert styles
```

---

## 3. API ENDPOINTS

### Machines API (machines_tools_api.php)

#### Add Machine
```bash
POST /machines_tools_api.php
action=add_machine
machine_name=CNC-001
machine_code=CNC-001
purchase_date=2023-01-15
maintenance_interval_days=90
```

#### Get All Machines
```bash
GET /machines_tools_api.php?action=get_machines
```
Response: Array with maintenance status, days until due, etc.

#### Get Machine by ID
```bash
GET /machines_tools_api.php?action=get_machine&id=1
```

#### Update Machine
```bash
POST /machines_tools_api.php
action=update_machine
id=1
machine_name=New Name
maintenance_interval_days=60
```

#### Log Maintenance
```bash
POST /machines_tools_api.php
action=log_maintenance
machine_id=1
maintenance_date=2025-02-10
maintenance_type=Preventive
description=Routine maintenance completed
technician_name=John Doe
duration_hours=2
```

#### Get Maintenance History
```bash
GET /machines_tools_api.php?action=get_maintenance_history&machine_id=1
```

### Tools API (machines_tools_api.php)

#### Add Tool
```bash
POST /machines_tools_api.php
action=add_tool
tool_name=Drill Bit 10mm
tool_code=DB-10MM
max_life_strokes=100000
warning_threshold=1000
```

#### Get All Tools
```bash
GET /machines_tools_api.php?action=get_tools
```
Response: Includes remaining strokes, usage %, status, etc.

#### Get Tool by ID
```bash
GET /machines_tools_api.php?action=get_tool&id=1
```

#### Increment Tool Strokes (Automatic - called from save_log)
```bash
POST /machines_tools_api.php
action=increment_tool_strokes
tool_id=1
quantity=150
```

#### Refurbish Tool
```bash
POST /machines_tools_api.php
action=refurbish_tool
tool_id=1
refurbished_by=John Doe
refurbish_date=2025-02-10
notes=Tool sharpened and restored
```

#### Get Tool Stroke History
```bash
GET /machines_tools_api.php?action=get_tool_stroke_history&tool_id=1
```

#### Get Tool Refurbish History
```bash
GET /machines_tools_api.php?action=get_tool_refurb_history&tool_id=1
```

### Alerts API (alerts.php)

#### Get Active Alerts
```bash
GET /alerts.php?action=get_alerts
```
Response: All active alerts with full details

#### Get Alert Count
```bash
GET /alerts.php?action=get_alert_count
```
Response: Number of unread alerts

#### Mark Alert as Read
```bash
POST /alerts.php
action=mark_alert_read
alert_id=1
```

#### Resolve Alert
```bash
POST /alerts.php
action=resolve_alert
alert_id=1
```

---

## 4. INTEGRATION POINTS

### A. Production Entry Form (entry_log.php)

Add tool selector to the production entry form:

```html
<div class="form-group">
    <label>Tool Used *</label>
    <select id="toolId" name="tool_id" class="form-input" required>
        <option value="">-- Select Tool --</option>
    </select>
</div>

<script>
// Load tools on page load
document.addEventListener('DOMContentLoaded', function() {
    fetch('machines_tools_api.php?action=get_tools')
        .then(res => res.json())
        .then(data => {
            if (data.success) {
                const select = document.getElementById('toolId');
                data.data.forEach(tool => {
                    const remaining = tool.max_life_strokes - tool.current_strokes_used;
                    const statusColor = remaining <= 0 ? '🔴' : 
                                       remaining <= tool.warning_threshold ? '🟠' : '🟢';
                    const option = document.createElement('option');
                    option.value = tool.id;
                    option.textContent = `${statusColor} ${tool.tool_name} (${remaining} strokes left)`;
                    select.appendChild(option);
                });
            }
        });
});
</script>
```

### B. Production Log Submission (api.php - save_log)

The save_log function now automatically:
1. Checks if tool_id and produced_qty are provided
2. Gets the tool details
3. Updates tool's current_strokes_used by adding produced_qty
4. Logs to tool_stroke_history table

**This is ALREADY integrated in api.php** - No additional changes needed!

### C. Navbar Integration

Add this to your navbar.html or navbar section:

```html
<!-- Alert Bell Icon -->
<div style="position: relative; margin-left: 15px;">
    <button id="alertBell" class="nav-alert-btn" onclick="showAlertDropdown()" title="Notifications">
        <i class="fas fa-bell"></i>
        <span id="alertCount" class="alert-badge">0</span>
    </button>
    <div id="alertDropdown" class="alert-dropdown"></div>
</div>

<script src="navbar_alerts.js"></script>
```

Add this CSS to style.css:

```css
.nav-alert-btn {
    background: none;
    border: none;
    font-size: 20px;
    cursor: pointer;
    color: #666;
    position: relative;
    padding: 8px;
    transition: 0.2s;
    display: inline-flex;
    align-items: center;
    justify-content: center;
}

.nav-alert-btn:hover {
    color: #013B3B;
    transform: scale(1.1);
}

.nav-alert-btn.has-alerts {
    color: #dc3545;
    animation: ring 2s infinite;
}

@keyframes ring {
    0%, 100% { transform: rotate(0); }
    10%, 20% { transform: rotate(-15deg); }
    30%, 50%, 70%, 90% { transform: rotate(15deg); }
    40%, 60%, 80% { transform: rotate(-15deg); }
}

.alert-badge {
    position: absolute;
    top: -5px;
    right: -5px;
    background: #dc3545;
    color: white;
    border-radius: 50%;
    width: 24px;
    height: 24px;
    display: none;
    align-items: center;
    justify-content: center;
    font-size: 11px;
    font-weight: bold;
    border: 2px solid white;
}

.alert-dropdown {
    position: absolute;
    top: 100%;
    right: 0;
    background: white;
    border: 1px solid #ddd;
    border-radius: 6px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    z-index: 1000;
    min-width: 380px;
    margin-top: 10px;
    display: none;
}

.alert-dropdown.show {
    display: block;
}
```

---

## 5. ADMIN DASHBOARD

Access the admin dashboard at: **`machines_tools_admin.php`**

Features:
- **Machines Tab**: View all machines, add new machines, log maintenance
- **Tools Tab**: View all tools, add new tools, refurbish/reset tools
- **Alerts Tab**: View all active alerts by severity

### Dashboard Capabilities:

#### Machines Management:
- Add new machines with purchase date and interval
- View maintenance status (Overdue/Critical/Due Soon/Active)
- Log maintenance with type, technician info, duration
- Automatic next due date calculation (adds interval to today)
- View complete maintenance history

#### Tools Management:
- Add new tools with max stroke life
- Set warning threshold for low stroke alerts
- View usage percentage and remaining strokes
- Refurbish/reset tool strokes (logs previous condition)
- Track refurbishment history

#### Alert System:
- Real-time alert generation
- Color-coded severity levels (Critical/High/Medium/Low)
- Alert dismissal and resolution
- Auto-refresh every 30 seconds

---

## 6. ALERT LOGIC

### Machine Maintenance Alerts:
Generated when:
- Next due date is within 15 days: **Medium severity**
- Next due date is within 5 days: **High severity**
- Maintenance is overdue: **Critical severity**

Alert message: `"Machine [Name] is due for maintenance in X days (on [date])."`

### Tool Stroke Alerts:
Generated when:
- Remaining strokes ≤ warning_threshold: **High severity**
- Remaining strokes ≤ warning_threshold/10: **Critical severity**

Alert message: `"Tool [Name] Critical! Only X strokes remaining. Max: Y, Used: Z. Replace soon!"`

### Alert Status:
- **is_read** = 0: Unread (shows in navbar badge)
- **is_read** = 1: Read
- **is_active** = 1: Active alert
- **is_active** = 0: Resolved/Dismissed
- **resolved_at**: Timestamp when alert was resolved

---

## 7. WORKFLOW EXAMPLES

### Workflow 1: Machine Maintenance Cycle

1. **Admin adds machine** → "CNC Machine 001"
   - Purchase date: 2023-01-15
   - Interval: 90 days
   - Next due: TODAY + 90 days

2. **System generates alert** when next_due_date is within 15 days

3. **Technician logs maintenance** in admin panel
   - Maintenance date: TODAY
   - Type: Preventive
   - Technician: John Doe
   - Duration: 2 hours

4. **System updates automatically**:
   - last_maintenance_date = TODAY
   - next_due_date = TODAY + 90 days
   - Alert resolved and deactivated

### Workflow 2: Tool Usage Tracking

1. **Admin adds tool** → "Drill Bit 10mm"
   - Max strokes: 100,000
   - Warning threshold: 1,000

2. **Operator produces parts** in entry_log.php
   - Selects tool: "Drill Bit 10mm"
   - Produces: 150 units
   - Submits form

3. **System automatically**:
   - Saves production log
   - Updates tool: current_strokes_used += 150
   - Logs to tool_stroke_history
   - Calculates remaining: 100,000 - current = X strokes

4. **System generates alerts** when remaining ≤ threshold:
   - Shows in navbar bell (red)
   - Toast notification appears
   - Admin can see in machines_tools_admin.php

5. **Admin refurbishes tool**:
   - Logs refurbishment with details
   - Resets current_strokes_used = 0
   - Tool status changes to Active
   - Alert automatically resolved

---

## 8. VALIDATION & ERROR HANDLING

### Validation Rules:
- Machine name must be unique
- Tool name must be unique
- Maintenance interval > 0
- Max strokes > 0
- Tool strokes cannot exceed max_life_strokes
- Produced quantity must be > 0 (from production entry)

### Error Responses:
All API endpoints return JSON:
```json
{
  "success": false,
  "error": "Error message"
}
```

---

## 9. PERFORMANCE NOTES

### Alert Caching:
- Alert checks are performed on-demand (GET /alerts.php?action=get_alerts)
- Navbar caches alerts for 30 seconds
- Admin dashboard auto-refreshes every 30 seconds

### Database Optimization:
- Indexes on: next_due_date, status, current_strokes_used
- Limited alert history (keep 1000 most recent)
- Archive old maintenance records annually

### Scalability:
- Supports unlimited machines and tools
- Can handle thousands of tools with efficient queries
- Alert system designed for real-time performance

---

## 10. USER ROLES & PERMISSIONS

### Admin Only:
- Add/modify machines
- Add/modify tools
- Log maintenance
- Refurbish tools
- Manage alerts
- Access machines_tools_admin.php

### All Logged-In Users:
- View tool status in production entry
- Automatic tool stroke tracking on production save
- View navbar alerts
- See alert count in bell icon

---

## 11. COMMON TASKS

### Q: How do I add a new machine?
A: Go to **machines_tools_admin.php** → **Machines tab** → **Add New Machine** button

### Q: How do I check if a tool needs replacement?
A: Go to **machines_tools_admin.php** → **Tools tab** → See "Status" column (Critical = needs replacement)

### Q: How are tool strokes automatically updated?
A: When you enter production data in **entry_log.php** and select a tool, the strokes are added automatically when you submit.

### Q: How do I reset a tool after sharpening?
A: Go to **machines_tools_admin.php** → **Tools tab** → Find tool → Click **Refurbish** button → Submit form

### Q: Why is an alert showing?
A: Check **machines_tools_admin.php** → **Alerts tab** to see all active alerts and their details

### Q: How do I dismiss an alert?
A: Currently alerts auto-resolve when condition is fixed (maintenance logged or tool refurbished). You can manually resolve via database or admin panel.

---

## 12. TROUBLESHOOTING

### Tool strokes not updating:
1. Check if tool_id is being sent in production form
2. Check if tool exists in tools_extended table
3. Review browser console for JavaScript errors
4. Check api.php error logs

### Alerts not showing:
1. Run `GET /alerts.php?action=get_alerts` to manually check
2. Verify system_alerts table has entries
3. Check navbar_alerts.js is loaded
4. Clear browser cache

### Machine maintenance not resetting:
1. Verify machine exists in machines table
2. Check if next_due_date calculation is correct (should be today + interval)
3. Verify maintenance_date is being passed correctly
4. Check MySQL date format (YYYY-MM-DD)

---

## 13. SECURITY NOTES

- All admin functions require `requireAdmin()` check
- All logged-in functions require `requireLogin()` check
- SQL injection protection via prepared statements (PDO)
- XSS protection via HTML escaping
- CSRF tokens recommended for production (add if needed)

---

## 14. FUTURE ENHANCEMENTS

Potential improvements:
- Predictive maintenance based on usage trends
- Cost tracking for maintenance and tools
- Export reports (PDF/Excel)
- Mobile app for technicians
- SMS/Email alerts integration
- Maintenance approvals workflow
- Multi-site support
- Tool allocation/assignment tracking

---

## QUICK START CHECKLIST

- [ ] Run SQL script to create tables
- [ ] Include machines_tools_api.php in api.php (if needed)
- [ ] Add alert bell to navbar HTML
- [ ] Add navbar_alerts.js script to navbar
- [ ] Add navbar alert CSS to style.css
- [ ] Add tool selector to entry_log.php
- [ ] Verify api.php has tool stroke tracking in save_log
- [ ] Access machines_tools_admin.php and add machines/tools
- [ ] Test production entry with tool selection
- [ ] Verify alerts appear in navbar

---

**Created:** February 10, 2025
**System Version:** 1.0
**Status:** Complete & Ready for Deployment
