# Complete Comparison Chart Implementation Guide

## Overview
This guide provides the complete, working JavaScript code for the **Combined Field Comparison Chart** with Bar and Line visualization options.

---

## Key Components

### 1. **Data Storage (Global Variable)**
```javascript
window.comparisonChartData = {
    data1: null,           // First field data
    data2: null,           // Second field data
    label1: '',            // First field label
    label2: '',            // Second field label
    currentType: 'bar'     // Current chart type
};
```

### 2. **Chart Type Toggle Function**
The `switchComparisonChart()` function allows switching between Bar and Line charts:
```javascript
function switchComparisonChart(chartType) {
    // Updates button states
    // Calls appropriate chart rendering function
    // Maintains data and redraws
}
```

**Usage:**
- `switchComparisonChart('bar')` - Shows bar chart
- `switchComparisonChart('line')` - Shows line chart

### 3. **Bar Chart Function**
`buildDistributionComparison()` renders a comparison bar chart with:
- Dual datasets (Field 1 & Field 2)
- Smart orientation (horizontal for >8 items, vertical for ≤8 items)
- Interactive tooltips with custom labels
- Hover effects with color enhancement
- Responsive sizing

**Features:**
- Displays top 12 values
- Auto-pads shorter datasets
- Beautiful color gradients
- Full responsiveness

### 4. **Line Chart Function**
`buildLineComparison()` renders a comparison line chart with:
- Smooth curves (tension: 0.4)
- Filled areas under lines
- Large interactive points
- Multiple dataset support

**Features:**
- Curved trend visualization
- Point highlighting on hover
- Legend and tooltips
- Smooth animations

---

## Complete JavaScript Code

### HTML Setup (Buttons & Canvas)
```html
<!-- Chart Type Toggle Buttons -->
<button onclick="switchComparisonChart('bar')" id="btnComparisonBar">
    <i class="fas fa-chart-bar"></i> Bar
</button>
<button onclick="switchComparisonChart('line')" id="btnComparisonLine">
    <i class="fas fa-chart-line"></i> Line
</button>

<!-- Chart Canvas Container -->
<div style="position: relative; height: 450px; width: 100%; min-height: 350px; background: #fafafa;">
    <canvas id="comparisonDistributionChart"></canvas>
</div>
```

### Complete JavaScript Functions

#### Function 1: Build Comparison Charts
```javascript
function buildComparisonCharts(data1, data2, label1, label2) {
    try {
        // Store data for chart type switching
        window.comparisonChartData = {
            data1: data1,
            data2: data2,
            label1: label1,
            label2: label2,
            currentType: 'bar'
        };
        
        // Display bar chart by default
        buildDistributionComparison(data1, data2, label1, label2);
    } catch (e) {
        console.error('Error in distribution comparison:', e);
    }
}
```

#### Function 2: Switch Chart Type
```javascript
function switchComparisonChart(chartType) {
    if (!window.comparisonChartData.data1 || !window.comparisonChartData.data2) {
        console.warn('No comparison data available');
        return;
    }

    // Update button styles
    document.getElementById('btnComparisonBar').style.background = 
        chartType === 'bar' ? '#1abc9c' : '#f0f0f0';
    document.getElementById('btnComparisonBar').style.color = 
        chartType === 'bar' ? 'white' : '#555';
    document.getElementById('btnComparisonLine').style.background = 
        chartType === 'line' ? '#1abc9c' : '#f0f0f0';
    document.getElementById('btnComparisonLine').style.color = 
        chartType === 'line' ? 'white' : '#555';

    window.comparisonChartData.currentType = chartType;

    // Render appropriate chart
    if (chartType === 'bar') {
        buildDistributionComparison(
            window.comparisonChartData.data1,
            window.comparisonChartData.data2,
            window.comparisonChartData.label1,
            window.comparisonChartData.label2
        );
    } else if (chartType === 'line') {
        buildLineComparison(
            window.comparisonChartData.data1,
            window.comparisonChartData.data2,
            window.comparisonChartData.label1,
            window.comparisonChartData.label2
        );
    }
}
```

#### Function 3: Build Bar Chart
```javascript
function buildDistributionComparison(data1, data2, label1, label2) {
    try {
        const ctx = document.getElementById('comparisonDistributionChart');
        if (!ctx) {
            console.error('Distribution chart context not found');
            return;
        }

        const values1 = data1.data || [];
        const values2 = data2.data || [];
        
        if (!values1 || values1.length === 0 || !values2 || values2.length === 0) {
            console.error('Invalid data for distribution comparison');
            ctx.parentElement.innerHTML = 
                '<div style="text-align: center; padding: 40px; color: #999;">' +
                '<i class="fas fa-info-circle"></i> No data available</div>';
            return;
        }

        // Get top values
        const displayCount = Math.min(Math.max(values1.length, values2.length), 12);
        const top1 = values1.slice(0, displayCount);
        const top2 = values2.slice(0, displayCount);

        // Pad arrays
        while (top1.length < displayCount) top1.push({ value: 'N/A', count: 0 });
        while (top2.length < displayCount) top2.push({ value: 'N/A', count: 0 });

        // Destroy previous chart
        if (window.comparisonDistributionChart) {
            window.comparisonDistributionChart.destroy();
        }

        // Format labels
        const labels = top1.map((v, i) => {
            if (!v.value) return 'Item ' + (i + 1);
            let val = String(v.value).substring(0, 12);
            if (val.length === 12) val += '...';
            return val;
        });

        // Create chart configuration
        const chartConfig = {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: label1,
                        data: top1.map(v => parseInt(v.count) || 0),
                        backgroundColor: 'rgba(102, 126, 234, 0.85)',
                        borderColor: '#667eea',
                        borderWidth: 2,
                        borderRadius: 6,
                        hoverBackgroundColor: 'rgba(102, 126, 234, 1)',
                        hoverBorderColor: '#667eea',
                        hoverBorderWidth: 3
                    },
                    {
                        label: label2,
                        data: top2.map(v => parseInt(v.count) || 0),
                        backgroundColor: 'rgba(240, 147, 251, 0.85)',
                        borderColor: '#f093fb',
                        borderWidth: 2,
                        borderRadius: 6,
                        hoverBackgroundColor: 'rgba(240, 147, 251, 1)',
                        hoverBorderColor: '#f093fb',
                        hoverBorderWidth: 3
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                indexAxis: displayCount > 8 ? 'y' : 'x',
                plugins: { 
                    legend: { 
                        display: true, 
                        position: 'top',
                        labels: {
                            font: { size: 13, weight: 'bold' },
                            padding: 15,
                            usePointStyle: true,
                            pointStyle: 'rect'
                        }
                    },
                    tooltip: {
                        backgroundColor: 'rgba(0, 0, 0, 0.85)',
                        padding: 15,
                        cornerRadius: 8,
                        titleFont: { size: 13 },
                        bodyFont: { size: 12 },
                        displayColors: true,
                        callbacks: {
                            label: function(context) {
                                return context.dataset.label + ': ' + context.parsed.y + ' records';
                            }
                        }
                    }
                },
                scales: { 
                    y: { 
                        beginAtZero: true,
                        ticks: { font: { size: 11 } },
                        title: { display: true, text: 'Count', font: { size: 12, weight: 'bold' } }
                    },
                    x: {
                        ticks: { font: { size: 11 } }
                    }
                }
            }
        };

        // Render chart
        window.comparisonDistributionChart = new Chart(ctx, chartConfig);
        console.log('✓ Bar chart rendered with ' + displayCount + ' items');
    } catch (e) {
        console.error('Error rendering distribution chart:', e);
    }
}
```

#### Function 4: Build Line Chart
```javascript
function buildLineComparison(data1, data2, label1, label2) {
    try {
        const ctx = document.getElementById('comparisonDistributionChart');
        if (!ctx) {
            console.error('Line chart context not found');
            return;
        }

        const values1 = data1.data || [];
        const values2 = data2.data || [];
        
        if (!values1 || values1.length === 0 || !values2 || values2.length === 0) {
            console.error('Invalid data for line comparison');
            ctx.parentElement.innerHTML = 
                '<div style="text-align: center; padding: 40px; color: #999;">' +
                '<i class="fas fa-info-circle"></i> No data available</div>';
            return;
        }

        const displayCount = Math.min(Math.max(values1.length, values2.length), 12);
        const top1 = values1.slice(0, displayCount);
        const top2 = values2.slice(0, displayCount);

        while (top1.length < displayCount) top1.push({ value: 'N/A', count: 0 });
        while (top2.length < displayCount) top2.push({ value: 'N/A', count: 0 });

        if (window.comparisonDistributionChart) {
            window.comparisonDistributionChart.destroy();
        }

        const labels = top1.map((v, i) => {
            if (!v.value) return 'Item ' + (i + 1);
            let val = String(v.value).substring(0, 12);
            if (val.length === 12) val += '...';
            return val;
        });

        window.comparisonDistributionChart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: label1,
                        data: top1.map(v => parseInt(v.count) || 0),
                        borderColor: '#667eea',
                        backgroundColor: 'rgba(102, 126, 234, 0.15)',
                        borderWidth: 3,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 6,
                        pointBackgroundColor: '#667eea',
                        pointBorderColor: '#ffffff',
                        pointBorderWidth: 2,
                        pointHoverRadius: 8
                    },
                    {
                        label: label2,
                        data: top2.map(v => parseInt(v.count) || 0),
                        borderColor: '#f093fb',
                        backgroundColor: 'rgba(240, 147, 251, 0.15)',
                        borderWidth: 3,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 6,
                        pointBackgroundColor: '#f093fb',
                        pointBorderColor: '#ffffff',
                        pointBorderWidth: 2,
                        pointHoverRadius: 8
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: {
                    intersect: false,
                    mode: 'index'
                },
                plugins: { 
                    legend: { 
                        display: true, 
                        position: 'top',
                        labels: {
                            font: { size: 13, weight: 'bold' },
                            padding: 15,
                            usePointStyle: true,
                            pointStyle: 'circle'
                        }
                    },
                    tooltip: {
                        backgroundColor: 'rgba(0, 0, 0, 0.85)',
                        padding: 15,
                        cornerRadius: 8,
                        titleFont: { size: 13 },
                        bodyFont: { size: 12 },
                        displayColors: true,
                        callbacks: {
                            label: function(context) {
                                return context.dataset.label + ': ' + context.parsed.y + ' records';
                            }
                        }
                    }
                },
                scales: { 
                    y: { 
                        beginAtZero: true,
                        ticks: { font: { size: 11 } },
                        title: { display: true, text: 'Count', font: { size: 12, weight: 'bold' } }
                    },
                    x: {
                        ticks: { font: { size: 11 } }
                    }
                }
            }
        });
        console.log('✓ Line chart rendered with ' + displayCount + ' items');
    } catch (e) {
        console.error('Error rendering line comparison chart:', e);
    }
}
```

---

## How It Works

### Flow Diagram
```
1. User selects Field 1 & Field 2
2. Clicks "Compare Fields" button
3. loadFieldComparison() fetches data
4. buildComparisonCharts() is called
   ├─ Stores data in window.comparisonChartData
   └─ Calls buildDistributionComparison() (default)
5. Bar chart appears
6. User clicks Line button
7. switchComparisonChart('line') executes
   └─ Calls buildLineComparison()
8. Line chart appears instantly
```

---

## Troubleshooting

### Chart Not Appearing
1. **Check Browser Console** (F12 → Console)
   - Look for any JavaScript errors
   - Check for log messages starting with ✓

2. **Verify Canvas Element**
   - Ensure `<canvas id="comparisonDistributionChart">` exists
   - Check parent container has height: 450px

3. **Check Data**
   - Both fields must have data
   - Data format: `[{value: "...", count: 123}, ...]`

4. **Verify Chart.js**
   - Must be loaded: `<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>`

### Chart Renders but Empty
- Check if data arrays have length > 0
- Verify API returns correct data format
- Look for "No data available" message

---

## Visual Features

✨ **Bar Chart**
- Grouped bars for easy comparison
- Auto-horizontal for many items
- Rounded corners and smooth shadows

✨ **Line Chart**
- Smooth curves with filled areas
- Large interactive points
- Better for trend visualization

✨ **Interactive**
- Hover effects on bars/lines
- Custom tooltips with labels
- Responsive to all screen sizes

---

## Browser Support
- Chrome/Edge: ✅ Full Support
- Firefox: ✅ Full Support
- Safari: ✅ Full Support
- IE 11: ❌ Not Supported

---

**Last Updated:** January 23, 2026
**Chart.js Version:** 3.9+
