/* sidebar.js - Dynamic role-wise menu and guards */
document.addEventListener("DOMContentLoaded", async function () {
    try {
        const sessionRes = await fetch('api.php?action=check_session', { cache: 'no-store' });
        const sessionData = await sessionRes.json();

        if (!sessionData.logged_in) {
            window.location.href = 'login.php';
            return;
        }

        const role = String(sessionData.role || '').trim();
        const roleLower = role.toLowerCase();
        const userName = sessionData.name || role || 'User';
        localStorage.setItem('user_role', role);

        let permissionMap = {};
        try {
            const permissionRes = await fetch('api.php?action=get_my_page_permissions', { cache: 'no-store' });
            const permissionData = await permissionRes.json();
            if (permissionData.success && permissionData.permissions && typeof permissionData.permissions === 'object') {
                permissionMap = permissionData.permissions;
            }
        } catch (permError) {
            console.warn('Permission map load failed. Using fallback rules.', permError);
        }

        const pageKeyFromFile = {
            'index.php': 'tools_dashboard',
            'admin_dashboard.php': 'admin_dashboard',
            'register_emp.php': 'register_emp',
            'admin_fields.php': 'admin_fields',
            'all_employees.php': 'all_employees',
            'search_emp.php': 'search_emp',
            'entry_log.php': 'entry_log',
            'view_log.php': 'view_log',
            'analytics.php': 'analytics',
            'analytics_advanced.php': 'analytics',
            'analytics_old_backup.php': 'analytics',
            'maintenance.php': 'maintenance'
        };

        const fileFromPageKey = {
            'tools_dashboard': 'index.php',
            'admin_dashboard': 'admin_dashboard.php',
            'register_emp': 'register_emp.php',
            'admin_fields': 'admin_fields.php',
            'all_employees': 'all_employees.php',
            'search_emp': 'search_emp.php',
            'entry_log': 'entry_log.php',
            'view_log': 'view_log.php',
            'analytics': 'analytics.php',
            'maintenance': 'maintenance.php'
        };

        const currentPage = window.location.pathname.split('/').pop();

        const hasPageAccess = (pageKey) => {
            if (!pageKey) return true;
            if (roleLower === 'administrator' || roleLower === 'admin') return true;

            if (Object.prototype.hasOwnProperty.call(permissionMap, pageKey)) {
                return Boolean(permissionMap[pageKey]);
            }

            // Legacy fallback if permission map is unavailable
            if (roleLower === 'controller' && ['register_emp', 'admin_fields', 'admin_dashboard'].includes(pageKey)) {
                return false;
            }
            if (roleLower === 'supervisor' && pageKey === 'admin_dashboard') {
                return false;
            }
            return true;
        };

        const getFirstAllowedPage = () => {
            const priority = [
                'tools_dashboard',
                'view_log',
                'entry_log',
                'analytics',
                'all_employees',
                'search_emp',
                'maintenance',
                'register_emp',
                'admin_fields'
            ];

            for (const pageKey of priority) {
                if (hasPageAccess(pageKey)) {
                    return fileFromPageKey[pageKey] || 'index.php';
                }
            }
            return 'index.php';
        };

        const currentPageKey = pageKeyFromFile[currentPage] || null;
        if (currentPageKey && !hasPageAccess(currentPageKey)) {
            window.location.href = `${getFirstAllowedPage()}?access_denied=1`;
            return;
        }

        let menuHtml = `
            <li class="menu-label">MAIN MENU</li>
            ${hasPageAccess('tools_dashboard') ? '<li><a href="index.php" id="link-index"><span class="icon"><i class="fas fa-house"></i></span> Dashboard</a></li>' : ''}
        `;

        const adminItems = [];
        if (hasPageAccess('admin_dashboard')) adminItems.push('<li><a href="admin_dashboard.php" id="link-admin-dash"><span class="icon"><i class="fas fa-gear"></i></span> Admin Dashboard</a></li>');
        if (hasPageAccess('register_emp')) adminItems.push('<li><a href="register_emp.php" id="link-register"><span class="icon"><i class="fas fa-user-plus"></i></span> Register Staff</a></li>');
        if (hasPageAccess('admin_fields')) adminItems.push('<li><a href="admin_fields.php" id="link-fields"><span class="icon"><i class="fas fa-screwdriver-wrench"></i></span> Manage Fields</a></li>');
        if (hasPageAccess('maintenance')) adminItems.push('<li><a href="maintenance.php" id="link-maintenance"><span class="icon"><i class="fas fa-toolbox"></i></span> Maintenance</a></li>');

        if (adminItems.length > 0) {
            menuHtml += `<li class="menu-label">ADMINISTRATION</li>${adminItems.join('')}`;
        }

        const operationItems = [];
        if (hasPageAccess('all_employees')) operationItems.push('<li><a href="all_employees.php" id="link-list"><span class="icon"><i class="fas fa-list-check"></i></span> Staff List</a></li>');
        if (hasPageAccess('search_emp')) operationItems.push('<li><a href="search_emp.php" id="link-search"><span class="icon"><i class="fas fa-magnifying-glass"></i></span> Search</a></li>');
        if (operationItems.length > 0) {
            menuHtml += `<li class="menu-label">OPERATIONS</li>${operationItems.join('')}`;
        }

        const productionItems = [];
        if (hasPageAccess('entry_log')) productionItems.push('<li><a href="entry_log.php" id="link-entry"><span class="icon"><i class="fas fa-pen"></i></span> New Entry</a></li>');
        if (hasPageAccess('view_log')) productionItems.push('<li><a href="view_log.php" id="link-view"><span class="icon"><i class="fas fa-chart-column"></i></span> View Reports</a></li>');
        if (productionItems.length > 0) {
            menuHtml += `<li class="menu-label">PRODUCTION LOGS</li>${productionItems.join('')}`;
        }

        const analyticsItems = [];
        if (hasPageAccess('analytics')) {
            analyticsItems.push('<li><a href="analytics.php" id="link-analytics"><span class="icon"><i class="fas fa-chart-line"></i></span> Analytics Dashboard</a></li>');
            analyticsItems.push('<li><a href="analytics_advanced.php" id="link-analytics-adv"><span class="icon"><i class="fas fa-chart-pie"></i></span> Advanced Analytics</a></li>');
        }
        if (analyticsItems.length > 0) {
            menuHtml += `<li class="menu-label">ANALYTICS</li>${analyticsItems.join('')}`;
        }

        const sidebarHTML = `
            <button class="mobile-toggle-btn" onclick="toggleSidebar()">&#9776;</button>
            <div class="sidebar-overlay" onclick="toggleSidebar()"></div>
            <div class="sidebar" id="mySidebar">
                <div class="sidebar-logo">
                    <img src="logo.png" alt="Logo" onerror="this.style.display='none'" style="height:35px;">
                    <div class="logo-text">
                        <h2 style="font-size:18px;">MEKTRON</h2>
                        <span class="user-badge">${userName}</span>
                    </div>
                </div>
                <ul class="sidebar-menu">
                    ${menuHtml}
                </ul>
                <div class="sidebar-footer">
                    <button onclick="logoutUser()" class="logout-btn">
                        <span class="icon"><i class="fas fa-right-from-bracket"></i></span> Logout
                    </button>
                </div>
            </div>
        `;

        document.body.insertAdjacentHTML('afterbegin', sidebarHTML);

        const linkMap = {
            'index.php': 'link-index',
            '': 'link-index',
            'admin_dashboard.php': 'link-admin-dash',
            'register_emp.php': 'link-register',
            'admin_fields.php': 'link-fields',
            'maintenance.php': 'link-maintenance',
            'all_employees.php': 'link-list',
            'search_emp.php': 'link-search',
            'entry_log.php': 'link-entry',
            'view_log.php': 'link-view',
            'analytics.php': 'link-analytics',
            'analytics_advanced.php': 'link-analytics-adv'
        };

        if (linkMap[currentPage]) {
            document.getElementById(linkMap[currentPage])?.classList.add('active');
        }
    } catch (error) {
        console.error('Sidebar initialization failed:', error);
    }
});

// --- TOGGLE FUNCTION ---
function toggleSidebar() {
    const sidebar = document.getElementById('mySidebar');
    const overlay = document.querySelector('.sidebar-overlay');

    if (sidebar) {
        sidebar.classList.toggle('active');
        overlay.classList.toggle('active');
    }
}

function logoutUser() {
    Swal.fire({
        title: 'Logout Confirmation',
        text: 'Are you sure you want to logout?',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonColor: '#013B3B',
        cancelButtonColor: '#6c757d',
        confirmButtonText: 'Yes, Logout',
        cancelButtonText: 'Cancel',
        backdrop: true,
        allowOutsideClick: false
    }).then((result) => {
        if (result.isConfirmed) {
            fetch('api.php?action=logout').then(() => {
                Swal.fire({
                    title: 'Logged Out',
                    text: 'You have been successfully logged out',
                    icon: 'success',
                    confirmButtonColor: '#013B3B',
                    allowOutsideClick: false
                }).then(() => {
                    window.location.href = 'login.php';
                });
            });
        }
    });
}

document.addEventListener("DOMContentLoaded", () => {
    fetchAlerts();
});

async function fetchAlerts() {
    try {
        const res = await fetch('notifications.php?action=get_notifications');
        const data = await res.json();

        if (data.success && data.total_count > 0) {
            const badge = document.getElementById('notificationBadgeHeader');
            const btn = document.querySelector('.notification-btn');

            if (btn) {
                let badgeEl = btn.querySelector('.notification-badge');
                if (!badgeEl) {
                    badgeEl = document.createElement('span');
                    badgeEl.className = 'notification-badge';
                    btn.appendChild(badgeEl);
                }
                badgeEl.textContent = data.total_count > 99 ? '99+' : data.total_count;
                badgeEl.style.display = 'inline-flex';
            }

            if (badge) {
                if (data.critical_count > 0) {
                    badge.textContent = `Critical: ${data.critical_count}`;
                } else if (data.warning_count > 0) {
                    badge.textContent = `Warning: ${data.warning_count}`;
                } else {
                    badge.textContent = `Alerts: ${data.total_count}`;
                }
            }

            window.currentAlerts = data.notifications;
        }
    } catch (e) {
        console.error('Error fetching alerts:', e);
    }
}

function showAlertDetails() {
    if (!window.currentAlerts || window.currentAlerts.length === 0) {
        Swal.fire('No Alerts', 'System is running smoothly.', 'success');
        return;
    }

    let htmlList = '<div style="text-align:left; max-height:300px; overflow-y:auto;">';
    window.currentAlerts.forEach((a) => {
        const priority = (a.priority || a.severity || '').toLowerCase();
        const color = priority === 'critical' ? '#dd4b39' : '#f39c12';
        const msg = a.msg || a.message || 'Alert';
        htmlList += `<div style="padding:10px; border-bottom:1px solid #eee; border-left:4px solid ${color}; margin-bottom:5px;">${msg}</div>`;
    });
    htmlList += '</div>';

    Swal.fire({
        title: 'Active Alerts',
        html: htmlList,
        showCloseButton: true,
        showConfirmButton: false
    });
}
