/* Mektron Production Log - Master Script 
   Unified Logic for Entry, View, Register, and Search 
*/

// ===== MODERN CENTERED MODAL SYSTEM =====
function showModal(title, message = '', type = 'alert', callback = null) {
    let overlay = document.getElementById('modalOverlay');
    if (!overlay) {
        overlay = document.createElement('div');
        overlay.id = 'modalOverlay';
        overlay.className = 'modal-overlay';
        document.body.appendChild(overlay);
        
        if (!document.querySelector('style[data-modals]')) {
            const style = document.createElement('style');
            style.setAttribute('data-modals', 'true');
            style.textContent = `
                .modal-overlay {
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0, 0, 0, 0.55);
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    padding: 20px;
                    z-index: 10000;
                    animation: fadeIn 0.3s cubic-bezier(0.4, 0, 0.2, 1);
                    backdrop-filter: blur(2px);
                }
                @keyframes fadeIn { 
                    from { opacity: 0; } 
                    to { opacity: 1; } 
                }
                @keyframes zoomIn { 
                    from { 
                        transform: scale(0.85) translateY(-30px); 
                        opacity: 0; 
                    } 
                    to { 
                        transform: scale(1) translateY(0); 
                        opacity: 1; 
                    } 
                }
                .modal-content {
                    background: white;
                    border-radius: 16px;
                    box-shadow: 0 25px 80px rgba(0, 0, 0, 0.35);
                    max-width: 540px;
                    width: 100%;
                    max-height: 85vh;
                    padding: 0;
                    overflow: auto;
                    animation: zoomIn 0.35s cubic-bezier(0.34, 1.56, 0.64, 1);
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
                    display: flex;
                    flex-direction: column;
                }
                .modal-header {
                    padding: 32px 32px 24px;
                    border-bottom: 1px solid #f0f0f0;
                    display: flex;
                    align-items: flex-start;
                    gap: 16px;
                    flex-shrink: 0;
                }
                .modal-icon {
                    font-size: 28px;
                    flex-shrink: 0;
                    width: 44px;
                    height: 44px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    border-radius: 50%;
                    margin-top: 2px;
                }
                .modal-icon.success { 
                    color: #00a65a; 
                    background: rgba(0, 166, 90, 0.1);
                }
                .modal-icon.error { 
                    color: #dd4b39; 
                    background: rgba(221, 75, 57, 0.1);
                }
                .modal-icon.warning { 
                    color: #f39c12; 
                    background: rgba(243, 156, 18, 0.1);
                }
                .modal-icon.info { 
                    color: #3c8dbc; 
                    background: rgba(60, 141, 188, 0.1);
                }
                .modal-title {
                    font-size: 20px;
                    font-weight: 700;
                    color: #1a1a1a;
                    margin: 0;
                    line-height: 1.4;
                    flex: 1;
                }
                .modal-body {
                    padding: 24px 32px;
                    color: #555;
                    font-size: 14px;
                    line-height: 1.7;
                    flex: 1;
                    overflow-y: auto;
                }
                .modal-input {
                    width: 100%;
                    padding: 12px 14px;
                    border: 1.5px solid #e0e0e0;
                    border-radius: 8px;
                    font-size: 14px;
                    font-family: inherit;
                    box-sizing: border-box;
                    margin-top: 12px;
                    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
                    background: #fafafa;
                }
                .modal-input:focus {
                    outline: none;
                    border-color: #3c8dbc;
                    background: white;
                    box-shadow: 0 0 0 4px rgba(60, 141, 188, 0.12);
                }
                .modal-input::placeholder {
                    color: #aaa;
                }
                .modal-footer {
                    padding: 24px 32px;
                    border-top: 1px solid #f0f0f0;
                    display: flex;
                    gap: 12px;
                    justify-content: flex-end;
                    background: #fafafa;
                    flex-shrink: 0;
                }
                .modal-btn {
                    padding: 12px 32px;
                    border: none;
                    border-radius: 8px;
                    font-size: 14px;
                    font-weight: 600;
                    cursor: pointer;
                    transition: all 0.25s cubic-bezier(0.4, 0, 0.2, 1);
                    min-width: 110px;
                    display: inline-flex;
                    align-items: center;
                    justify-content: center;
                    white-space: nowrap;
                }
                .modal-btn-ok { 
                    background: linear-gradient(135deg, #00a65a 0%, #008c47 100%);
                    color: white;
                    box-shadow: 0 4px 15px rgba(0, 166, 90, 0.25);
                }
                .modal-btn-ok:hover { 
                    transform: translateY(-2px);
                    box-shadow: 0 6px 25px rgba(0, 166, 90, 0.35);
                }
                .modal-btn-ok:active {
                    transform: translateY(0);
                }
                .modal-btn-cancel { 
                    background: #f5f5f5;
                    color: #444;
                    border: 1px solid #e0e0e0;
                }
                .modal-btn-cancel:hover { 
                    background: #efefef;
                    border-color: #d0d0d0;
                    transform: translateY(-2px);
                }
                .modal-btn-cancel:active {
                    transform: translateY(0);
                }
                .modal-btn-confirm { 
                    background: linear-gradient(135deg, #dd4b39 0%, #c93222 100%);
                    color: white;
                    box-shadow: 0 4px 15px rgba(221, 75, 57, 0.25);
                }
                .modal-btn-confirm:hover { 
                    transform: translateY(-2px);
                    box-shadow: 0 6px 25px rgba(221, 75, 57, 0.35);
                }
                .modal-btn-confirm:active {
                    transform: translateY(0);
                }
                @media (max-width: 600px) {
                    .modal-overlay {
                        padding: 16px;
                    }
                    .modal-content {
                        max-width: 100%;
                        max-height: 90vh;
                        border-radius: 14px;
                    }
                    .modal-header {
                        padding: 24px 20px 18px;
                        gap: 12px;
                    }
                    .modal-icon {
                        width: 40px;
                        height: 40px;
                    }
                    .modal-title {
                        font-size: 18px;
                    }
                    .modal-body {
                        padding: 20px;
                    }
                    .modal-footer {
                        padding: 20px;
                    }
                    .modal-btn {
                        flex: 1;
                        min-width: auto;
                        padding: 11px 20px;
                    }
                }
            `;
            document.head.appendChild(style);
        }
    }
    
    const icons = {
        success: 'fas fa-check-circle',
        error: 'fas fa-exclamation-circle',
        warning: 'fas fa-warning',
        info: 'fas fa-info-circle'
    };
    
    const modal = document.createElement('div');
    modal.className = 'modal-content';
    
    let buttonsHTML = '';
    if (type === 'alert') {
        buttonsHTML = `<button class="modal-btn modal-btn-ok" onclick="this.closest('.modal-overlay').remove()">OK</button>`;
    } else if (type === 'confirm') {
        buttonsHTML = `<button class="modal-btn modal-btn-confirm" onclick="window.modalResult = true; this.closest('.modal-overlay').remove()">Confirm</button>
                       <button class="modal-btn modal-btn-cancel" onclick="window.modalResult = false; this.closest('.modal-overlay').remove()">Cancel</button>`;
    }
    
    modal.innerHTML = `
        <div class="modal-header">
            <div class="modal-icon ${type}"><i class="fas ${icons[type]}"></i></div>
            <h2 class="modal-title">${title}</h2>
        </div>
        <div class="modal-body">${message}</div>
        ${buttonsHTML ? `<div class="modal-footer">${buttonsHTML}</div>` : ''}
    `;
    
    overlay.innerHTML = '';
    overlay.appendChild(modal);
    overlay.onclick = (e) => { if (e.target === overlay) overlay.remove(); };
    
    if (type === 'confirm' && callback) {
        const checkInterval = setInterval(() => {
            if (overlay.parentElement === null) {
                clearInterval(checkInterval);
                callback(window.modalResult || false);
            }
        }, 100);
    }
}



// --- GLOBAL VARIABLES ---
let fieldConfig = []; // Stores the definition of dynamic fields
let logsCache = [];   // Stores fetched logs for filtering
let allEmployeesCacheSearch = []; // Stores employees for search

// ==========================================
// 1. CENTRAL INITIALIZATION
// ==========================================

document.addEventListener('DOMContentLoaded', function() {
    console.log("System Initializing...");

    // 1. Check which page we are on and Init accordingly
    if(document.getElementById('dynamicFormContainer') || document.getElementById('dynamicFieldsContainer')) {
        // ENTRY PAGE
        initEntryPage();
        loadEmployeeDropdown();
        document.getElementById('empIdSelect')?.addEventListener('change', fetchEmployeeName);
        document.getElementById('productionForm')?.addEventListener('submit', saveLog);
    } 
    else if(document.getElementById('logTableBody')) {
        // VIEW/REPORT PAGE
        initViewPage();
    }
    else if(document.getElementById('registerForm')) {
        // ADMIN REGISTER PAGE
        fetchNextID();
        document.getElementById('registerForm').addEventListener('submit', registerEmployee);
        if(document.getElementById('allEmpBody') || document.getElementById('employeeGrid')) {
            loadAllEmployees();
        }
    }
    else if(document.getElementById('searchIdInput')) {
        // SEARCH PAGE
        initSearchPage();
    }
});

// ==========================================
// 2. CORE: FETCH CONFIG & BUILDERS
// ==========================================

// --- ENTRY PAGE INITIALIZATION ---
function initEntryPage() {
    fetch('api.php?action=get_custom_fields&type=active')
    .then(res => res.json())
    .then(data => {
        fieldConfig = data;
        renderEntryForm(data); // Build the inputs
    })
    .catch(err => console.error("Config Load Error:", err));
}

// --- VIEW PAGE INITIALIZATION ---
function initViewPage() {
    fetch('api.php?action=get_report_fields')
    .then(res => res.json())
    .then(data => {
        fieldConfig = data;
        buildTableHeader(data); // 1. Build Headers
        loadOperatorFilter();   // 2. Load Filters
        loadAllLogs();          // 3. Load Data
    });
}

// ==========================================
// 3. DYNAMIC FORM BUILDER (Entry Page)
// ==========================================
function renderEntryForm(fields) {
    // Support both ID variations based on your HTML
    const container = document.getElementById('dynamicFormContainer') || document.getElementById('dynamicFieldsContainer');
    if(!container) return;
    
    container.innerHTML = ''; // Clear existing

    fields.forEach(f => {
        const div = document.createElement('div');
        
        // Apply Bootstrap classes if using Bootstrap, else standard
        div.className = 'form-group col-md-3 mb-3'; 
        div.style.padding = "10px"; // Fallback padding

        const label = document.createElement('label');
        label.className = 'form-label';
        label.textContent = f.field_label;
        label.style.fontWeight = "bold";

        let input;
        const nameLower = f.field_name.toLowerCase();

        // --- INTELLIGENT INPUT TYPE DETECTION ---

        // A. Dropdown for Shifts
        if (nameLower.includes('shift')) {
            input = document.createElement('select');
            input.className = 'form-select form-control';
            ["Select Shift", "Shift A", "Shift B", "Shift C", "General"].forEach(opt => {
                const option = document.createElement('option');
                option.value = opt === "Select Shift" ? "" : opt;
                option.textContent = opt;
                input.appendChild(option);
            });
        }
        // B. Dropdown for Status
        else if (nameLower.includes('status')) {
            input = document.createElement('select');
            input.className = 'form-select form-control';
            ["Pending", "In Progress", "Completed", "Hold"].forEach(opt => {
                const option = document.createElement('option');
                option.value = opt;
                option.textContent = opt;
                input.appendChild(option);
            });
        }
        // C. Standard Inputs
        else {
            input = document.createElement('input');
            input.className = 'form-control';
            input.name = f.field_name;
            input.id = f.field_name; // Critical for calculations
            input.placeholder = `Enter ${f.field_label}`;

            if(nameLower.includes('date')) input.type = 'date';
            else if(nameLower.includes('time')) {
                input.type = 'time';
                input.onchange = calculateDuration; // Attach Event
            }
            else if(nameLower.includes('qty') || nameLower.includes('norms')) {
                input.type = 'number';
                input.step = "0.01";
                if(f.field_name === 'produced_qty' || f.field_name === 'norms') {
                    input.oninput = calculateProductivity; // Attach Event
                }
            }
            else {
                input.type = 'text';
            }
            
            // Read-only logic for calc fields
            if(nameLower === 'duration' || nameLower === 'productivity') {
                input.readOnly = true;
                input.style.backgroundColor = "#e9ecef";
            }
        }

        // Final Assembly
        input.name = f.field_name; // Ensure name is set for FormData
        div.appendChild(label);
        div.appendChild(input);
        container.appendChild(div);
    });
}

// ==========================================
// 4. CALCULATIONS
// ==========================================
function calculateDuration() {
    const s = document.getElementById('start_time')?.value; 
    const e = document.getElementById('end_time')?.value;   
    
    if(s && e) {
        let startTime = new Date("2000-01-01 " + s);
        let endTime = new Date("2000-01-01 " + e);
        let diff = (endTime - startTime) / 60000; // minutes

        if(diff < 0) diff += 1440; // Handle overnight shifts
        
        const hrs = Math.floor(diff/60);
        const mins = diff % 60;
        
        const durField = document.getElementById('duration');
        if(durField) durField.value = `${hrs}:${mins.toString().padStart(2,'0')}`;
    }
}

function calculateProductivity() {
    const p = parseFloat(document.getElementById('produced_qty')?.value) || 0;
    const n = parseFloat(document.getElementById('norms')?.value) || 0;
    const prodField = document.getElementById('productivity');
    
    if(n > 0 && prodField) {
        prodField.value = ((p / n) * 100).toFixed(2) + '%';
    }
}

// ==========================================
// 5. DATA SAVING (Entry Page)
// ==========================================
function saveLog(e) {
    e.preventDefault();
    const empIdSelect = document.getElementById('empIdSelect');
    if(!empIdSelect.value) { showModal('⚠ Required', 'Please select an Employee ID', 'alert'); return; }

    const formData = new FormData(document.getElementById('productionForm'));
    // Ensure manual fields are added if they are outside the form tag or special select2
    formData.append('empId', empIdSelect.value);
    formData.append('operator', document.getElementById('operatorName').value);

    fetch('api.php?action=save_log', { method: 'POST', body: formData })
    .then(res => res.json())
    .then(data => {
        if(data.success) {
            showModal('✓ Success', 'Production entry saved successfully!', 'alert');
            // Reset logic while keeping context
            const currId = empIdSelect.value;
            const currOp = document.getElementById('operatorName').value;
            document.getElementById('productionForm').reset();
            empIdSelect.value = currId;
            document.getElementById('operatorName').value = currOp;
        } else {
            showModal('✗ Error', 'Failed to save: ' + data.error, 'alert');
        }
    })
    .catch(err => console.error(err));
}

function loadEmployeeDropdown() {
    const select = document.getElementById('empIdSelect');
    if(!select) return;
    fetch('api.php?action=get_employees').then(r=>r.json()).then(data => {
        data.sort((a, b) => a.name.localeCompare(b.name));
        select.innerHTML = '<option value="">-- Select ID --</option>';
        data.forEach(emp => {
            const opt = document.createElement('option');
            opt.value = emp.emp_unique_id;
            opt.textContent = `${emp.name} (${emp.emp_unique_id})`;
            opt.dataset.name = emp.name;
            select.appendChild(opt);
        });
    });
}

function fetchEmployeeName() {
    const select = document.getElementById('empIdSelect');
    const nameInput = document.getElementById('operatorName');
    if(select.selectedIndex > 0) {
        nameInput.value = select.options[select.selectedIndex].dataset.name;
    }
}

// ==========================================
// 6. VIEW PAGE LOGIC (Headers & Data)
// ==========================================
function buildTableHeader(fields) {
    const thead = document.querySelector('#logTableHead');
    if (!thead) return;
    
    let html = `
        <th style="border:1px solid #333; padding:5px;">ID</th>
        <th style="border:1px solid #333; padding:5px;">Operator</th>
        <th style="border:1px solid #333; padding:5px;">Date</th>
        <th style="border:1px solid #333; padding:5px;">Start</th>
        <th style="border:1px solid #333; padding:5px;">End</th>
        <th style="border:1px solid #333; padding:5px;">Dur</th>
    `;
    
    // Dynamic Columns
    fields.forEach(f => {
        html += `<th style="border:1px solid #333; padding:5px; background:#f4f6f7;">${f.field_label}</th>`;
    });
    
    html += `<th style="border:1px solid #333; padding:5px;">Sign</th>`;
    thead.innerHTML = html;
}

function loadAllLogs() {
    fetch('api.php?action=get_logs')
    .then(res => res.json())
    .then(data => {
        logsCache = data;
        populateMonthFilter(data);
        filterData();
    });
}

function filterData() {
    const opVal = document.getElementById('operatorFilter').value;
    const monthVal = document.getElementById('monthFilter').value;
    const fromVal = document.getElementById('fromDateFilter').value;
    const toVal = document.getElementById('toDateFilter').value;

    let displayData = logsCache.filter(row => {
        // Operator Check
        let opMatch = (opVal === 'all') || (row.operator_name === opVal);
        
        // Date Logic
        let dateMatch = true;
        if(fromVal && row.log_date < fromVal) dateMatch = false;
        if(toVal && row.log_date > toVal) dateMatch = false;
        
        // Month Logic
        let monthMatch = (monthVal === 'all') || (row.log_month === monthVal);

        return opMatch && dateMatch && monthMatch;
    });

    // Update Header Display
    document.getElementById('dispOperator').textContent = opVal === 'all' ? "ALL" : opVal;
    
    renderTableRows(displayData);
}

function renderTableRows(data) {
    const tbody = document.getElementById('logTableBody');
    tbody.innerHTML = '';

    if(data.length === 0) {
        tbody.innerHTML = '<tr><td colspan="100%" style="text-align:center; padding:15px;">No Records Found</td></tr>';
        return;
    }

    data.forEach(row => {
        const tr = document.createElement('tr');
        
        // 1. Static Cols
        let rowHtml = `
            <td style="border:1px solid #999; padding:4px;">${row.id || ''}</td>
            <td style="border:1px solid #999; padding:4px;">${row.operator_name || ''}</td>
            <td style="border:1px solid #999; padding:4px;">${row.log_date || ''}</td>
            <td style="border:1px solid #999; padding:4px;">${row.start_time || ''}</td>
            <td style="border:1px solid #999; padding:4px;">${row.end_time || ''}</td>
            <td style="border:1px solid #999; padding:4px;">${row.duration || ''}</td>
        `;

        // 2. Dynamic Cols (Match Config Order)
        fieldConfig.forEach(f => {
            // Check if value exists in standard row OR custom_data
            let val = row[f.field_name];
            
            // If not found in top level, check custom_data
            if(val === undefined && row.custom_data) {
                try {
                    const parsed = JSON.parse(row.custom_data);
                    val = parsed[f.field_name];
                } catch(e) {}
            }
            rowHtml += `<td style="border:1px solid #999; padding:4px;">${val || '-'}</td>`;
        });

        // 3. Sign Col
        rowHtml += `<td style="border:1px solid #999;"></td>`;
        
        tr.innerHTML = rowHtml;
        tbody.appendChild(tr);
    });
}

// Helpers
function loadOperatorFilter() {
    const select = document.getElementById('operatorFilter');
    fetch('api.php?action=get_employees').then(r=>r.json()).then(data=>{
        data.sort((a,b)=>a.name.localeCompare(b.name));
        select.innerHTML = '<option value="all">-- Show All --</option>';
        data.forEach(e => {
            select.innerHTML += `<option value="${e.name}">${e.name}</option>`;
        });
    });
}

function populateMonthFilter(data) {
    const select = document.getElementById('monthFilter');
    const months = [...new Set(data.map(d => d.log_month))].filter(Boolean).sort().reverse();
    select.innerHTML = '<option value="all">-- All Months --</option>';
    months.forEach(m => {
        select.innerHTML += `<option value="${m}">${m}</option>`;
    });
}

function clearFilters() {
    document.getElementById('operatorFilter').value = 'all';
    document.getElementById('monthFilter').value = 'all';
    document.getElementById('fromDateFilter').value = '';
    document.getElementById('toDateFilter').value = '';
    filterData();
}

// ==========================================
// 7. ADMIN / REGISTER / SEARCH LOGIC
// ==========================================

function fetchNextID() {
    fetch('api.php?action=get_next_id').then(r=>r.json())
    .then(d => { if(d.next_id && document.getElementById('autoEmpId')) document.getElementById('autoEmpId').value = d.next_id; });
}

function registerEmployee(e) {
    e.preventDefault();
    const formData = new FormData(document.getElementById('registerForm')); // Use Form ID directly
    fetch('api.php?action=register_employee', { method: 'POST', body: formData })
    .then(res => res.json())
    .then(data => {
        if(data.success) {
            showModal('✓ Success', 'Employee registered successfully!', 'alert');
            document.getElementById('registerForm').reset();
            fetchNextID();
            if(document.getElementById('employeeGrid')) loadAllEmployees();
        } else {
            showModal('✗ Error', 'Registration failed: ' + data.error, 'alert');
        }
    });
}

/* --- UPDATED SCRIPT.JS --- */

// 1. ENTRY PAGE: Only load ACTIVE employees
// --- FIX: Only show ACTIVE employees in the dropdown ---
function loadEmployeeDropdown() {
    const select = document.getElementById('empIdSelect');
    if(!select) return;

    fetch('api.php?action=get_employees')
    .then(res => res.json())
    .then(data => {
        // 1. FILTERING LOGIC
        // We filter the list to keep ONLY people who are NOT 'Inactive'
        const activeStaff = data.filter(emp => {
            // If status is missing (null), assume they are Active.
            // Convert to lowercase to handle 'Inactive' or 'inactive'
            const status = emp.status ? emp.status.toLowerCase() : 'active';
            return status !== 'inactive';
        });

        // 2. Sort Alphabetically
        activeStaff.sort((a, b) => a.name.localeCompare(b.name));
        
        // 3. Build the Dropdown
        select.innerHTML = '<option value="">-- Select ID --</option>';
        activeStaff.forEach(emp => {
            const opt = document.createElement('option');
            opt.value = emp.emp_unique_id;
            opt.textContent = `${emp.name} (${emp.emp_unique_id})`;
            opt.dataset.name = emp.name;
            select.appendChild(opt);
        });
        
        console.log("Dropdown loaded. Total:", data.length, "Active:", activeStaff.length);
    })
    .catch(err => console.error("Dropdown Error:", err));
}

// 2. VIEW PAGE: Show Status Buttons
function loadAllEmployees(roleOverride = null) {
    const grid = document.getElementById('employeeGrid');
    if(!grid) return;
    
    const resolvedRole = roleOverride !== null ? roleOverride : localStorage.getItem('user_role');
    const userRole = String(resolvedRole || '').trim().toLowerCase();
    const canManageEmployeeStatus = userRole === 'administrator' || userRole === 'admin' || userRole === 'supervisor';

    fetch('api.php?action=get_employees')
    .then(res => res.json())
    .then(data => {
        grid.innerHTML = '';
        if(data.length === 0) {
            grid.innerHTML = '<p style="grid-column:1/-1; text-align:center;">No employees found.</p>';
            return;
        }

        // Sort employees by emp_unique_id in ascending numerical order (ID 1, 2, 3, etc.)
        data.sort((a, b) => {
            // Extract numeric part from ID (e.g., "EMP001" -> 1, "EMP002" -> 2)
            const idA = parseInt(a.emp_unique_id.replace(/\D/g, '')) || 0;
            const idB = parseInt(b.emp_unique_id.replace(/\D/g, '')) || 0;
            return idA - idB;  // Ascending order: lowest ID first
        });

        data.forEach((emp, i) => {
            // Determine Status
            const isInactive = emp.status === 'Inactive';
            const statusLabel = isInactive ? 'Inactive' : 'Active';
            const statusClass = isInactive ? 'badge-red' : 'badge-green';
            
            // Determine Button Action (Flip the status)
            const nextStatus = isInactive ? 'Active' : 'Inactive';
            const btnText = isInactive ? 'Activate Profile' : 'Deactivate Profile';
            const btnClass = isInactive ? 'btn-activate' : 'btn-deactivate';

            // Status action visibility: Administrator/Admin/Supervisor can toggle
            let actionHtml = '';
            if(canManageEmployeeStatus) {
                actionHtml = `<button onclick="updateStatus('${emp.emp_unique_id}', '${nextStatus}')" class="${btnClass}">${btnText}</button>`;
            }
            // Controllers see no action button - just view the employee info

            const card = `
                <div class="emp-card" style="animation-delay: ${i * 0.05}s; border-left-color: ${isInactive ? '#e74c3c' : '#2ecc71'};">
                    <div class="emp-header">
                        <span class="emp-id">#${emp.emp_unique_id}</span>
                        <span class="${statusClass}">${statusLabel}</span>
                    </div>
                    
                    <h3 class="emp-name" style="${isInactive ? 'opacity:0.5' : ''}">${emp.name}</h3>
                    <div class="emp-designation">${emp.designation}</div>
                    
                    <div class="emp-details">
                        <span>📅 Joined:</span>
                        <strong>${emp.join_date}</strong>
                    </div>

                    ${actionHtml}
                </div>
            `;
            grid.innerHTML += card;
        });
    });
}

// 3. NEW FUNCTION: Toggle Status
function updateStatus(id, newStatus) {
    if(!confirm(`Are you sure you want to make this employee ${newStatus}?`)) return;

    const formData = new FormData();
    formData.append('id', id);
    formData.append('status', newStatus);

    fetch('api.php?action=toggle_status', { method: 'POST', body: formData })
    .then(res => res.json())
    .then(data => {
        if(data.success) {
            loadAllEmployees(); // Refresh Grid
        } else {
            showModal('✗ Error', 'Failed to update status: ' + data.error, 'alert');
        }
    });
}

function deleteEmployee(id) {
    if(confirm(`Delete ${id}?`)) {
        const fd = new FormData(); fd.append('id', id);
        fetch('api.php?action=delete_employee', {method:'POST', body:fd})
        .then(r=>r.json()).then(d=>{ if(d.success) loadAllEmployees(); });
    }
}

// --- SEARCH PAGE ---
function initSearchPage() {
    fetch('api.php?action=get_employees').then(r=>r.json()).then(d=>{ allEmployeesCacheSearch=d; });
}
function searchEmployeeDetails() {
    let term = document.getElementById('searchIdInput').value || document.getElementById('searchNameInput').value;
    if(!term) { showModal('⚠ Required', 'Please enter Employee ID or Name', 'alert'); return; }
    
    // Reuse logic: We can actually fetch log data for this specific user here
    // For brevity, assuming you have a backend endpoint for this
    console.log("Searching for:", term);
}
function registerEmployee(e) {
    e.preventDefault();
    
    // 1. Create FormData from the form
    // Since we added name="join_date" to HTML, this now automatically grabs the date!
    const formData = new FormData(document.getElementById('registerForm')); 

    // 2. FORCE DATA MAPPING (Safety Net)
    // We manually overwrite variables to guarantee they are correct
    
    // Fix ID (often disabled/readonly in HTML)
    const idInput = document.getElementById('autoEmpId');
    if(idInput) {
        formData.set('emp_unique_id', idInput.value); 
        formData.set('id', idInput.value);
    }

    // Fix Date (Ensure it is never null)
    const dateInput = document.getElementById('joinDate');
    if(dateInput && dateInput.value) {
        formData.set('join_date', dateInput.value); // Send exact DB column name
    } else {
        // If empty, auto-fill with Today's Date
        const today = new Date().toISOString().split('T')[0];
        formData.set('join_date', today);
    }

    // 3. Debugging (Optional: Check Console to see what is being sent)
    console.log("Submitting ID:", formData.get('emp_unique_id'));
    console.log("Submitting Date:", formData.get('join_date'));

    // 4. Send to Backend
    fetch('api.php?action=register_employee', { method: 'POST', body: formData })
    .then(res => res.json())
    .then(data => {
        if(data.success) {
            showModal('✓ Success', 'Employee registered successfully!', 'alert');
            document.getElementById('registerForm').reset();
            
            // Refill the ID and Date for the next person
            fetchNextID(); 
            document.getElementById('joinDate').valueAsDate = new Date();
            
            // Refresh table if visible
            if(typeof loadAllEmployees === 'function') loadAllEmployees();
        } else {
            showModal('✗ Error', 'Database error: ' + (data.error || 'Unknown'), 'alert');
        }
    })
    .catch(err => {
        console.error(err);
        showModal('✗ Error', 'Network error - please check your connection', 'alert');
    });
}

// ===== ADVANCED EMPLOYEE SEARCH & AUTO-FILL SYSTEM =====

// Global cache for employees
let employeeSearchCache = [];

// Initialize employee cache on page load
function initializeEmployeeCache() {
    console.log('Initializing employee cache...');
    fetch('api.php?action=get_employees')
        .then(r => {
            console.log('Employee API Status:', r.status);
            return r.json();
        })
        .then(data => {
            if (Array.isArray(data)) {
                employeeSearchCache = data;
                console.log('✓ Loaded ' + employeeSearchCache.length + ' employees');
            } else {
                console.warn('⚠ Invalid employee data format');
                employeeSearchCache = [];
            }
        })
        .catch(err => {
            console.error('❌ Failed to load employees:', err);
            employeeSearchCache = [];
        });
}

// Auto-fill employee name from ID
function autoFillEmployeeName(inputId = 'searchIdInput', outputId = 'searchNameInput') {
    const idInput = document.getElementById(inputId);
    const nameInput = document.getElementById(outputId);
    
    if (!idInput || !nameInput) return;
    
    const idValue = idInput.value.trim().toLowerCase();
    if (!idValue) {
        nameInput.value = '';
        return;
    }

    if (!employeeSearchCache || employeeSearchCache.length === 0) {
        console.warn('Employee cache is empty');
        return;
    }

    const found = employeeSearchCache.find(emp => {
        const empId = (emp.emp_unique_id || '').toString().toLowerCase();
        const empNum = (emp.id || '').toString();
        return empId === idValue || empNum === idValue;
    });

    if (found && found.name) {
        nameInput.value = found.name;
        console.log('✓ Found employee:', found.name);
        
        // Auto-fetch employee data after name is filled
        setTimeout(() => fetchEmployeeReport(idValue), 300);
    } else {
        nameInput.value = '';
        console.warn('No employee found with ID:', idValue);
    }
}

// Auto-fill employee ID from name
function autoFillEmployeeId(inputName = 'searchNameInput', outputId = 'searchIdInput') {
    const nameInput = document.getElementById(inputName);
    const idInput = document.getElementById(outputId);
    
    if (!nameInput || !idInput) return;
    
    const nameValue = nameInput.value.trim().toLowerCase();
    if (!nameValue) {
        idInput.value = '';
        return;
    }

    if (!employeeSearchCache || employeeSearchCache.length === 0) {
        console.warn('Employee cache is empty');
        return;
    }

    const found = employeeSearchCache.find(emp => {
        const empName = (emp.name || '').toString().toLowerCase();
        return empName === nameValue;
    });

    if (found && found.emp_unique_id) {
        idInput.value = found.emp_unique_id;
        console.log('✓ Found employee ID:', found.emp_unique_id);
        
        // Auto-fetch employee data after ID is filled
        setTimeout(() => fetchEmployeeReport(found.emp_unique_id), 300);
    } else {
        idInput.value = '';
        console.warn('No employee found with name:', nameValue);
    }
}

// Fetch and display complete employee report
function fetchEmployeeReport(term) {
    if (!term) {
        console.warn('No search term provided');
        return;
    }

    console.log('Fetching report for:', term);
    
    // Show loading state
    const resultDiv = document.getElementById('searchResult');
    const spinner = document.getElementById('loadingSpinner');
    
    if (resultDiv) resultDiv.style.display = 'none';
    if (spinner) spinner.style.display = 'block';

    fetch(`api.php?action=search_employee_full&term=${encodeURIComponent(term)}`)
        .then(res => {
            if (!res.ok) throw new Error('HTTP Error: ' + res.status);
            return res.json();
        })
        .then(data => {
            if (spinner) spinner.style.display = 'none';

            console.log('Report Data:', data);

            if (!data || !data.profile) {
                Swal.fire({
                    icon: 'error',
                    title: 'Not Found',
                    text: 'Employee record not found.',
                    confirmButtonColor: '#e74c3c'
                });
                return;
            }

            // Display employee profile
            const p = data.profile;
            if (document.getElementById('resId')) {
                document.getElementById('resId').textContent = p.emp_unique_id || p.id || 'N/A';
            }
            if (document.getElementById('resName')) {
                document.getElementById('resName').textContent = p.name || 'N/A';
            }
            if (document.getElementById('resRole')) {
                document.getElementById('resRole').textContent = p.designation || 'Operator';
            }
            if (document.getElementById('resDate')) {
                document.getElementById('resDate').textContent = p.join_date || 'N/A';
            }

            // Display work logs in table
            displayEmployeeLogs(data.logs || []);

            // Display employee detailed analytics - 3 CHARTS ONLY
            if (p.name) {
                fetchEmployeeDetailedAnalytics(p.name, function(analyticsData) {
                    renderEmployeeLast7DaysChart('employeeLast7DaysChart', analyticsData);
                    renderEmployeeEfficiencyChart('employeeEfficiencyChart', analyticsData);
                    renderEmployeeProductivityChart('employeeProductivityChart', analyticsData);
                });
            }

            // Show results
            if (resultDiv) resultDiv.style.display = 'block';

            // Removed success notification - no longer distracting
        })
        .catch(err => {
            if (spinner) spinner.style.display = 'none';
            console.error('Report Fetch Error:', err);

            Swal.fire({
                icon: 'error',
                title: 'Error',
                text: 'Failed to fetch employee data. ' + err.message,
                confirmButtonColor: '#e74c3c'
            });
        });
}

// Display employee work logs in a table
function displayEmployeeLogs(logs) {
    const tbody = document.getElementById('searchTableBody');
    const thead = document.getElementById('searchTableHead');

    if (!tbody) return;

    // Render headers - DYNAMIC based on actual data columns
    if (thead && logs && logs.length > 0) {
        // Get all available columns from the first log record
        let columns = Object.keys(logs[0]);
        
        // Filter out system columns (id, emp_id, operator_name is kept)
        const excludeColumns = ['id', 'emp_id'];
        const filteredColumns = columns.filter(col => !excludeColumns.includes(col));
        
        // Reorder columns to put important ones first
        const priorityColumns = ['log_date', 'operator_name', 'start_time', 'end_time', 'duration'];
        const reorderedColumns = priorityColumns.filter(col => filteredColumns.includes(col))
            .concat(filteredColumns.filter(col => !priorityColumns.includes(col)));

        let headerHtml = '<tr>';
        reorderedColumns.forEach(col => {
            // Format column name for display
            const displayName = col
                .replace(/_/g, ' ')
                .split(' ')
                .map(word => word.charAt(0).toUpperCase() + word.slice(1))
                .join(' ');
            
            headerHtml += `<th>${displayName}</th>`;
        });
        headerHtml += '</tr>';
        thead.innerHTML = headerHtml;

        console.log('Dynamic columns detected:', reorderedColumns);
    }

    // Clear body
    tbody.innerHTML = '';

    if (!logs || logs.length === 0) {
        tbody.innerHTML = `<tr><td colspan="20" style="text-align:center; padding:20px; color:#999;">No work logs found.</td></tr>`;
        return;
    }

    console.log('Rendering ' + logs.length + ' log entries with all available fields');

    logs.forEach((log, idx) => {
        // Get columns from first log
        let columns = Object.keys(logs[0]);
        
        // Filter out system columns
        const excludeColumns = ['id', 'emp_id'];
        const filteredColumns = columns.filter(col => !excludeColumns.includes(col));
        
        // Reorder columns to put important ones first
        const priorityColumns = ['log_date', 'operator_name', 'start_time', 'end_time', 'duration'];
        const reorderedColumns = priorityColumns.filter(col => filteredColumns.includes(col))
            .concat(filteredColumns.filter(col => !priorityColumns.includes(col)));

        const row = document.createElement('tr');
        let cellHtml = '';

        reorderedColumns.forEach(colName => {
            let value = log[colName];
            
            // Handle null/undefined values
            if (value === null || value === undefined || value === '') {
                value = '-';
            }
            
            // Format date column
            if (colName === 'log_date' && value !== '-') {
                try {
                    const dateObj = new Date(value);
                    if (!isNaN(dateObj)) {
                        value = dateObj.toISOString().split('T')[0];
                    }
                } catch(e) {
                    // Keep original value if parsing fails
                }
            }

            cellHtml += `<td>${value}</td>`;
        });

        row.innerHTML = cellHtml;
        tbody.appendChild(row);
    });
}

// Perform search with validation
function performEmployeeSearch() {
    const idTerm = document.getElementById('searchIdInput') ? document.getElementById('searchIdInput').value.trim() : '';
    const nameTerm = document.getElementById('searchNameInput') ? document.getElementById('searchNameInput').value.trim() : '';
    const term = idTerm || nameTerm;

    if (!term) {
        Swal.fire({
            icon: 'warning',
            title: 'Required Field',
            text: 'Please enter Employee ID or Name',
            confirmButtonColor: '#3498db'
        });
        return;
    }

    console.log('Performing search for:', term);
    fetchEmployeeReport(term);
}

// ===== ADVANCED ANALYTICS FUNCTIONS =====

// FETCH EMPLOYEES FOR DROPDOWN
function fetchEmployeesForDropdown(callback) {
    fetch('api.php?action=get_employees')
        .then(res => res.json())
        .then(data => {
            if (typeof callback === 'function') callback(Array.isArray(data) ? data : []);
        })
        .catch(err => {
            console.error('Employee fetch error:', err);
            if (typeof callback === 'function') callback([]);
        });
}

// POPULATE EMPLOYEE DROPDOWN
function populateEmployeeDropdown() {
    fetchEmployeesForDropdown(function(employees) {
        const select = document.getElementById('employeeSelect');
        if (!select) return;
        
        select.innerHTML = '<option value="">-- All Employees --</option>';
        employees.forEach(emp => {
            const option = document.createElement('option');
            option.value = emp.name || emp.emp_unique_id || '';
            option.textContent = emp.name || emp.emp_unique_id || 'Unknown';
            select.appendChild(option);
        });
    });
}

// POPULATE FIELD DROPDOWN
function populateFieldDropdown() {
    const fieldSelect = document.getElementById('fieldSelect');
    if (!fieldSelect) return;
    
    // Common field names to populate
    const fields = [
        'wo_no',
        'mc_no', 
        'tool_no',
        'part_name',
        'operation',
        'category'
    ];
    
    fieldSelect.innerHTML = '<option value="">-- All Fields --</option>';
    fields.forEach(field => {
        const option = document.createElement('option');
        option.value = field;
        option.textContent = field.replace(/_/g, ' ').toUpperCase();
        fieldSelect.appendChild(option);
    });
}

// APPLY DATE FILTER
function applyDateFilter() {
    const startDate = document.getElementById('startDate') ? document.getElementById('startDate').value : '';
    const endDate = document.getElementById('endDate') ? document.getElementById('endDate').value : '';
    
    if (startDate && endDate) {
        console.log('Applying date filter:', startDate, 'to', endDate);
        applyAllFilters();
    }
}

// ANALYZE FIELD DATA
function analyzeFieldData() {
    const fieldName = document.getElementById('fieldSelect') ? document.getElementById('fieldSelect').value : '';
    if (fieldName) {
        console.log('Analyzing field:', fieldName);
        applyAllFilters();
    }
}

// ANALYZE EMPLOYEE DATA
function analyzeEmployeeData() {
    const employeeName = document.getElementById('employeeSelect') ? document.getElementById('employeeSelect').value : '';
    if (employeeName) {
        console.log('Analyzing employee:', employeeName);
        applyAllFilters();
    }
}

// APPLY ALL FILTERS TO ANALYTICS
function applyAllFilters() {
    const startDate = document.getElementById('startDate') ? document.getElementById('startDate').value : '';
    const endDate = document.getElementById('endDate') ? document.getElementById('endDate').value : '';
    const fieldName = document.getElementById('fieldSelect') ? document.getElementById('fieldSelect').value : '';
    const employeeName = document.getElementById('employeeSelect') ? document.getElementById('employeeSelect').value : '';
    
    // Fetch filtered analytics
    fetchFilteredDayWiseAnalytics(startDate, endDate, function(data) {
        renderDayWiseChart('dayWiseChartContainer', data);
    });
    
    if (!fieldName) {
        fetchFieldWiseAnalytics(function(data) {
            renderFieldWiseChart('fieldWiseChartContainer', data);
        });
    } else {
        console.log('Field-wise filtering by:', fieldName);
    }
    
    if (!employeeName) {
        fetchEmployeeWiseAnalytics(function(data) {
            renderEmployeeWiseTable('employeeAnalyticsTable', data);
        });
    } else {
        fetchEmployeeDetailedAnalytics(employeeName, function(data) {
            renderEmployeeDetailedAnalyticsChart('employeeDetailAnalyticsContainer', data);
        });
    }
}

// FETCH FILTERED DAY-WISE ANALYTICS
function fetchFilteredDayWiseAnalytics(startDate, endDate, callback) {
    let url = 'api.php?action=get_daywise_analytics';
    if (startDate) url += '&start_date=' + encodeURIComponent(startDate);
    if (endDate) url += '&end_date=' + encodeURIComponent(endDate);
    
    fetch(url)
        .then(res => res.json())
        .then(data => {
            if (typeof callback === 'function') callback(data);
        })
        .catch(err => {
            console.error('Filtered analytics error:', err);
            if (typeof callback === 'function') callback([]);
        });
}

// RESET ALL FILTERS
function resetAllFilters() {
    if (document.getElementById('startDate')) document.getElementById('startDate').value = '';
    if (document.getElementById('endDate')) document.getElementById('endDate').value = '';
    if (document.getElementById('fieldSelect')) document.getElementById('fieldSelect').value = '';
    if (document.getElementById('employeeSelect')) document.getElementById('employeeSelect').value = '';
    
    // Reload all analytics without filters
    fetchDayWiseAnalytics(function(data) {
        renderDayWiseChart('dayWiseChartContainer', data);
    });
    
    fetchFieldWiseAnalytics(function(data) {
        renderFieldWiseChart('fieldWiseChartContainer', data);
    });
    
    fetchEmployeeWiseAnalytics(function(data) {
        renderEmployeeWiseTable('employeeAnalyticsTable', data);
    });
    
    console.log('✓ All filters reset');
}

// FETCH DAY-WISE PRODUCTION ANALYTICS

// FIELD-WISE PRODUCTION ANALYTICS
function fetchFieldWiseAnalytics(callback) {
    fetch('api.php?action=get_fieldwise_analytics')
        .then(res => res.json())
        .then(data => {
            if (typeof callback === 'function') callback(data);
        })
        .catch(err => {
            console.error('Field-wise analytics error:', err);
            if (typeof callback === 'function') callback([]);
        });
}

// EMPLOYEE-WISE ANALYTICS
function fetchEmployeeWiseAnalytics(callback) {
    fetch('api.php?action=get_employeewise_analytics')
        .then(res => res.json())
        .then(data => {
            if (typeof callback === 'function') callback(data);
        })
        .catch(err => {
            console.error('Employee-wise analytics error:', err);
            if (typeof callback === 'function') callback([]);
        });
}

// TOOL-WISE WORKER ANALYTICS
function fetchToolWorkerAnalytics(toolName, callback) {
    fetch(`api.php?action=get_tool_worker_analytics&tool=${encodeURIComponent(toolName)}`)
        .then(res => res.json())
        .then(data => {
            if (typeof callback === 'function') callback(data);
        })
        .catch(err => {
            console.error('Tool-worker analytics error:', err);
            if (typeof callback === 'function') callback([]);
        });
}

// EMPLOYEE DETAILED ANALYTICS (for search page)
function fetchEmployeeDetailedAnalytics(employeeName, callback) {
    fetch(`api.php?action=get_employee_full_analytics&employee=${encodeURIComponent(employeeName)}`)
        .then(res => res.json())
        .then(data => {
            if (typeof callback === 'function') callback(data);
        })
        .catch(err => {
            console.error('Employee detailed analytics error:', err);
            if (typeof callback === 'function') callback([]);
        });
}

// RENDER DAY-WISE BREAKDOWN TABLE (DETAILED)
function renderDayWiseBreakdownTable(containerId = 'dayWiseBreakdownTable', data = []) {
    if (!data || data.length === 0) {
        console.warn('No data for day-wise breakdown');
        return;
    }

    const container = document.getElementById(containerId);
    if (!container) {
        console.warn(`Container ${containerId} not found`);
        return;
    }

    let html = `
    <table class="analytics-table">
        <thead>
            <tr>
                <th>Date</th>
                <th>Total Entries</th>
                <th>Plan Qty</th>
                <th>Produced Qty</th>
                <th>Accepted Qty</th>
                <th>NC Qty</th>
                <th>Quality %</th>
                <th>Efficiency %</th>
            </tr>
        </thead>
        <tbody>
    `;

    data.slice(0, 30).forEach(entry => {
        const planQty = parseFloat(entry.total_plan_qty) || 0;
        const producedQty = parseFloat(entry.total_produced_qty) || 0;
        const acceptedQty = parseFloat(entry.total_accepted_qty) || 0;
        const ncQty = parseFloat(entry.nc_qty) || 0;
        const entries = entry.total_entries || 0;
        
        const quality = producedQty > 0 ? ((acceptedQty / producedQty) * 100).toFixed(2) : 0;
        const efficiency = planQty > 0 ? ((producedQty / planQty) * 100).toFixed(2) : 0;

        html += `
            <tr>
                <td><strong>${entry.date || '-'}</strong></td>
                <td>${entries}</td>
                <td>${planQty.toFixed(2)}</td>
                <td>${producedQty.toFixed(2)}</td>
                <td>${acceptedQty.toFixed(2)}</td>
                <td style="color: #e74c3c;"><strong>${ncQty.toFixed(2)}</strong></td>
                <td>${quality}%</td>
                <td>${efficiency}%</td>
            </tr>
        `;
    });

    html += `
        </tbody>
    </table>
    `;

    container.innerHTML = html;
    console.log('✓ Day-wise breakdown table rendered');
}

// POPULATE TOOL DROPDOWN - Fetch all tools from database
function populateToolDropdown() {
    const toolSelect = document.getElementById('toolSelect');
    if (!toolSelect) return;
    
    fetch('api.php?action=get_all_tools')
        .then(res => res.json())
        .then(data => {
            toolSelect.innerHTML = '<option value="">-- All Tools --</option>';
            if (Array.isArray(data)) {
                data.forEach(tool => {
                    const option = document.createElement('option');
                    option.value = tool.tool_no || tool.name || '';
                    option.textContent = tool.tool_no || tool.name || 'Unknown';
                    toolSelect.appendChild(option);
                });
            }
        })
        .catch(err => {
            console.error('Error fetching tools:', err);
            toolSelect.innerHTML = '<option value="">-- All Tools --</option>';
        });
}

// ANALYZE TOOL DATA
function analyzeToolData() {
    const toolName = document.getElementById('toolSelect') ? document.getElementById('toolSelect').value : '';
    if (toolName) {
        console.log('Analyzing tool:', toolName);
        // Fetch tool-wise data and render
        fetchToolWiseAnalytics(toolName, function(data) {
            renderToolWiseChart('toolWiseChartContainer', data);
            renderToolWiseBreakdownTable('toolWiseBreakdownTable', data);
        });
    } else {
        // Show all tools
        fetchToolWiseAnalytics('', function(data) {
            renderToolWiseChart('toolWiseChartContainer', data);
            renderToolWiseBreakdownTable('toolWiseBreakdownTable', data);
        });
    }
}

// FETCH TOOL-WISE PRODUCTION ANALYTICS
function fetchToolWiseAnalytics(toolName, callback) {
    const url = toolName ? `api.php?action=get_tool_production_analytics&tool_no=${encodeURIComponent(toolName)}` : 'api.php?action=get_all_tool_production_analytics';
    fetch(url)
        .then(res => res.json())
        .then(data => {
            if (typeof callback === 'function') callback(data);
        })
        .catch(err => {
            console.error('Tool-wise analytics error:', err);
            if (typeof callback === 'function') callback([]);
        });
}

// RENDER TOOL-WISE BREAKDOWN TABLE
function renderToolWiseBreakdownTable(containerId = 'toolWiseBreakdownTable', data = []) {
    if (!data || data.length === 0) {
        const container = document.getElementById(containerId);
        if (container) container.innerHTML = '<p class="no-data">No tool-wise data available</p>';
        return;
    }

    const container = document.getElementById(containerId);
    if (!container) {
        console.warn(`Container ${containerId} not found`);
        return;
    }

    let html = `
    <table class="analytics-table">
        <thead>
            <tr>
                <th>Tool No</th>
                <th>Total Entries</th>
                <th>Plan Qty</th>
                <th>Produced Qty</th>
                <th>Accepted Qty</th>
                <th style="color: #dc3545;">NC Qty</th>
                <th>Quality %</th>
                <th>Efficiency %</th>
            </tr>
        </thead>
        <tbody>
    `;

    data.slice(0, 30).forEach(row => {
        const planQty = parseFloat(row.total_plan_qty) || 0;
        const producedQty = parseFloat(row.total_produced_qty) || 0;
        const acceptedQty = parseFloat(row.total_accepted_qty) || 0;
        const ncQty = producedQty - acceptedQty;
        
        const qualityPercent = producedQty > 0 ? ((acceptedQty / producedQty) * 100).toFixed(2) : 0;
        const efficiencyPercent = planQty > 0 ? ((producedQty / planQty) * 100).toFixed(2) : 0;
        
        html += `
            <tr>
                <td><strong>${row.tool_no || row.field_name || '-'}</strong></td>
                <td>${row.total_entries || 0}</td>
                <td>${planQty.toFixed(0)}</td>
                <td>${producedQty.toFixed(0)}</td>
                <td>${acceptedQty.toFixed(0)}</td>
                <td style="color: #dc3545; font-weight: 600;">${ncQty.toFixed(0)}</td>
                <td>${qualityPercent}%</td>
                <td>${efficiencyPercent}%</td>
            </tr>
        `;
    });

    html += `
        </tbody>
    </table>
    `;

    container.innerHTML = html;
}

// RENDER TOOL-WISE CHART (4 BARS PER TOOL: PLAN, PRODUCED, ACCEPTED, NC)
function renderToolWiseChart(containerId = 'toolWiseChartContainer', data = []) {
    if (!data || data.length === 0) {
        console.warn('No data for tool-wise chart');
        return;
    }

    const tools = data.map(d => d.tool_no || d.field_name || 'Unknown').slice(0, 15);
    const plannedQty = data.map(d => parseFloat(d.total_plan_qty) || 0).slice(0, 15);
    const producedQty = data.map(d => parseFloat(d.total_produced_qty) || 0).slice(0, 15);
    const acceptedQty = data.map(d => parseFloat(d.total_accepted_qty) || 0).slice(0, 15);
    const ncQty = data.map((d, i) => (producedQty[i] - acceptedQty[i]) || 0).slice(0, 15);

    const canvas = document.getElementById(containerId);
    if (!canvas) {
        console.warn(`Chart canvas ${containerId} not found`);
        return;
    }

    // Destroy existing chart if it exists
    if (window.toolWiseChartInstance && typeof window.toolWiseChartInstance.destroy === 'function') {
        try {
            window.toolWiseChartInstance.destroy();
        } catch (e) {
            console.warn('Could not destroy previous chart');
        }
    }

    const ctx = canvas.getContext('2d');
    if (!ctx) {
        console.error('Could not get canvas context for tool-wise chart');
        return;
    }

    window.toolWiseChartInstance = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: tools,
            datasets: [
                {
                    label: 'Plan Qty',
                    data: plannedQty,
                    backgroundColor: '#3c8dbc',
                    borderColor: '#3c8dbc',
                    borderWidth: 0
                },
                {
                    label: 'Produced Qty',
                    data: producedQty,
                    backgroundColor: '#00a65a',
                    borderColor: '#00a65a',
                    borderWidth: 0
                },
                {
                    label: 'Accepted Qty',
                    data: acceptedQty,
                    backgroundColor: '#f39c12',
                    borderColor: '#f39c12',
                    borderWidth: 0
                },
                {
                    label: 'NC Qty',
                    data: ncQty,
                    backgroundColor: '#dc3545',
                    borderColor: '#dc3545',
                    borderWidth: 0
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: true,
                    position: 'top',
                    labels: {
                        font: { size: 12, weight: 'bold' },
                        padding: 15
                    }
                },
                title: {
                    display: true,
                    text: 'Tool-Wise Production Analysis [Plan > Produced > Accepted > NC]',
                    font: { size: 14, weight: 'bold' },
                    padding: 20
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    title: {
                        display: true,
                        text: 'Quantity'
                    }
                },
                x: {
                    title: {
                        display: true,
                        text: 'Tool'
                    }
                }
            }
        }
    });
}

// RENDER DAY-WISE ANALYTICS CHART

function renderDayWiseChart(containerId = 'dayWiseChartContainer', data = []) {
    if (!data || data.length === 0) {
        console.warn('No data for day-wise chart');
        return;
    }

    const dates = data.map(d => d.date).reverse();
    const plannedQty = data.map(d => parseFloat(d.total_plan_qty) || 0).reverse();
    const producedQty = data.map(d => parseFloat(d.total_produced_qty) || 0).reverse();
    const acceptedQty = data.map(d => parseFloat(d.total_accepted_qty) || 0).reverse();
    const ncQty = data.map(d => parseFloat(d.nc_qty) || 0).reverse();

    const canvas = document.getElementById(containerId);
    if (!canvas) {
        console.warn(`Chart canvas ${containerId} not found`);
        return;
    }

    // Destroy existing chart if it exists
    if (window.dayWiseChartInstance && typeof window.dayWiseChartInstance.destroy === 'function') {
        try {
            window.dayWiseChartInstance.destroy();
        } catch(e) {
            console.warn('Error destroying chart:', e);
        }
    }

    try {
        const ctx = canvas.getContext('2d');
        window.dayWiseChartInstance = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: dates,
                datasets: [
                    {
                        label: 'Plan Qty',
                        data: plannedQty,
                        backgroundColor: '#3498db',
                        borderColor: '#2980b9',
                        borderWidth: 1
                    },
                    {
                        label: 'Produced Qty',
                        data: producedQty,
                        backgroundColor: '#2ecc71',
                        borderColor: '#27ae60',
                        borderWidth: 1
                    },
                    {
                        label: 'Accepted Qty',
                        data: acceptedQty,
                        backgroundColor: '#f39c12',
                        borderColor: '#d68910',
                        borderWidth: 1
                    },
                    {
                        label: 'NC Qty',
                        data: ncQty,
                        backgroundColor: '#e74c3c',
                        borderColor: '#c0392b',
                        borderWidth: 1
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                        labels: { font: { size: 12 }, padding: 15 }
                    },
                    title: {
                        display: true,
                        text: 'Day-Wise Production Analytics',
                        font: { size: 14, weight: 'bold' }
                    }
                },
                scales: {
                    x: { stacked: false },
                    y: { stacked: false, beginAtZero: true }
                }
            }
        });
        console.log('✓ Day-wise chart rendered');
    } catch(err) {
        console.error('Error rendering day-wise chart:', err);
    }
}

// RENDER FIELD-WISE ANALYTICS CHART
function renderFieldWiseChart(containerId = 'fieldWiseChartContainer', data = []) {
    if (!data || data.length === 0) {
        console.warn('No data for field-wise chart');
        return;
    }

    const tools = data.map(d => d.field_name || 'Unknown').slice(0, 10);
    const plannedQty = data.map(d => parseFloat(d.total_plan_qty) || 0).slice(0, 10);
    const producedQty = data.map(d => parseFloat(d.total_produced_qty) || 0).slice(0, 10);
    const acceptedQty = data.map(d => parseFloat(d.total_accepted_qty) || 0).slice(0, 10);
    const ncQty = data.map(d => parseFloat(d.nc_qty) || 0).slice(0, 10);

    const canvas = document.getElementById(containerId);
    if (!canvas) {
        console.warn(`Chart canvas ${containerId} not found`);
        return;
    }

    if (window.fieldWiseChartInstance && typeof window.fieldWiseChartInstance.destroy === 'function') {
        try {
            window.fieldWiseChartInstance.destroy();
        } catch(e) {
            console.warn('Error destroying chart:', e);
        }
    }

    try {
        const ctx = canvas.getContext('2d');
        window.fieldWiseChartInstance = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: tools,
                datasets: [
                    {
                        label: 'Plan Qty',
                        data: plannedQty,
                        backgroundColor: '#3498db'
                    },
                    {
                        label: 'Produced Qty',
                        data: producedQty,
                        backgroundColor: '#2ecc71'
                    },
                    {
                        label: 'Accepted Qty',
                        data: acceptedQty,
                        backgroundColor: '#f39c12'
                    },
                    {
                        label: 'NC Qty',
                        data: ncQty,
                        backgroundColor: '#e74c3c'
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                indexAxis: 'y',
                plugins: {
                    legend: {
                        position: 'top',
                        labels: { font: { size: 11 }, padding: 10 }
                    },
                    title: {
                        display: true,
                        text: 'Field/Tool-Wise Production Analytics',
                        font: { size: 14, weight: 'bold' }
                    }
                },
                scales: {
                    x: { stacked: false, beginAtZero: true }
                }
            }
        });
        console.log('✓ Field-wise chart rendered');
    } catch(err) {
        console.error('Error rendering field-wise chart:', err);
    }
}

// RENDER EMPLOYEE-WISE ANALYTICS TABLE
function renderEmployeeWiseTable(containerId = 'employeeAnalyticsTable', data = []) {
    if (data.length === 0) {
        console.warn('No employee data');
        return;
    }

    const container = document.getElementById(containerId);
    if (!container) {
        console.warn(`Container ${containerId} not found`);
        return;
    }

    let html = `
    <table class="analytics-table">
        <thead>
            <tr>
                <th>Employee Name</th>
                <th>Days</th>
                <th>Tools</th>
                <th>Plan Qty</th>
                <th>Produced</th>
                <th>Accepted</th>
                <th>NC Qty</th>
                <th>Quality %</th>
            </tr>
        </thead>
        <tbody>
    `;

    data.slice(0, 15).forEach(emp => {
        const planQty = parseFloat(emp.total_plan_qty) || 0;
        const producedQty = parseFloat(emp.total_produced_qty) || 0;
        const acceptedQty = parseFloat(emp.total_accepted_qty) || 0;
        const ncQty = parseFloat(emp.nc_qty) || 0;
        const qualityRate = producedQty > 0 ? ((acceptedQty / producedQty) * 100).toFixed(2) : 0;

        html += `
            <tr>
                <td><strong>${emp.employee_name || '-'}</strong></td>
                <td>${emp.working_days || 0}</td>
                <td>${emp.tools_handled || 0}</td>
                <td>${planQty.toFixed(2)}</td>
                <td>${producedQty.toFixed(2)}</td>
                <td>${acceptedQty.toFixed(2)}</td>
                <td style="color: #e74c3c;">${ncQty.toFixed(2)}</td>
                <td>${qualityRate}%</td>
            </tr>
        `;
    });

    html += `
        </tbody>
    </table>
    `;

    container.innerHTML = html;
}

// RENDER EMPLOYEE DETAILED ANALYTICS (for search page)
function renderEmployeeDetailedAnalyticsChart(containerId = 'employeeDetailAnalyticsContainer', data = []) {
    if (data.length === 0) {
        console.warn('No employee detailed analytics data');
        return;
    }

    const container = document.getElementById(containerId);
    if (!container) {
        console.warn(`Container ${containerId} not found`);
        return;
    }

    let html = '<div class="analytics-summary">';
    
    // Calculate totals
    let totalPlan = 0, totalProduced = 0, totalAccepted = 0, totalNC = 0;
    data.forEach(entry => {
        totalPlan += parseFloat(entry.plan_qty) || 0;
        totalProduced += parseFloat(entry.produced_qty) || 0;
        totalAccepted += parseFloat(entry.accepted_qty) || 0;
        totalNC += parseFloat(entry.nc_qty) || 0;
    });

    const qualityRate = totalProduced > 0 ? ((totalAccepted / totalProduced) * 100).toFixed(2) : 0;
    const efficiency = totalPlan > 0 ? ((totalProduced / totalPlan) * 100).toFixed(2) : 0;

    html += `
        <div class="summary-cards">
            <div class="summary-card">
                <div class="card-label">Total Plan Qty</div>
                <div class="card-value">${totalPlan.toFixed(2)}</div>
            </div>
            <div class="summary-card">
                <div class="card-label">Total Produced</div>
                <div class="card-value">${totalProduced.toFixed(2)}</div>
            </div>
            <div class="summary-card">
                <div class="card-label">Total Accepted</div>
                <div class="card-value">${totalAccepted.toFixed(2)}</div>
            </div>
            <div class="summary-card" style="background: #e74c3c;">
                <div class="card-label">NC Qty</div>
                <div class="card-value">${totalNC.toFixed(2)}</div>
            </div>
            <div class="summary-card">
                <div class="card-label">Quality Rate</div>
                <div class="card-value">${qualityRate}%</div>
            </div>
            <div class="summary-card">
                <div class="card-label">Efficiency</div>
                <div class="card-value">${efficiency}%</div>
            </div>
        </div>
    `;

    // Get last 7 days of data and group by date
    const last7Days = {};
    const sortedData = data.sort((a, b) => new Date(b.date) - new Date(a.date)).slice(0, 100);
    
    sortedData.forEach(entry => {
        const date = entry.date || 'Unknown';
        if (!last7Days[date]) {
            last7Days[date] = {
                plan: 0,
                produced: 0,
                accepted: 0,
                nc: 0
            };
        }
        last7Days[date].plan += parseFloat(entry.plan_qty) || 0;
        last7Days[date].produced += parseFloat(entry.produced_qty) || 0;
        last7Days[date].accepted += parseFloat(entry.accepted_qty) || 0;
        last7Days[date].nc += parseFloat(entry.nc_qty) || 0;
    });

    // Convert to array and take last 7 unique dates
    const dateKeys = Object.keys(last7Days).slice(0, 7).reverse();
    
    // Add chart if we have data
    if (dateKeys.length > 0) {
        html += `
            <div style="margin-top: 30px; background: #fafafa; padding: 20px; border-radius: 8px; border: 1px solid #eee;">
                <h4 style="margin-top: 0; color: #333; display: flex; align-items: center; gap: 8px;">
                    <i class="fas fa-chart-bar"></i> Day-Wise Production Breakdown (Last 7 Days)
                </h4>
                <div style="position: relative; height: 350px; width: 100%;">
                    <canvas id="employeeDayWiseChart"></canvas>
                </div>
            </div>
        `;
    }

    // Render detailed table
    html += `
        <div style="margin-top: 30px;">
            <h4 style="color: #333; display: flex; align-items: center; gap: 8px;">
                <i class="fas fa-table"></i> Detailed Daily Records
            </h4>
            <table class="analytics-table">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Tool</th>
                        <th>Operation</th>
                        <th>Part Name</th>
                        <th>Plan</th>
                        <th>Produced</th>
                        <th>Accepted</th>
                        <th>NC</th>
                        <th>Quality %</th>
                    </tr>
                </thead>
                <tbody>
    `;

    data.slice(0, 20).forEach(entry => {
        const plan = parseFloat(entry.plan_qty) || 0;
        const produced = parseFloat(entry.produced_qty) || 0;
        const accepted = parseFloat(entry.accepted_qty) || 0;
        const nc = parseFloat(entry.nc_qty) || 0;
        const quality = produced > 0 ? ((accepted / produced) * 100).toFixed(2) : 0;

        html += `
            <tr>
                <td>${entry.date || '-'}</td>
                <td>${entry.tool_no || '-'}</td>
                <td>${entry.operation || '-'}</td>
                <td>${entry.part_name || '-'}</td>
                <td>${plan.toFixed(2)}</td>
                <td>${produced.toFixed(2)}</td>
                <td>${accepted.toFixed(2)}</td>
                <td style="color: #e74c3c;">${nc.toFixed(2)}</td>
                <td>${quality}%</td>
            </tr>
        `;
    });

    html += `
                </tbody>
            </table>
        </div>
        </div>
    `;

    container.innerHTML = html;

    // Render day-wise chart if container exists
    if (dateKeys.length > 0) {
        setTimeout(() => {
            renderEmployeeDayWiseChart(dateKeys, last7Days);
        }, 100);
    }
}

// RENDER EMPLOYEE DAY-WISE GROUPED BAR CHART
function renderEmployeeDayWiseChart(dateKeys, dayData) {
    const ctx = document.getElementById('employeeDayWiseChart');
    if (!ctx) {
        console.warn('Chart container employeeDayWiseChart not found');
        return;
    }

    // Destroy existing chart if it exists
    if (window.employeeDayWiseChartInstance && typeof window.employeeDayWiseChartInstance.destroy === 'function') {
        window.employeeDayWiseChartInstance.destroy();
    }

    // Prepare data
    const planData = dateKeys.map(date => dayData[date].plan);
    const producedData = dateKeys.map(date => dayData[date].produced);
    const acceptedData = dateKeys.map(date => dayData[date].accepted);
    const ncData = dateKeys.map(date => dayData[date].nc);

    window.employeeDayWiseChartInstance = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: dateKeys,
            datasets: [
                {
                    label: 'Plan Qty',
                    data: planData,
                    backgroundColor: '#3498db',
                    borderColor: '#2980b9',
                    borderWidth: 1,
                    borderRadius: 4
                },
                {
                    label: 'Produced Qty',
                    data: producedData,
                    backgroundColor: '#2ecc71',
                    borderColor: '#27ae60',
                    borderWidth: 1,
                    borderRadius: 4
                },
                {
                    label: 'Accepted Qty',
                    data: acceptedData,
                    backgroundColor: '#f39c12',
                    borderColor: '#d68910',
                    borderWidth: 1,
                    borderRadius: 4
                },
                {
                    label: 'NC Qty',
                    data: ncData,
                    backgroundColor: '#e74c3c',
                    borderColor: '#c0392b',
                    borderWidth: 1,
                    borderRadius: 4
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            indexAxis: 'x',
            plugins: {
                legend: {
                    position: 'top',
                    labels: {
                        font: { size: 13, weight: 'bold' },
                        padding: 15,
                        usePointStyle: true
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0,0,0,0.8)',
                    padding: 12,
                    titleFont: { size: 13, weight: 'bold' },
                    bodyFont: { size: 12 },
                    borderColor: '#fff',
                    borderWidth: 1,
                    displayColors: true,
                    callbacks: {
                        label: function(context) {
                            return context.dataset.label + ': ' + context.parsed.y.toFixed(2);
                        }
                    }
                }
            },
            scales: {
                x: {
                    stacked: false,
                    grid: { display: false },
                    ticks: { font: { size: 12 } }
                },
                y: {
                    stacked: false,
                    beginAtZero: true,
                    grid: { color: 'rgba(0,0,0,0.05)' },
                    ticks: { font: { size: 12 } }
                }
            }
        }
    });
}

// RENDER TOOL-WISE WORKER BREAKDOWN
function renderToolWorkerBreakdown(containerId = 'toolWorkerContainer', data = []) {
    if (data.length === 0) {
        console.warn('No tool-worker data');
        return;
    }

    const container = document.getElementById(containerId);
    if (!container) {
        console.warn(`Container ${containerId} not found`);
        return;
    }

    let html = `
    <table class="analytics-table">
        <thead>
            <tr>
                <th>Worker Name</th>
                <th>Entries</th>
                <th>Days</th>
                <th>Plan Qty</th>
                <th>Produced Qty</th>
                <th>Accepted Qty</th>
                <th>NC Qty</th>
            </tr>
        </thead>
        <tbody>
    `;

    data.forEach(worker => {
        const plan = parseFloat(worker.plan_qty) || 0;
        const produced = parseFloat(worker.produced_qty) || 0;
        const accepted = parseFloat(worker.accepted_qty) || 0;
        const nc = parseFloat(worker.nc_qty) || 0;

        html += `
            <tr>
                <td><strong>${worker.worker_name || '-'}</strong></td>
                <td>${worker.entries || 0}</td>
                <td>${worker.days || 0}</td>
                <td>${plan.toFixed(2)}</td>
                <td>${produced.toFixed(2)}</td>
                <td>${accepted.toFixed(2)}</td>
                <td style="color: #e74c3c;">${nc.toFixed(2)}</td>
            </tr>
        `;
    });

    html += `
        </tbody>
    </table>
    `;

    container.innerHTML = html;
}

// RENDER EMPLOYEE INTERACTIVE BAR CHART (Last 7 Days with 4 bars per day)
function renderEmployeeLast7DaysChart(containerId = 'employeeLast7DaysChart', data = []) {
    if (!data || data.length === 0) {
        console.warn('No employee data for 7-day chart');
        return;
    }

    try {
        // Group data by date for last 7 days
        const dateMap = {};
        data.forEach(entry => {
            const date = entry.date || entry.log_date;
            if (!dateMap[date]) {
                dateMap[date] = {
                    plan_qty: 0,
                    produced_qty: 0,
                    accepted_qty: 0,
                    nc_qty: 0
                };
            }
            dateMap[date].plan_qty += parseFloat(entry.plan_qty) || 0;
            dateMap[date].produced_qty += parseFloat(entry.produced_qty) || 0;
            dateMap[date].accepted_qty += parseFloat(entry.accepted_qty) || 0;
            dateMap[date].nc_qty += parseFloat(entry.nc_qty) || 0;
        });

        const dates = Object.keys(dateMap).sort().slice(-7); // Last 7 days
        const planQty = dates.map(d => dateMap[d].plan_qty);
        const producedQty = dates.map(d => dateMap[d].produced_qty);
        const acceptedQty = dates.map(d => dateMap[d].accepted_qty);
        const ncQty = dates.map(d => dateMap[d].nc_qty);

        const canvas = document.getElementById(containerId);
        if (!canvas) {
            console.warn(`Chart canvas ${containerId} not found`);
            return;
        }

        // Destroy existing chart
        if (window.employee7DaysChartInstance && typeof window.employee7DaysChartInstance.destroy === 'function') {
            try {
                window.employee7DaysChartInstance.destroy();
            } catch(e) {
                console.warn('Error destroying previous chart:', e);
            }
        }

        const ctx = canvas.getContext('2d');
        window.employee7DaysChartInstance = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: dates.map(d => {
                    const date = new Date(d);
                    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
                }),
                datasets: [
                    {
                        label: 'Plan Qty',
                        data: planQty,
                        backgroundColor: '#3498db',
                        borderColor: '#2980b9',
                        borderWidth: 1,
                        order: 1
                    },
                    {
                        label: 'Produced Qty',
                        data: producedQty,
                        backgroundColor: '#2ecc71',
                        borderColor: '#27ae60',
                        borderWidth: 1,
                        order: 2
                    },
                    {
                        label: 'Accepted Qty',
                        data: acceptedQty,
                        backgroundColor: '#f39c12',
                        borderColor: '#d68910',
                        borderWidth: 1,
                        order: 3
                    },
                    {
                        label: 'NC Qty',
                        data: ncQty,
                        backgroundColor: '#e74c3c',
                        borderColor: '#c0392b',
                        borderWidth: 1,
                        order: 4
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            font: { size: 12 },
                            padding: 15,
                            boxWidth: 15
                        }
                    },
                    title: {
                        display: true,
                        text: 'Last 7 Days - Production Analysis (Plan | Produced | Accepted | NC)',
                        font: { size: 14, weight: 'bold' }
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return context.dataset.label + ': ' + context.parsed.y.toFixed(2);
                            }
                        }
                    }
                },
                scales: {
                    x: {
                        stacked: false,
                        grid: { display: false }
                    },
                    y: {
                        stacked: false,
                        beginAtZero: true,
                        title: { display: true, text: 'Quantity' }
                    }
                }
            }
        });
        console.log('✓ Employee 7-day chart rendered successfully');
    } catch(err) {
        console.error('Error rendering 7-day chart:', err);
    }
}

// RENDER EMPLOYEE EFFICIENCY CHART (Efficiency % over 7 days)
function renderEmployeeEfficiencyChart(containerId = 'employeeEfficiencyChart', data = []) {
    if (!data || data.length === 0) {
        console.warn('No employee data for efficiency chart');
        return;
    }

    try {
        // Group data by date for last 7 days
        const dateMap = {};
        data.forEach(entry => {
            const date = entry.date || entry.log_date;
            if (!dateMap[date]) {
                dateMap[date] = {
                    plan_qty: 0,
                    produced_qty: 0
                };
            }
            dateMap[date].plan_qty += parseFloat(entry.plan_qty) || 0;
            dateMap[date].produced_qty += parseFloat(entry.produced_qty) || 0;
        });

        const dates = Object.keys(dateMap).sort().slice(-7); // Last 7 days
        const efficiencyPercent = dates.map(d => {
            const plan = dateMap[d].plan_qty || 0;
            return plan > 0 ? ((dateMap[d].produced_qty / plan) * 100).toFixed(2) : 0;
        });

        const canvas = document.getElementById(containerId);
        if (!canvas) {
            console.warn(`Chart canvas ${containerId} not found`);
            return;
        }

        // Destroy existing chart
        if (window.employeeEfficiencyChartInstance && typeof window.employeeEfficiencyChartInstance.destroy === 'function') {
            try {
                window.employeeEfficiencyChartInstance.destroy();
            } catch(e) {
                console.warn('Error destroying previous chart:', e);
            }
        }

        const ctx = canvas.getContext('2d');
        window.employeeEfficiencyChartInstance = new Chart(ctx, {
            type: 'line',
            data: {
                labels: dates.map(d => {
                    const date = new Date(d);
                    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
                }),
                datasets: [
                    {
                        label: 'Efficiency %',
                        data: efficiencyPercent,
                        backgroundColor: 'rgba(52, 152, 219, 0.1)',
                        borderColor: '#3498db',
                        borderWidth: 3,
                        fill: true,
                        tension: 0.4,
                        pointRadius: 6,
                        pointBackgroundColor: '#2980b9',
                        pointBorderColor: 'white',
                        pointBorderWidth: 2
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            font: { size: 12 },
                            padding: 15,
                            boxWidth: 15
                        }
                    },
                    title: {
                        display: true,
                        text: 'Efficiency Trend (Produced ÷ Plan × 100)',
                        font: { size: 13, weight: 'bold' }
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return 'Efficiency: ' + context.parsed.y + '%';
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 150,
                        title: { display: true, text: 'Efficiency %' }
                    }
                }
            }
        });
        console.log('✓ Employee efficiency chart rendered successfully');
    } catch(err) {
        console.error('Error rendering efficiency chart:', err);
    }
}

// RENDER EMPLOYEE PRODUCTIVITY CHART (Quality % over 7 days)
function renderEmployeeProductivityChart(containerId = 'employeeProductivityChart', data = []) {
    if (!data || data.length === 0) {
        console.warn('No employee data for productivity chart');
        return;
    }

    try {
        // Group data by date for last 7 days
        const dateMap = {};
        data.forEach(entry => {
            const date = entry.date || entry.log_date;
            if (!dateMap[date]) {
                dateMap[date] = {
                    produced_qty: 0,
                    accepted_qty: 0
                };
            }
            dateMap[date].produced_qty += parseFloat(entry.produced_qty) || 0;
            dateMap[date].accepted_qty += parseFloat(entry.accepted_qty) || 0;
        });

        const dates = Object.keys(dateMap).sort().slice(-7); // Last 7 days
        const qualityPercent = dates.map(d => {
            const produced = dateMap[d].produced_qty || 0;
            return produced > 0 ? ((dateMap[d].accepted_qty / produced) * 100).toFixed(2) : 0;
        });
        const productivity = dates.map(d => dateMap[d].produced_qty);

        const canvas = document.getElementById(containerId);
        if (!canvas) {
            console.warn(`Chart canvas ${containerId} not found`);
            return;
        }

        // Destroy existing chart
        if (window.employeeProductivityChartInstance && typeof window.employeeProductivityChartInstance.destroy === 'function') {
            try {
                window.employeeProductivityChartInstance.destroy();
            } catch(e) {
                console.warn('Error destroying previous chart:', e);
            }
        }

        const ctx = canvas.getContext('2d');
        window.employeeProductivityChartInstance = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: dates.map(d => {
                    const date = new Date(d);
                    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
                }),
                datasets: [
                    {
                        label: 'Produced Qty',
                        data: productivity,
                        backgroundColor: '#2ecc71',
                        borderColor: '#27ae60',
                        borderWidth: 1,
                        yAxisID: 'y'
                    },
                    {
                        label: 'Quality %',
                        data: qualityPercent,
                        borderColor: '#f39c12',
                        backgroundColor: 'rgba(243, 156, 18, 0.1)',
                        borderWidth: 3,
                        type: 'line',
                        fill: false,
                        tension: 0.4,
                        pointRadius: 5,
                        pointBackgroundColor: '#f39c12',
                        pointBorderColor: 'white',
                        pointBorderWidth: 2,
                        yAxisID: 'y1'
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: {
                    mode: 'index',
                    intersect: false
                },
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            font: { size: 12 },
                            padding: 15,
                            boxWidth: 15
                        }
                    },
                    title: {
                        display: true,
                        text: 'Productivity & Quality (Bars: Produced | Line: Quality %)',
                        font: { size: 13, weight: 'bold' }
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                if (context.dataset.yAxisID === 'y1') {
                                    return context.dataset.label + ': ' + context.parsed.y + '%';
                                } else {
                                    return context.dataset.label + ': ' + context.parsed.y.toFixed(0);
                                }
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: { display: true, text: 'Produced Quantity' }
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        beginAtZero: true,
                        max: 100,
                        title: { display: true, text: 'Quality %' },
                        grid: { drawOnChartArea: false }
                    }
                }
            }
        });
        console.log('✓ Employee productivity chart rendered successfully');
    } catch(err) {
        console.error('Error rendering productivity chart:', err);
    }
}

// Initialize cache when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeEmployeeCache);
} else {
    initializeEmployeeCache();
}
