// Notification System JavaScript Module
// File: notifications.js
// Purpose: Handles notification UI interactions and API calls

class NotificationManager {
    constructor() {
        this.notifications = [];
        this.refreshInterval = 30000; // 30 seconds
        this.init();
    }
    
    init() {
        this.setupEventListeners();
        this.loadNotifications();
        this.startAutoRefresh();
    }
    
    setupEventListeners() {
        const closeBtn = document.getElementById('notificationClose');
        const overlay = document.getElementById('notificationOverlay');
        const settingsBtn = document.getElementById('settingsBtn');
        const clearBtn = document.getElementById('clearBtn');
        const saveSettingsBtn = document.getElementById('savSettingsBtn');
        const cancelSettingsBtn = document.getElementById('cancelSettingsBtn');
        
        if (closeBtn) closeBtn.addEventListener('click', () => this.closeSidebar());
        if (overlay) overlay.addEventListener('click', () => this.closeSidebar());
        if (settingsBtn) settingsBtn.addEventListener('click', () => this.openSettings());
        if (clearBtn) clearBtn.addEventListener('click', () => this.clearAll());
        if (saveSettingsBtn) saveSettingsBtn.addEventListener('click', () => this.saveSettings());
        if (cancelSettingsBtn) cancelSettingsBtn.addEventListener('click', () => this.closeSettings());
        
        // Keyboard shortcut: Escape to close
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                this.closeSidebar();
                this.closeSettings();
            }
        });
    }
    
    async loadNotifications() {
        try {
            const response = await fetch('notifications.php?action=get_notifications');
            if (!response.ok) {
                console.error('Notifications API error:', response.status);
                return;
            }
            
            const data = await response.json();
            
            if (data.success) {
                this.notifications = data.notifications || [];
                this.render();
            } else {
                console.error('Notifications API error:', data.error);
            }
        } catch (error) {
            console.error('Failed to load notifications:', error);
        }
    }
    
    render() {
        const btn = document.querySelector('.notification-btn');
        const content = document.getElementById('notificationContent');
        const badge = document.getElementById('notificationBadgeHeader');
        
        if (!content) return;
        
        if (this.notifications.length === 0) {
            content.innerHTML = `
                <div class="notification-empty">
                    <i class="fas fa-bell-slash"></i>
                    <p>No notifications at the moment</p>
                </div>
            `;
            
            // Remove badge if present
            const existingBadge = btn?.querySelector('.notification-badge');
            if (existingBadge) {
                existingBadge.remove();
            }
            
            if (badge) badge.textContent = '';
        } else {
            // Render notification items
            content.innerHTML = this.notifications.map(notif => `
                <div class="notification-item ${notif.severity || 'info'}">
                    <div class="notification-item-header">
                        <div class="notification-item-title">
                            <i class="fas fa-${notif.type === 'machine' ? 'cogs' : 'tools'} notification-icon ${notif.type}"></i>
                            ${this.escapeHtml(notif.title)}
                        </div>
                        <span class="notification-severity ${notif.severity || 'info'}">${notif.severity || 'info'}</span>
                    </div>
                    <div class="notification-item-message">${this.escapeHtml(notif.message)}</div>
                    <div class="notification-item-meta">
                        <span>${this.escapeHtml(notif.item_name || 'Unknown')}</span>
                        <span>${this.escapeHtml(notif.timestamp || 'Now')}</span>
                    </div>
                    <div class="notification-item-action">
                        <button onclick="window.location.href='maintenance.php'" class="btn-action">
                            <i class="fas fa-arrow-right"></i> Go to Maintenance
                        </button>
                    </div>
                </div>
            `).join('');
            
            // Update button badge
            const criticalCount = this.notifications.filter(n => n.severity === 'critical').length;
            const totalCount = this.notifications.length;
            
            // Create or update badge
            let badgeEl = btn?.querySelector('.notification-badge');
            if (!badgeEl) {
                badgeEl = document.createElement('span');
                badgeEl.className = 'notification-badge';
                btn?.appendChild(badgeEl);
            }
            badgeEl.textContent = totalCount;
            
            // Update header badge
            if (badge) {
                badge.textContent = criticalCount > 0 
                    ? `${criticalCount} CRITICAL` 
                    : `${totalCount} ALERT${totalCount !== 1 ? 'S' : ''}`;
            }
        }
    }
    
    openSidebar() {
        const sidebar = document.getElementById('notificationSidebar');
        const overlay = document.getElementById('notificationOverlay');
        
        if (sidebar) sidebar.classList.add('open');
        if (overlay) overlay.classList.add('open');
    }
    
    closeSidebar() {
        const sidebar = document.getElementById('notificationSidebar');
        const overlay = document.getElementById('notificationOverlay');
        
        if (sidebar) sidebar.classList.remove('open');
        if (overlay) overlay.classList.remove('open');
    }
    
    openSettings() {
        this.closeSidebar();
        
        const modal = document.getElementById('notificationSettingsModal');
        if (modal) {
            modal.style.display = 'flex';
            this.loadSettings();
        }
    }
    
    closeSettings() {
        const modal = document.getElementById('notificationSettingsModal');
        if (modal) {
            modal.style.display = 'none';
        }
    }
    
    async loadSettings() {
        try {
            const response = await fetch('notifications.php?action=get_notification_settings');
            const data = await response.json();
            
            if (data.success && data.settings) {
                const machineInput = document.getElementById('machineThreshold');
                const toolInput = document.getElementById('toolThreshold');
                
                if (machineInput) machineInput.value = data.settings.machine_days_threshold;
                if (toolInput) toolInput.value = data.settings.tool_strokes_threshold;
            }
        } catch (error) {
            console.error('Failed to load settings:', error);
        }
    }
    
    async saveSettings() {
        const machineThreshold = document.getElementById('machineThreshold').value;
        const toolThreshold = document.getElementById('toolThreshold').value;
        
        // Validation
        if (!machineThreshold || !toolThreshold) {
            alert('Please fill in all fields');
            return;
        }
        
        if (isNaN(machineThreshold) || isNaN(toolThreshold)) {
            alert('Please enter valid numbers');
            return;
        }
        
        if (machineThreshold < 1 || toolThreshold < 1) {
            alert('Values must be greater than 0');
            return;
        }
        
        const fd = new FormData();
        fd.append('action', 'update_notification_settings');
        fd.append('machine_days_threshold', machineThreshold);
        fd.append('tool_strokes_threshold', toolThreshold);
        
        try {
            const response = await fetch('notifications.php', { 
                method: 'POST', 
                body: fd 
            });
            const data = await response.json();
            
            if (data.success) {
                alert('Settings saved successfully!');
                this.closeSettings();
                this.loadNotifications(); // Refresh notifications
            } else {
                alert('Error: ' + (data.error || 'Unknown error'));
            }
        } catch (error) {
            alert('Failed to save settings: ' + error.message);
        }
    }
    
    clearAll() {
        if (confirm('Are you sure you want to clear all notifications?')) {
            this.notifications = [];
            this.render();
            this.closeSidebar();
        }
    }
    
    startAutoRefresh() {
        // Initial load already done in init()
        // Set up interval for subsequent refreshes
        setInterval(() => {
            // Only refresh if sidebar is open (performance optimization)
            const sidebar = document.getElementById('notificationSidebar');
            if (sidebar && sidebar.classList.contains('open')) {
                this.loadNotifications();
            }
        }, this.refreshInterval);
        
        // Also refresh even when closed, but less frequently
        setInterval(() => {
            this.loadNotifications();
        }, this.refreshInterval * 2);
    }
    
    escapeHtml(text) {
        if (!text) return '';
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, m => map[m]);
    }
}

// Global function for button click handler
function toggleNotificationSidebar() {
    const sidebar = document.getElementById('notificationSidebar');
    if (!sidebar) return;
    
    if (sidebar.classList.contains('open')) {
        window.notificationManager?.closeSidebar();
    } else {
        window.notificationManager?.openSidebar();
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    // Only initialize if notification UI is present
    if (document.getElementById('notificationSidebar')) {
        window.notificationManager = new NotificationManager();
    }
});

// Listen for storage events (for multi-tab sync)
window.addEventListener('storage', (e) => {
    if (e.key === 'notification_refresh') {
        window.notificationManager?.loadNotifications();
    }
});
