<?php
// api.php - Final Master Version (Fixed Field Saving & Logging)
session_start();
ini_set('display_errors', 0);
error_reporting(E_ALL);
header('Content-Type: application/json');

// --- 1. DATABASE CONNECTION ---
try {
    // UPDATE THESE CREDENTIALS IF NEEDED
  $host = 'localhost'; 
    $db = 'zemaraim_mektron_db'; 
    $user = 'zemaraim'; 
    $pass = '8wwxk]-X3Kx]CJ]5';          // Default XAMPP password is empty
    
    $pdo = new PDO("mysql:host=$host;dbname=$db;charset=utf8mb4", $user, $pass, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
    ]);
} catch (\PDOException $e) { 
    echo json_encode(['success' => false, 'error' => 'DB Connection Failed: ' . $e->getMessage()]); 
    exit; 
}

$action = $_GET['action'] ?? $_POST['action'] ?? '';

// --- SECURITY HELPERS ---
function requireLogin() {
    if (!isset($_SESSION['role'])) {
        echo json_encode(['success' => false, 'error' => 'Unauthorized']); exit;
    }
}
function requireAdmin() {
    requireLogin();
    // Only "Administrator" role can access admin endpoints
    if ($_SESSION['role'] !== 'Administrator') {
        echo json_encode(['success' => false, 'error' => 'Access Denied - Administrator role required']); exit;
    }
}

function getProductionLogColumns(PDO $pdo) {
    static $columnMap = null;
    if ($columnMap !== null) {
        return $columnMap;
    }

    $columnMap = [];
    $stmt = $pdo->query("SHOW COLUMNS FROM production_logs");
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $col) {
        $name = (string)($col['Field'] ?? '');
        if ($name !== '') {
            $columnMap[$name] = true;
        }
    }
    return $columnMap;
}

function hasProductionLogColumn(PDO $pdo, $columnName) {
    if (!is_string($columnName) || $columnName === '') {
        return false;
    }
    $columns = getProductionLogColumns($pdo);
    return isset($columns[$columnName]);
}

function getProductionLogNonEmptyCount(PDO $pdo, $columnName) {
    if (!hasProductionLogColumn($pdo, $columnName)) {
        return -1;
    }

    try {
        $stmt = $pdo->query("SELECT COUNT(*) as cnt
                             FROM production_logs
                             WHERE `$columnName` IS NOT NULL
                               AND TRIM(CAST(`$columnName` AS CHAR)) != ''");
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)($row['cnt'] ?? 0);
    } catch (Exception $e) {
        return -1;
    }
}

function resolveProductionQtyColumn(PDO $pdo) {
    $preferred = ['produced_qty', 'f_produced_qty', 'production_qty', 'qty_produced', 'produced'];
    foreach ($preferred as $col) {
        if (hasProductionLogColumn($pdo, $col)) {
            return $col;
        }
    }

    try {
        $stmt = $pdo->query("SELECT field_name, field_label 
                             FROM custom_fields 
                             WHERE is_active = 1
                             ORDER BY display_order ASC");
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $field) {
            $fieldName = (string)($field['field_name'] ?? '');
            $fieldLabel = strtolower(trim((string)($field['field_label'] ?? '')));
            $fieldNameLower = strtolower(trim($fieldName));
            if (
                strpos($fieldNameLower, 'produced') !== false ||
                strpos($fieldNameLower, 'production_qty') !== false ||
                strpos($fieldNameLower, 'qty_produced') !== false ||
                strpos($fieldLabel, 'produced') !== false ||
                strpos($fieldLabel, 'production qty') !== false
            ) {
                if (hasProductionLogColumn($pdo, $fieldName)) {
                    return $fieldName;
                }
            }
        }
    } catch (Exception $e) {
        // Keep fallback behavior
    }

    return null;
}

function resolveAcceptedQtyColumn(PDO $pdo) {
    $preferred = ['accepted_qty', 'f_accepted_qty', 'accept_qty', 'accepted_quantity', 'accepted'];
    foreach ($preferred as $col) {
        if (hasProductionLogColumn($pdo, $col)) {
            return $col;
        }
    }

    try {
        $stmt = $pdo->query("SELECT field_name, field_label 
                             FROM custom_fields 
                             WHERE is_active = 1
                             ORDER BY display_order ASC");
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $field) {
            $fieldName = (string)($field['field_name'] ?? '');
            $fieldLabel = strtolower(trim((string)($field['field_label'] ?? '')));
            $fieldNameLower = strtolower(trim($fieldName));
            if (
                strpos($fieldNameLower, 'accepted') !== false ||
                strpos($fieldNameLower, 'accept_qty') !== false ||
                strpos($fieldLabel, 'accepted') !== false ||
                strpos($fieldLabel, 'accept qty') !== false
            ) {
                if (hasProductionLogColumn($pdo, $fieldName)) {
                    return $fieldName;
                }
            }
        }
    } catch (Exception $e) {
        // Keep fallback behavior
    }

    return null;
}

function resolvePlanQtyColumn(PDO $pdo) {
    $preferred = ['plan_qty', 'f_plan_qty', 'planned_qty', 'target_qty', 'plan'];
    foreach ($preferred as $col) {
        if (hasProductionLogColumn($pdo, $col)) {
            return $col;
        }
    }

    try {
        $stmt = $pdo->query("SELECT field_name, field_label 
                             FROM custom_fields 
                             WHERE is_active = 1
                             ORDER BY display_order ASC");
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $field) {
            $fieldName = (string)($field['field_name'] ?? '');
            $fieldLabel = strtolower(trim((string)($field['field_label'] ?? '')));
            $fieldNameLower = strtolower(trim($fieldName));
            if (
                strpos($fieldNameLower, 'plan') !== false ||
                strpos($fieldNameLower, 'target') !== false ||
                strpos($fieldLabel, 'plan') !== false ||
                strpos($fieldLabel, 'target') !== false
            ) {
                if (hasProductionLogColumn($pdo, $fieldName)) {
                    return $fieldName;
                }
            }
        }
    } catch (Exception $e) {
        // Keep fallback behavior
    }

    return null;
}

function resolveDurationColumn(PDO $pdo) {
    $preferred = ['duration', 'f_duration'];
    foreach ($preferred as $col) {
        if (hasProductionLogColumn($pdo, $col)) {
            return $col;
        }
    }
    return null;
}

function resolveToolColumn(PDO $pdo) {
    $preferred = ['tool_no', 'f_tool_no'];
    $bestColumn = null;
    $bestCount = -1;

    foreach ($preferred as $col) {
        if (hasProductionLogColumn($pdo, $col)) {
            $count = getProductionLogNonEmptyCount($pdo, $col);
            if ($count > $bestCount) {
                $bestCount = $count;
                $bestColumn = $col;
            }
        }
    }

    if ($bestColumn !== null && $bestCount > 0) {
        return $bestColumn;
    }

    try {
        $stmt = $pdo->query("SELECT field_name, field_label FROM custom_fields WHERE is_active = 1 ORDER BY display_order ASC");
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $field) {
            $fieldName = (string)($field['field_name'] ?? '');
            $fieldLabel = strtolower(trim((string)($field['field_label'] ?? '')));
            if ((strpos(strtolower($fieldName), 'tool') !== false || strpos($fieldLabel, 'tool') !== false) && hasProductionLogColumn($pdo, $fieldName)) {
                $count = getProductionLogNonEmptyCount($pdo, $fieldName);
                if ($count > $bestCount) {
                    $bestCount = $count;
                    $bestColumn = $fieldName;
                }
            }
        }
    } catch (Exception $e) {
        // Keep fallback behavior
    }

    return $bestColumn;
}

function resolvePartColumn(PDO $pdo) {
    $preferred = ['part_name', 'f_part_name'];
    foreach ($preferred as $col) {
        if (hasProductionLogColumn($pdo, $col)) {
            return $col;
        }
    }

    try {
        $stmt = $pdo->query("SELECT field_name, field_label FROM custom_fields WHERE is_active = 1 ORDER BY display_order ASC");
        foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $field) {
            $fieldName = (string)($field['field_name'] ?? '');
            $fieldLabel = strtolower(trim((string)($field['field_label'] ?? '')));
            if ((strpos(strtolower($fieldName), 'part') !== false || strpos($fieldLabel, 'part') !== false) && hasProductionLogColumn($pdo, $fieldName)) {
                return $fieldName;
            }
        }
    } catch (Exception $e) {
        // Keep fallback behavior
    }

    return null;
}

function numericSumExpression($columnName) {
    if (!is_string($columnName) || $columnName === '') {
        return '0';
    }
    return "COALESCE(SUM(CAST(NULLIF(`$columnName`, '') AS DECIMAL(18,2))), 0)";
}

// =========================================================
// 2. AUTHENTICATION & SESSION
// =========================================================

if ($action === 'login' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $role = $_POST['role'] ?? '';
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';

    // Validate inputs
    if (empty($role) || empty($username) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'All fields are required']);
        exit;
    }

    // Validate role is one of the allowed roles
    $validRoles = ['Administrator', 'Supervisor', 'Controller'];
    if (!in_array($role, $validRoles)) {
        echo json_encode(['success' => false, 'message' => 'Invalid role selected']);
        exit;
    }

    // Query user by username
    $stmt = $pdo->prepare("SELECT id, username, password, role FROM users WHERE username = ?");
    $stmt->execute([$username]);
    $account = $stmt->fetch();

    // Verify password matches
    if ($account && $account['password'] === $password) {
        // If user selected Supervisor, they can use Controller credentials
        if ($role === 'Supervisor' && $account['role'] === 'Controller') {
            // Allow: Log them in as Supervisor using Controller credentials
            $_SESSION['user_id'] = $account['username'];
            $_SESSION['role'] = 'Supervisor'; // Set as Supervisor
            $_SESSION['name'] = 'Supervisor';
            echo json_encode(['success' => true, 'role' => 'Supervisor']);
        }
        // If user selected Controller, they must be Controller
        else if ($role === 'Controller' && $account['role'] === 'Controller') {
            $_SESSION['user_id'] = $account['username'];
            $_SESSION['role'] = $account['role']; 
            $_SESSION['name'] = 'Controller';
            echo json_encode(['success' => true, 'role' => $account['role']]);
        }
        // If user selected Administrator, they must be Administrator
        else if ($role === 'Administrator' && $account['role'] === 'Administrator') {
            $_SESSION['user_id'] = $account['username'];
            $_SESSION['role'] = $account['role']; 
            $_SESSION['name'] = 'Administrator';
            echo json_encode(['success' => true, 'role' => $account['role']]);
        }
        // If user selected Supervisor but has Supervisor credentials
        else if ($role === 'Supervisor' && $account['role'] === 'Supervisor') {
            $_SESSION['user_id'] = $account['username'];
            $_SESSION['role'] = $account['role']; 
            $_SESSION['name'] = 'Supervisor';
            echo json_encode(['success' => true, 'role' => $account['role']]);
        }
        else {
            echo json_encode(['success' => false, 'message' => 'Access denied. Role mismatch.']);
        }
    } else {
        echo json_encode(['success' => false, 'message' => 'Invalid username or password']);
    }
    exit;
}

if ($action === 'check_session') {
    if (isset($_SESSION['role'])) {
        echo json_encode(['logged_in' => true, 'role' => $_SESSION['role'], 'name' => $_SESSION['name']]);
    } else { echo json_encode(['logged_in' => false]); }
    exit;
}

if ($action === 'logout') { 
    // Complete session destruction to prevent back button access
    $_SESSION = array(); // Clear all session variables
    
    // Send cache-prevention headers
    header("Cache-Control: no-store, no-cache, must-revalidate, max-age=0, private");
    header("Pragma: no-cache");
    header("Expires: Thu, 19 Nov 1981 08:52:00 GMT");
    
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    session_destroy(); 
    echo json_encode(['success' => true]); 
    exit; 
}

// =========================================================
// 3. FIELD MANAGEMENT (THE FIX)
// =========================================================

// GET FIELDS
if ($action === 'get_custom_fields') {
    requireLogin();
    $sql = "SELECT * FROM custom_fields";
    
    // Filter for active fields only (used by Entry Form)
    if (isset($_GET['type']) && $_GET['type'] === 'active') {
        $sql .= " WHERE is_active = 1";
    }
    
    $sql .= " ORDER BY display_order ASC"; 
    $stmt = $pdo->query($sql);
    echo json_encode($stmt->fetchAll());
    exit;
}

// ADD FIELD (Correctly saves Type & Parent)
if ($action === 'add_custom_field' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    
    $label = $_POST['label'];
    
    // 1. Capture Type (Default to 'text')
    $type = $_POST['type'] ?? 'text'; 
    
    // 2. Capture Parent ID (Empty string becomes NULL)
    $parentId = !empty($_POST['parent_id']) ? $_POST['parent_id'] : NULL;
    
    // 3. Generate safe field name (e.g. "M/C No" -> "f_mc_no")
    // Add "f_" prefix to avoid MySQL reserved keywords
    $fieldName = 'f_' . strtolower(trim(preg_replace('/[^A-Za-z0-9]+/', '_', $label), '_'));
    
    // List of MySQL reserved keywords to check
    $reservedKeywords = ['date', 'time', 'timestamp', 'key', 'value', 'year', 'month', 'day', 
                         'hour', 'minute', 'second', 'interval', 'limit', 'order', 'group', 
                         'where', 'select', 'from', 'insert', 'update', 'delete', 'create', 
                         'drop', 'alter', 'table', 'column', 'row', 'index', 'data', 'type'];
    
    // Additional safety: if it's a reserved keyword even with f_ prefix, add extra suffix
    $baseName = substr($fieldName, 2);
    if (in_array($baseName, $reservedKeywords)) {
        $fieldName = 'f_' . $baseName . '_custom';
    }
    
    try {
        // Get next display order
        $stmtIdx = $pdo->query("SELECT MAX(display_order) as max_idx FROM custom_fields");
        $nextIdx = ($stmtIdx->fetch()['max_idx'] ?? 0) + 1;

        // 4. CRITICAL: Create actual database column in production_logs table
        // Determine SQL data type based on field_type
        $sqlDataType = 'TEXT';
        if ($type === 'date') $sqlDataType = 'DATE';
        elseif ($type === 'time') $sqlDataType = 'TIME';
        elseif ($type === 'number') $sqlDataType = 'DECIMAL(10,2)';
        
        // Try to add column to production_logs table with backticks for safety
        $alterSql = "ALTER TABLE production_logs ADD COLUMN `" . $fieldName . "` " . $sqlDataType . " DEFAULT NULL";
        
        try {
            $pdo->exec($alterSql);
        } catch (Exception $colErr) {
            // Column may already exist from previous attempt - that's okay
            // Log but don't fail
            error_log("Column creation warning: " . $colErr->getMessage());
        }

        // Insert into custom_fields metadata table
        $stmt = $pdo->prepare("INSERT INTO custom_fields (field_label, field_name, field_type, parent_field_id, display_order, is_active) VALUES (?, ?, ?, ?, ?, 1)");
        $stmt->execute([$label, $fieldName, $type, $parentId, $nextIdx]);
        
        echo json_encode(['success' => true, 'field_name' => $fieldName]);
    } catch (Exception $e) { echo json_encode(['success' => false, 'error' => $e->getMessage()]); }
    exit;
}

// TOGGLE VISIBILITY
if ($action === 'toggle_field' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    $stmt = $pdo->prepare("UPDATE custom_fields SET is_active = ? WHERE id = ?");
    $stmt->execute([$_POST['status'], $_POST['id']]);
    echo json_encode(['success' => true]); exit;
}

// UPDATE ORDER
if ($action === 'update_field_order' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    if (!isset($_POST['order'])) { echo json_encode(['success' => false]); exit; }
    
    $pdo->beginTransaction();
    $stmt = $pdo->prepare("UPDATE custom_fields SET display_order = ? WHERE id = ?");
    $counter = 1;
    foreach ($_POST['order'] as $id) {
        $stmt->execute([$counter, $id]);
        $counter++;
    }
    $pdo->commit();
    echo json_encode(['success' => true]); exit;
}

// DELETE FIELD
if ($action === 'delete_custom_field' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    $stmt = $pdo->prepare("DELETE FROM custom_fields WHERE id = ?");
    $stmt->execute([$_POST['id']]);
    echo json_encode(['success' => true]); exit;
}

// =========================================================
// 4. DROPDOWN OPTIONS & DEPENDENCIES
// =========================================================

// GET OPTIONS
// GET OPTIONS (Updated to fetch Norm Value)
// GET OPTIONS (Updated to include norm_value)
if ($action === 'get_options') {
    requireLogin();
    $fieldId = $_GET['field_id'];
    
    // Select norm_value too
    $sql = "SELECT id, label, parent_option_id, norm_value FROM field_options WHERE field_id = ?";
    $params = [$fieldId];

    if (isset($_GET['parent_opt_id'])) {
        $sql .= " AND parent_option_id = ?";
        $params[] = $_GET['parent_opt_id'];
    }
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    echo json_encode($stmt->fetchAll());
    exit;
}

// ADD OPTION
// --- ADD OPTION (Modified to save Life Limit) ---
if ($action === 'add_option' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    $fid = $_POST['field_id'];
    $label = $_POST['label'];
    $norm = $_POST['norm_value'];
    $parent = isset($_POST['parent_opt_id']) ? $_POST['parent_opt_id'] : NULL;
    
    // NEW: Get the Life Limit (Interval or Max Strokes)
    $life = isset($_POST['life_limit']) ? intval($_POST['life_limit']) : 0;
    
    // Default last service to today for new machines
    $date = date('Y-m-d');

    $stmt = $pdo->prepare("INSERT INTO field_options (field_id, label, norm_value, parent_option_id, life_limit, last_service) VALUES (?, ?, ?, ?, ?, ?)");
    if ($stmt->execute([$fid, $label, $norm, $parent, $life, $date])) {
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'error' => 'Insert failed']);
    }
    exit;
}

// DELETE OPTION
if ($action === 'delete_option' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    $stmt = $pdo->prepare("DELETE FROM field_options WHERE id = ?");
    $stmt->execute([$_POST['id']]);
    echo json_encode(['success' => true]); exit;
}

// UPDATE OPTION NORM VALUE (NEW)
if ($action === 'update_option_norm' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    $optionId = $_POST['option_id'];
    $normValue = $_POST['norm_value'];

    try {
        $stmt = $pdo->prepare("UPDATE field_options SET norm_value = ? WHERE id = ?");
        $stmt->execute([$normValue, $optionId]);
        echo json_encode(['success' => true]);
    } catch (Exception $e) { echo json_encode(['success' => false, 'error' => $e->getMessage()]); }
    exit;
}

// =========================================================
// 5. EMPLOYEE MANAGEMENT
// =========================================================

// --- ACTION: GET ALL EMPLOYEES ---
// --- ACTION: GET ALL EMPLOYEES ---
if ($action == 'get_employees') {
    // Explicitly select 'status' so the Javascript can see it
    $stmt = $pdo->query("SELECT id, emp_unique_id, name, designation, join_date, status FROM employees ORDER BY name ASC");
    $employees = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode($employees);
    exit;
}
// In api.php, inside your "register_employee" block:

if ($action == 'register_employee') {
    
    // 1. Get ID (Check multiple possible keys)
    $id = $_POST['emp_unique_id'] ?? $_POST['id'] ?? null;
    
    // 2. Get Name
    $name = $_POST['name'] ?? null;
    
    // 3. Get Designation
    $designation = $_POST['designation'] ?? 'Operator';
    
    // 4. CRITICAL FIX: Get Date (Check ALL possibilities)
    // We check 'join_date', 'joining_date', AND 'date'. If all fail, use TODAY.
    $join_date = $_POST['join_date'] ?? $_POST['joining_date'] ?? $_POST['date'] ?? date('Y-m-d');

    // 5. Validation
    if(!$id || !$name) {
        echo json_encode(['success' => false, 'error' => 'ID and Name are required']);
        exit;
    }

    // 6. Insert into Database
    try {
        // Ensure your query matches these variables exactly
        $stmt = $pdo->prepare("INSERT INTO employees (emp_unique_id, name, designation, join_date) VALUES (?, ?, ?, ?)");
        $stmt->execute([$id, $name, $designation, $join_date]);
        
        echo json_encode(['success' => true]);
    } catch (PDOException $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

if ($action === 'delete_employee' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    $stmt = $pdo->prepare("DELETE FROM employees WHERE emp_unique_id = ?");
    $stmt->execute([$_POST['id']]);
    echo json_encode(['success' => true]); exit;
}

if ($action === 'get_next_id') {
    requireAdmin();
    $stmt = $pdo->query("SELECT emp_unique_id FROM employees");
    $ids = $stmt->fetchAll(PDO::FETCH_COLUMN);
    $numbers = []; foreach ($ids as $id) if(strpos($id,'EMP')===0) $numbers[] = (int)str_replace('EMP', '', $id);
    $next = 1; while (in_array($next, $numbers)) $next++;
    echo json_encode(['next_id' => 'EMP' . str_pad($next, 3, '0', STR_PAD_LEFT)]);
    exit;
}

// =========================================================
// 6. PRODUCTION LOGS & DASHBOARD (SMART SAVE)
// =========================================================

if ($action === 'save_log' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireLogin();
    
    // ✅ VALIDATION: Accept multiple field name variations
    // Try multiple possible field names for Employee ID
    $empId = $_POST['empIdSelect'] ?? $_POST['emp_id'] ?? $_POST['empId'] ?? $_POST['employee_id'] ?? null;
    
    // Try multiple possible field names for Date
    $logDate = $_POST['log_date'] ?? $_POST['date'] ?? $_POST['log_date'] ?? null;
    
    if (empty($empId) || empty($logDate)) {
        echo json_encode(['success' => false, 'error' => 'Employee ID and Date are required']);
        exit;
    }
    
    // 1. Get ALL custom fields from database to ensure we have the correct field names
    $stmt = $pdo->query("SELECT field_name FROM custom_fields WHERE is_active = 1");
    $customFieldRows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $customFieldNames = array_map(fn($row) => $row['field_name'], $customFieldRows);
    
    // 2. Standard required columns
    $required_columns = ['log_date', 'operator_name', 'emp_id'];
    $optional_columns = ['start_time', 'end_time', 'duration'];
    $all_standard_columns = array_merge($required_columns, $optional_columns, ['action', 'empIdSelect', 'empId']);
    
    // 3. Build insert data with ALL columns (including empty ones for proper storage)
    $insert_data = [];
    
    // Add required fields (using the values we found)
    $insert_data['emp_id'] = trim($empId);
    $insert_data['operator_name'] = trim($_POST['operator_name'] ?? $_POST['name'] ?? '');
    $insert_data['log_date'] = trim($logDate);
    
    // Add optional standard columns if they have values
    foreach ($optional_columns as $col) {
        if (!empty($_POST[$col])) {
            $insert_data[$col] = trim($_POST[$col]);
        }
    }
    
    // 4. Add ALL custom fields from database (will use POST value if exists, or empty string if not)
    foreach ($customFieldNames as $fieldName) {
        $value = $_POST[$fieldName] ?? '';
        $trimmed_value = trim($value);
        // Store all values (including empty strings for custom fields)
        if (!empty($trimmed_value)) {
            $insert_data[$fieldName] = $trimmed_value;
        }
    }

    // ===== VALIDATION & CALCULATION BLOCK =====
    // Get the three required input values - handle field name variations
    $planQty = isset($_POST['plan_qty']) ? floatval($_POST['plan_qty']) : 
               (isset($_POST['f_plan_qty']) ? floatval($_POST['f_plan_qty']) : 0);
    
    $producedQty = isset($_POST['produced_qty']) ? intval($_POST['produced_qty']) : 
                   (isset($_POST['f_produced_qty']) ? intval($_POST['f_produced_qty']) : 0);
    $acceptedQty = isset($_POST['accepted_qty']) ? intval($_POST['accepted_qty']) : 
                   (isset($_POST['f_accepted_qty']) ? intval($_POST['f_accepted_qty']) : 0);

    // === FRONTEND VALIDATION (Backend fallback) ===
    if ($planQty <= 0) {
        echo json_encode(['success' => false, 'error' => 'Plan Qty must be greater than 0']);
        exit;
    }
    if ($producedQty < 0) {
        echo json_encode(['success' => false, 'error' => 'Produced Qty cannot be negative']);
        exit;
    }
    if ($acceptedQty < 0) {
        echo json_encode(['success' => false, 'error' => 'Accepted Qty cannot be negative']);
        exit;
    }
    if ($acceptedQty > $producedQty) {
        echo json_encode(['success' => false, 'error' => 'Accepted Qty cannot exceed Produced Qty']);
        exit;
    }

    // === AUTO-CALCULATE DERIVED VALUES (DO NOT STORE) ===
    // These are calculated on-the-fly and NOT stored in database
    $ncQty = $producedQty - $acceptedQty;  // NC Qty = Produced - Accepted
    $efficiency = ($planQty > 0) ? ($producedQty / $planQty) * 100 : 0;  // CORRECT: Produced/Plan * 100
    $ncPercent = ($producedQty > 0) ? ($ncQty / $producedQty) * 100 : 0;  // NC% = NC Qty / Produced * 100

    // === REMOVE MANUAL CALCULATION FIELDS FROM STORAGE ===
    // DO NOT store: f_efficiency, f_nc_percent, f_production_efficiency, f_nc, nc_qty
    // These will be calculated dynamically in the report
    unset($insert_data['f_efficiency']);
    unset($insert_data['f_nc_percent']);
    unset($insert_data['f_production_efficiency']);
    unset($insert_data['f_nc']);
    unset($insert_data['nc_qty']);
    
    // 5. Build SQL Dynamically for all columns
    $cols = implode(", ", array_map(fn($col) => "`$col`", array_keys($insert_data)));
    $placeholders = implode(", ", array_fill(0, count($insert_data), "?"));
    $params = array_values($insert_data);

    try {
        $sql = "INSERT INTO production_logs ($cols) VALUES ($placeholders)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $logId = $pdo->lastInsertId();
        
        // --- DYNAMIC TOOL STROKE UPDATE (field_options) ---
        // If a tool was selected (tool_id) and quantity produced is > 0
        if (!empty($_POST['tool_id'])) {
            $toolId = intval($_POST['tool_id']); 
            
            // Try to get produced quantity from multiple possible field names
            $qty = 0;
            if (!empty($_POST['produced_qty'])) {
                $qty = intval($_POST['produced_qty']);
            } elseif (!empty($_POST['f_produced_qty'])) {
                $qty = intval($_POST['f_produced_qty']);
            }
            
            // Only update if qty > 0
            if ($qty > 0) {
                error_log("✅ TOOL STROKE UPDATE START: Tool ID = $toolId, Qty to Add = $qty");
                
                // Verify tool exists first
                $toolCheck = $pdo->prepare("SELECT id, label, current_usage, life_limit FROM field_options WHERE id = ?");
                $toolCheck->execute([$toolId]);
                $toolData = $toolCheck->fetch(PDO::FETCH_ASSOC);
                
                if ($toolData) {
                    $currentUsage = intval($toolData['current_usage'] ?: 0);
                    $toolLabel = $toolData['label'] ?: 'Unknown';
                    $maxLife = intval($toolData['life_limit'] ?: 100000);
                    
                    error_log("✅ Tool found: {$toolLabel}");
                    error_log("   Current usage BEFORE: {$currentUsage}");
                    error_log("   Max life: {$maxLife}");
                    
                    // Increase 'current_usage' by the produced quantity
                    $updateTool = $pdo->prepare("UPDATE field_options SET current_usage = current_usage + ? WHERE id = ?");
                    $updateResult = $updateTool->execute([$qty, $toolId]);
                    
                    if ($updateResult) {
                        // Verify the update
                        $toolVerify = $pdo->prepare("SELECT current_usage FROM field_options WHERE id = ? LIMIT 1");
                        $toolVerify->execute([$toolId]);
                        $verifyData = $toolVerify->fetch(PDO::FETCH_ASSOC);
                        $newUsage = intval($verifyData['current_usage'] ?: 0);
                        $remainingStrokes = max(0, $maxLife - $newUsage);
                        
                        error_log("✅ Tool stroke updated SUCCESSFULLY");
                        error_log("   Current usage AFTER: {$newUsage}");
                        error_log("   Strokes increased by: {$qty}");
                        error_log("   Remaining strokes: {$remainingStrokes}");
                    } else {
                        error_log("❌ FAILED to update tool stroke");
                    }
                } else {
                    error_log("❌ Tool ID not found in field_options: $toolId");
                }
            } else {
                error_log("⚠️ Produced Qty is 0 or not provided, skipping tool stroke update");
            }
        } else {
            error_log("⚠️ Tool ID not provided in production entry");
        }
        
        // Activity log: track which login user entered which employee (ID + name) and log date
        try {
            $sessionUsername = $_SESSION['user_id'] ?? '';
            $userIdForLog = null;
            if (!empty($sessionUsername)) {
                $userStmt = $pdo->prepare("SELECT id FROM users WHERE username = ? LIMIT 1");
                $userStmt->execute([$sessionUsername]);
                $userRow = $userStmt->fetch(PDO::FETCH_ASSOC);
                $userIdForLog = $userRow['id'] ?? null;
            }

            $loggedEmpId = trim($insert_data['emp_id'] ?? '');
            $loggedEmpName = trim($_POST['operator_name'] ?? $_POST['name'] ?? '');
            if ($loggedEmpName === '' && $loggedEmpId !== '') {
                $empStmt = $pdo->prepare("SELECT name FROM employees WHERE emp_unique_id = ? LIMIT 1");
                $empStmt->execute([$loggedEmpId]);
                $empRow = $empStmt->fetch(PDO::FETCH_ASSOC);
                $loggedEmpName = trim($empRow['name'] ?? '');
            }
            if ($loggedEmpName === '') {
                $loggedEmpName = 'Unknown';
            }

            $employeeDataLabel = trim($loggedEmpId . ' - ' . $loggedEmpName, " -");
            $productionDate = trim($insert_data['log_date'] ?? '');
            $activityDetails = $productionDate !== '' ? "Log Date: {$productionDate}" : '';

            $activityStmt = $pdo->prepare("INSERT INTO activity_log (user_id, username, employee_name, record_count, action_type, details) VALUES (?, ?, ?, ?, 'DATA_ENTRY', ?)");
            $activityStmt->execute([$userIdForLog, $sessionUsername, $employeeDataLabel, 1, $activityDetails]);
        } catch (Exception $logEx) {
            // Keep production save successful even if activity logging fails
            error_log('Activity log insert failed in save_log: ' . $logEx->getMessage());
        }

        echo json_encode(['success' => true, 'message' => 'Data saved successfully to database columns']);
    } catch (Exception $e) { 
        echo json_encode(['success' => false, 'error' => $e->getMessage()]); 
    }
    exit;
}

// ✅ NEW: GET LOGS FOR REPORT PAGE (with all dynamic columns - NO JSON)
if ($action === 'get_logs') {
    requireLogin();
    
    // Get all columns from production_logs table
    $sql = "SELECT * FROM production_logs ORDER BY log_date DESC, operator_name ASC";
    
    $stmt = $pdo->query($sql);
    echo json_encode($stmt->fetchAll());
    exit;
}

// ✅ NEW: GET CUSTOM FIELDS CONFIG FOR REPORT (dynamic headers)
if ($action === 'get_report_fields') {
    requireLogin();
    
    // Get all active custom fields with their order
    $sql = "SELECT id, field_label, field_name, field_type FROM custom_fields 
            WHERE is_active = 1 
            ORDER BY display_order ASC";
    
    $stmt = $pdo->query($sql);
    $fields = $stmt->fetchAll();
    
    echo json_encode($fields);
    exit;
}

if ($action === 'get_dashboard_stats') {
    requireLogin();
    $totalStaff = $pdo->query("SELECT COUNT(*) FROM employees")->fetchColumn();
    $totalProd = $pdo->query("SELECT COUNT(*) FROM production_logs")->fetchColumn(); 
    
    $stmtBar = $pdo->query("SELECT log_date, COUNT(*) as total_qty FROM production_logs WHERE log_date >= DATE_SUB(CURDATE(), INTERVAL 7 DAY) GROUP BY log_date ORDER BY log_date ASC");
    
    echo json_encode([
        'staff_count' => $totalStaff,
        'total_production' => $totalProd,
        'bar_chart' => $stmtBar->fetchAll(),
        'pie_chart' => [] 
    ]);
    exit;
}
// --- ACTION: SEARCH EMPLOYEE FULL (Profile + Logs) ---
if ($action == 'search_employee_full') {
    $term = $_GET['term'] ?? '';
    
    // 1. Find Employee Profile
    $stmt = $pdo->prepare("SELECT * FROM employees WHERE emp_unique_id = ? OR name LIKE ? OR id = ? LIMIT 1");
    $stmt->execute([$term, "%$term%", $term]);
    $profile = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$profile) {
        echo json_encode(['success' => false, 'message' => 'Not Found']);
        exit;
    }

    // 2. Find Employee Logs (Using the found employee's ID)
    // We join dynamic fields logic if you are storing them as JSON in a column, 
    // or if standard columns, just select *.
    
    // Assuming 'logs' table stores 'emp_id' which links to employee 'emp_unique_id'
    $logStmt = $pdo->prepare("SELECT * FROM production_logs WHERE emp_id = ? ORDER BY log_date DESC LIMIT 50");
    $logStmt->execute([$profile['emp_unique_id']]);
    $logs = $logStmt->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'success' => true, 
        'profile' => $profile, 
        'logs' => $logs
    ]);
    exit;
}
// --- ACTION: TOGGLE STATUS ---
if ($action == 'toggle_status') {
    requireLogin();

    $currentRole = strtolower(trim($_SESSION['role'] ?? ''));
    if (!in_array($currentRole, ['administrator', 'admin', 'supervisor'])) {
        echo json_encode(['success' => false, 'error' => 'Access denied']);
        exit;
    }

    $id = $_POST['id'];
    $newStatus = $_POST['status']; // 'Active' or 'Inactive'

    try {
        $stmt = $pdo->prepare("UPDATE employees SET status = ? WHERE emp_unique_id = ?");
        $stmt->execute([$newStatus, $id]);
        echo json_encode(['success' => true]);
    } catch (PDOException $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}
// --- ADD THIS BLOCK TO api.php ---

if ($action === 'update_custom_field' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    
    $id = $_POST['id'];
    $label = $_POST['label'];
    $type = $_POST['type'];
    $parentId = !empty($_POST['parent_id']) ? $_POST['parent_id'] : NULL;

    try {
        // Update the metadata
        $stmt = $pdo->prepare("UPDATE custom_fields SET field_label = ?, field_type = ?, parent_field_id = ? WHERE id = ?");
        $stmt->execute([$label, $type, $parentId, $id]);
        
        echo json_encode(['success' => true]);
    } catch (Exception $e) { 
        echo json_encode(['success' => false, 'error' => $e->getMessage()]); 
    }
    exit;
}
// --- SETTINGS: GET & UPDATE TIME LIMIT ---
if ($action === 'get_delete_limit') {
    requireLogin();
    $stmt = $pdo->query("SELECT setting_value FROM system_settings WHERE setting_key = 'delete_limit_hours'");
    $limit = $stmt->fetchColumn() ?: 24; // Default to 24 if missing
    echo json_encode(['hours' => $limit]);
    exit;
}

if ($action === 'update_delete_limit') {
    requireAdmin();
    $hours = intval($_POST['hours']);
    $stmt = $pdo->prepare("INSERT INTO system_settings (setting_key, setting_value) VALUES ('delete_limit_hours', ?) ON DUPLICATE KEY UPDATE setting_value = ?");
    $stmt->execute([$hours, $hours]);
    echo json_encode(['success' => true]);
    exit;
}

// --- SINGLE DELETE (With Time Logic) ---
if ($action === 'delete_log_entry') {
    requireLogin();
    $id = intval($_POST['id'] ?? 0);
    
    if ($id <= 0) {
        echo json_encode(['success' => false, 'message' => 'Invalid entry ID']);
        exit;
    }
    
    try {
        // 1. Get the full log row (schema may vary across deployments)
        $stmt = $pdo->prepare("SELECT * FROM production_logs WHERE id = ?");
        $stmt->execute([$id]);
        $log = $stmt->fetch();

        if (!$log) {
            echo json_encode(['success' => false, 'message' => 'Entry not found']);
            exit;
        }

        // 2. Admin can delete anything anytime
        $currentRole = strtolower(trim($_SESSION['role'] ?? ''));
        if (in_array($currentRole, ['administrator', 'admin'], true)) {
            $delStmt = $pdo->prepare("DELETE FROM production_logs WHERE id = ?");
            $delStmt->execute([$id]);
            echo json_encode(['success' => true, 'message' => 'Entry deleted successfully']);
            exit;
        }

        // 3. Regular users have time limits
        $limitStmt = $pdo->query("SELECT setting_value FROM system_settings WHERE setting_key = 'delete_limit_hours'");
        $limitHours = max(0, intval($limitStmt->fetchColumn() ?: 24));

        // Calculate age using best available timestamp column
        $createdTime = false;
        if (!empty($log['created_at'])) {
            $createdTime = strtotime((string)$log['created_at']);
        }
        if ($createdTime === false && !empty($log['log_date'])) {
            $timePart = trim((string)($log['log_time'] ?? $log['start_time'] ?? ''));
            if ($timePart === '') {
                $timePart = '00:00:00';
            }
            $createdTime = strtotime(trim($log['log_date'] . ' ' . $timePart));
        }
        if ($createdTime === false) {
            echo json_encode(['success' => false, 'message' => 'Cannot determine entry date/time']);
            exit;
        }
        
        $currentTime = time();
        $hoursDiff = ($currentTime - $createdTime) / 3600;
        if ($hoursDiff < 0) {
            $hoursDiff = 0;
        }

        if ($hoursDiff > $limitHours) {
            $hoursOld = round($hoursDiff, 1);
            echo json_encode(['success' => false, 'message' => "Time limit exceeded. Entry is $hoursOld hours old (limit: $limitHours hours)"]);
            exit;
        }

        // 4. Delete the entry
        $delStmt = $pdo->prepare("DELETE FROM production_logs WHERE id = ?");
        $delStmt->execute([$id]);
        echo json_encode(['success' => true, 'message' => 'Entry deleted successfully']);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Delete error: ' . $e->getMessage()]);
    }
    exit;
}

// --- BULK DELETE (With Password Verification) ---
if ($action === 'delete_all_data') {
    requireAdmin();
    $password = $_POST['password'];
    $adminUser = $_SESSION['user_id'];

    // 1. Verify Password
    $stmt = $pdo->prepare("SELECT password FROM users WHERE username = ?");
    $stmt->execute([$adminUser]);
    $user = $stmt->fetch();

    // Note: Use password_verify($password, $user['password']) if you use hashes.
    // For simple text (based on your previous code):
    if ($user && $user['password'] === $password) {
        // 2. Delete All
        $pdo->exec("TRUNCATE TABLE production_logs"); // TRUNCATE wipes it clean and resets IDs
        echo json_encode(['success' => true]);
    } else {
        echo json_encode(['success' => false, 'error' => 'Incorrect Admin Password']);
    }
    exit;
}

// =========================================================
// 7. ANALYTICS & STATISTICS ENDPOINTS
// =========================================================

// GET FIELD DATA STATISTICS
if ($action === 'get_field_statistics') {
    requireLogin();
    
    $fieldName = $_GET['field_name'] ?? '';
    if (!$fieldName) {
        echo json_encode(['success' => false, 'error' => 'Field name required']);
        exit;
    }
    
    try {
        // Get the field info
        $stmt = $pdo->prepare("SELECT field_label, field_type FROM custom_fields WHERE field_name = ?");
        $stmt->execute([$fieldName]);
        $field = $stmt->fetch();
        
        if (!$field) {
            if (hasProductionLogColumn($pdo, $fieldName)) {
                $field = [
                    'field_label' => ucwords(str_replace('_', ' ', $fieldName)),
                    'field_type' => 'text'
                ];
            } else {
                echo json_encode(['success' => false, 'error' => 'Field not found']);
                exit;
            }
        }

        $qtyColumn = resolveProductionQtyColumn($pdo);
        $durationColumn = resolveDurationColumn($pdo);
        $qtyExpr = numericSumExpression($qtyColumn);
        $durationExpr = numericSumExpression($durationColumn);
        
        // Get statistics for this field
        $statsQuery = "SELECT `$fieldName` as value,
                              COUNT(*) as count,
                              $qtyExpr as total_production_qty,
                              $durationExpr as total_hours
                       FROM production_logs 
                      WHERE `$fieldName` IS NOT NULL AND `$fieldName` != '' 
                      GROUP BY `$fieldName` 
                      ORDER BY count DESC";
        
        $stmtStats = $pdo->query($statsQuery);
        $stats = $stmtStats->fetchAll();

        $totalRecords = 0;
        $totalProductionQty = 0.0;
        $totalHours = 0.0;
        foreach ($stats as &$row) {
            $row['count'] = (int)($row['count'] ?? 0);
            $row['total_production_qty'] = (float)($row['total_production_qty'] ?? 0);
            $row['total_hours'] = (float)($row['total_hours'] ?? 0);
            $totalRecords += $row['count'];
            $totalProductionQty += $row['total_production_qty'];
            $totalHours += $row['total_hours'];
        }
        unset($row);
        
        echo json_encode([
            'success' => true,
            'field_label' => $field['field_label'],
            'field_type' => $field['field_type'],
            'data' => $stats,
            'total_records' => (int)$totalRecords,
            'total_production_qty' => (float)$totalProductionQty,
            'total_hours' => (float)$totalHours
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET ALL ACTIVE FIELDS FOR DROPDOWN
if ($action === 'get_analytics_fields') {
    requireLogin();
    
    try {
        $stmt = $pdo->query("SELECT id, field_name, field_label FROM custom_fields 
                            WHERE is_active = 1 
                            ORDER BY display_order ASC");
        $fields = $stmt->fetchAll();
        
        echo json_encode([
            'success' => true,
            'fields' => $fields
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET EMPLOYEE-MACHINE/TOOL MAPPING
if ($action === 'get_employee_machine_mapping') {
    requireLogin();
    
    $machineFieldName = $_GET['machine_field'] ?? '';
    
    try {
        // Get all employees with the machines/tools they worked with
        $query = "SELECT DISTINCT emp_id, operator_name, `$machineFieldName` as machine 
                 FROM production_logs 
                 WHERE emp_id IS NOT NULL AND `$machineFieldName` IS NOT NULL 
                 ORDER BY emp_id, machine";
        
        $stmt = $pdo->query($query);
        $mappings = $stmt->fetchAll();
        
        // Group by employee
        $grouped = [];
        foreach ($mappings as $map) {
            if (!isset($grouped[$map['emp_id']])) {
                $grouped[$map['emp_id']] = [
                    'emp_id' => $map['emp_id'],
                    'operator_name' => $map['operator_name'],
                    'machines' => []
                ];
            }
            if ($map['machine']) {
                $grouped[$map['emp_id']]['machines'][] = $map['machine'];
            }
        }
        
        echo json_encode([
            'success' => true,
            'mappings' => array_values($grouped)
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET MACHINE/TOOL LIST WITH EMPLOYEE COUNT
if ($action === 'get_machine_tool_list') {
    requireLogin();
    
    $machineFieldName = $_GET['machine_field'] ?? '';
    if (!$machineFieldName) {
        echo json_encode(['success' => false, 'error' => 'Machine field is required']);
        exit;
    }
    
    try {
        $qtyColumn = resolveProductionQtyColumn($pdo);
        $durationColumn = resolveDurationColumn($pdo);
        $qtyExpr = numericSumExpression($qtyColumn);
        $durationExpr = numericSumExpression($durationColumn);

        // Get all unique machines and include production + hour totals
        $query = "SELECT `$machineFieldName` as machine, 
                        COUNT(DISTINCT emp_id) as employee_count,
                        COUNT(*) as total_operations,
                        $qtyExpr as total_production_qty,
                        $durationExpr as total_hours
                 FROM production_logs 
                 WHERE `$machineFieldName` IS NOT NULL AND `$machineFieldName` != ''
                 GROUP BY `$machineFieldName`
                 ORDER BY total_production_qty DESC, total_operations DESC";
        
        $stmt = $pdo->query($query);
        $machines = $stmt->fetchAll();
        foreach ($machines as &$machine) {
            $machine['total_operations'] = (int)($machine['total_operations'] ?? 0);
            $machine['employee_count'] = (int)($machine['employee_count'] ?? 0);
            $machine['total_production_qty'] = (float)($machine['total_production_qty'] ?? 0);
            $machine['total_hours'] = (float)($machine['total_hours'] ?? 0);
        }
        unset($machine);
        
        echo json_encode([
            'success' => true,
            'machines' => $machines
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET DETAILED ANALYTICS FOR DASHBOARD
if ($action === 'get_analytics_dashboard') {
    requireLogin();
    
    try {
        $qtyColumn = resolveProductionQtyColumn($pdo);
        $durationColumn = resolveDurationColumn($pdo);
        $qtyExpr = numericSumExpression($qtyColumn);
        $durationExpr = numericSumExpression($durationColumn);

        // 1. Total records
        $totalRecords = $pdo->query("SELECT COUNT(*) FROM production_logs")->fetchColumn();
        $totalProductionQty = $pdo->query("SELECT $qtyExpr FROM production_logs")->fetchColumn();
        $totalHours = $pdo->query("SELECT $durationExpr FROM production_logs")->fetchColumn();
        
        // 2. Employees with work data
        $employeeCount = $pdo->query("SELECT COUNT(DISTINCT emp_id) FROM production_logs")->fetchColumn();
        
        // 3. Daily stats by date (last 30 days)
        $dateQuery = "SELECT log_date,
                             COUNT(*) as count,
                             COUNT(*) as total_records,
                             $qtyExpr as total_quantity,
                             $durationExpr as total_hours
                      FROM production_logs 
                     WHERE log_date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
                     GROUP BY log_date
                     ORDER BY log_date ASC";
        $dateStats = $pdo->query($dateQuery)->fetchAll();
        foreach ($dateStats as &$day) {
            $day['count'] = (int)($day['count'] ?? 0);
            $day['total_records'] = (int)($day['total_records'] ?? 0);
            $day['total_quantity'] = (float)($day['total_quantity'] ?? 0);
            $day['total_hours'] = (float)($day['total_hours'] ?? 0);
        }
        unset($day);
        
        // 4. Top 10 employees by total production
        $topEmployeesQuery = "SELECT emp_id,
                                     operator_name,
                                     COUNT(*) as count,
                                     $qtyExpr as total_qty,
                                     $durationExpr as total_hours
                              FROM production_logs
                              WHERE emp_id IS NOT NULL
                              GROUP BY emp_id, operator_name
                              ORDER BY total_qty DESC, count DESC
                              LIMIT 10";
        $topEmployees = $pdo->query($topEmployeesQuery)->fetchAll();
        foreach ($topEmployees as &$employee) {
            $employee['count'] = (int)($employee['count'] ?? 0);
            $employee['total_qty'] = (float)($employee['total_qty'] ?? 0);
            $employee['total_hours'] = (float)($employee['total_hours'] ?? 0);
        }
        unset($employee);
        
        echo json_encode([
            'success' => true,
            'total_records' => (int)$totalRecords,
            'total_production_qty' => (float)$totalProductionQty,
            'total_quantity' => (float)$totalProductionQty,
            'total_hours' => (float)$totalHours,
            'employee_count' => (int)$employeeCount,
            'date_stats' => $dateStats,
            'top_employees' => $topEmployees
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET DATE-WISE STATISTICS (NEW ENDPOINT)
if ($action === 'get_date_wise_statistics') {
    requireLogin();
    
    try {
        $startDate = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
        $endDate = $_GET['end_date'] ?? date('Y-m-d');
        
        // Validate dates
        $startDate = date('Y-m-d', strtotime($startDate));
        $endDate = date('Y-m-d', strtotime($endDate));
        
        $query = "
            SELECT 
                log_date,
                COUNT(*) as total_records,
                COUNT(DISTINCT emp_id) as employee_count,
                MAX(HOUR(log_time)) as peak_hour
            FROM production_logs 
            WHERE log_date >= ? AND log_date <= ?
            GROUP BY log_date
            ORDER BY log_date ASC
        ";
        
        $stmt = $pdo->prepare($query);
        $stmt->execute([$startDate, $endDate]);
        $dateStats = $stmt->fetchAll();
        
        echo json_encode([
            'success' => true,
            'data' => $dateStats,
            'date_range' => [
                'start' => $startDate,
                'end' => $endDate
            ]
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET ALL DATES STATISTICS (ALL AVAILABLE DATES)
if ($action === 'get_all_dates_statistics') {
    requireLogin();
    
    try {
        $query = "
            SELECT 
                log_date,
                COUNT(*) as total_records,
                COUNT(DISTINCT emp_id) as employee_count,
                COUNT(DISTINCT CASE 
                    WHEN EXISTS (
                        SELECT 1 FROM custom_fields 
                        WHERE (field_name LIKE '%tool%' OR field_label LIKE '%tool%') 
                        AND is_active = 1 LIMIT 1
                    ) THEN 1 
                END) as tools_count
            FROM production_logs 
            GROUP BY log_date
            ORDER BY log_date DESC
        ";
        
        $stmt = $pdo->query($query);
        $dateStats = $stmt->fetchAll();
        
        echo json_encode([
            'success' => true,
            'data' => $dateStats
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET DAILY PRODUCTION DETAILS (All machines, employees, tools for a specific date)
if ($action === 'get_daily_production_details') {
    requireLogin();
    
    $logDate = $_GET['log_date'] ?? date('Y-m-d');
    
    if (!$logDate) {
        echo json_encode(['success' => false, 'error' => 'Log date required']);
        exit;
    }
    
    try {
        // Find machine and tool fields
        $fieldStmt = $pdo->prepare("SELECT field_name FROM custom_fields WHERE field_name LIKE ? OR field_label LIKE ? LIMIT 1");
        
        // Find machine field
        $fieldStmt->execute(['%machine%', '%machine%']);
        $machineFieldResult = $fieldStmt->fetch();
        $machineField = $machineFieldResult ? $machineFieldResult['field_name'] : null;
        
        // Find tool field
        $fieldStmt->execute(['%tool%', '%tool%']);
        $toolFieldResult = $fieldStmt->fetch();
        $toolField = $toolFieldResult ? $toolFieldResult['field_name'] : null;
        
        // Get total operations for this date
        $totalOpsQuery = "SELECT COUNT(*) as total_ops FROM production_logs WHERE log_date = ?";
        $stmtTotalOps = $pdo->prepare($totalOpsQuery);
        $stmtTotalOps->execute([$logDate]);
        $totalOps = $stmtTotalOps->fetch()['total_ops'] ?? 0;
        
        // Get machines breakdown (if machine field exists)
        $machines = [];
        if ($machineField) {
            $machinesQuery = "
                SELECT `$machineField` as machine_name, COUNT(*) as count 
                FROM production_logs 
                WHERE log_date = ? AND `$machineField` IS NOT NULL AND `$machineField` != ''
                GROUP BY `$machineField` 
                ORDER BY count DESC
            ";
            $stmtMachines = $pdo->prepare($machinesQuery);
            $stmtMachines->execute([$logDate]);
            $machines = $stmtMachines->fetchAll();
        }
        
        // Get tools breakdown (if tool field exists)
        $tools = [];
        if ($toolField) {
            $toolsQuery = "
                SELECT `$toolField` as tool_name, COUNT(*) as count 
                FROM production_logs 
                WHERE log_date = ? AND `$toolField` IS NOT NULL AND `$toolField` != ''
                GROUP BY `$toolField` 
                ORDER BY count DESC
            ";
            $stmtTools = $pdo->prepare($toolsQuery);
            $stmtTools->execute([$logDate]);
            $tools = $stmtTools->fetchAll();
        }
        
        // Get employees breakdown
        $empQuery = "
            SELECT emp_id, operator_name, COUNT(*) as count 
            FROM production_logs 
            WHERE log_date = ? AND emp_id IS NOT NULL
            GROUP BY emp_id 
            ORDER BY count DESC
        ";
        $stmtEmp = $pdo->prepare($empQuery);
        $stmtEmp->execute([$logDate]);
        $employees = $stmtEmp->fetchAll();
        
        echo json_encode([
            'success' => true,
            'total_operations' => (int)$totalOps,
            'machines' => $machines,
            'tools' => $tools,
            'employees' => $employees
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET MACHINE DETAIL ANALYSIS (NEW ENDPOINT)
if ($action === 'get_machine_detail_analysis') {
    requireLogin();
    
    $machineField = $_GET['machine_field'] ?? '';
    $machineName = $_GET['machine_name'] ?? '';
    
    if (!$machineField || !$machineName) {
        echo json_encode(['success' => false, 'error' => 'Required parameters missing']);
        exit;
    }
    
    try {
        $toolField = resolveToolColumn($pdo);
        $qtyColumn = resolveProductionQtyColumn($pdo);
        $acceptedColumn = resolveAcceptedQtyColumn($pdo);
        $durationColumn = resolveDurationColumn($pdo);
        $qtyExpr = numericSumExpression($qtyColumn);
        $acceptedExpr = numericSumExpression($acceptedColumn);
        $durationExpr = numericSumExpression($durationColumn);
        $ncExpr = '0';
        if ($qtyColumn && $acceptedColumn) {
            $ncExpr = "COALESCE(SUM(GREATEST(CAST(NULLIF(`$qtyColumn`, '') AS DECIMAL(18,2)) - CAST(NULLIF(`$acceptedColumn`, '') AS DECIMAL(18,2)), 0)), 0)";
        }

        // 1. Machine-level totals
        $summaryQuery = "
            SELECT 
                COUNT(*) as total_operations,
                COUNT(DISTINCT emp_id) as employee_count,
                $qtyExpr as total_production_qty,
                $durationExpr as total_hours,
                $acceptedExpr as total_accepted_qty,
                $ncExpr as total_nc_qty,
                CASE 
                    WHEN $qtyExpr > 0 THEN ROUND(($acceptedExpr / $qtyExpr) * 100, 2)
                    ELSE 0
                END as acceptance_rate
            FROM production_logs
            WHERE `$machineField` = ?
        ";
        $stmtSummary = $pdo->prepare($summaryQuery);
        $stmtSummary->execute([$machineName]);
        $summary = $stmtSummary->fetch() ?: [];
        
        // 2. Count of distinct tools used in this machine
        $toolsCount = 0;
        if ($toolField) {
            $toolsQuery = "SELECT COUNT(DISTINCT `$toolField`) as tools_count FROM production_logs WHERE `$machineField` = ? AND `$toolField` IS NOT NULL AND `$toolField` != ''";
            $stmtTools = $pdo->prepare($toolsQuery);
            $stmtTools->execute([$machineName]);
            $toolsCount = (int)($stmtTools->fetch()['tools_count'] ?? 0);
        }
        
        // 3. Daily averages (operations + production + hours)
        $avgDailyQuery = "
            SELECT 
                COALESCE(AVG(daily_count), 0) as avg_daily_operations,
                COALESCE(AVG(daily_qty), 0) as avg_daily_production,
                COALESCE(AVG(daily_hours), 0) as avg_daily_hours
            FROM (
                SELECT 
                    COUNT(*) as daily_count,
                    $qtyExpr as daily_qty,
                    $durationExpr as daily_hours
                FROM production_logs
                WHERE `$machineField` = ?
                GROUP BY log_date
            ) as daily_stats
        ";
        $stmtAvgDaily = $pdo->prepare($avgDailyQuery);
        $stmtAvgDaily->execute([$machineName]);
        $avgDaily = $stmtAvgDaily->fetch() ?: [];
        
        // 4. Tool-wise breakdown inside this machine
        $toolsBreakdown = [];
        if ($toolField) {
            $toolsBreakdownQuery = "
                SELECT 
                    `$toolField` as tool_name,
                    COUNT(*) as count,
                    $qtyExpr as total_production_qty,
                    $durationExpr as total_hours
                FROM production_logs
                WHERE `$machineField` = ? AND `$toolField` IS NOT NULL AND `$toolField` != ''
                GROUP BY `$toolField`
                ORDER BY total_production_qty DESC, count DESC
            ";
            $stmtToolsBreakdown = $pdo->prepare($toolsBreakdownQuery);
            $stmtToolsBreakdown->execute([$machineName]);
            $toolsBreakdown = $stmtToolsBreakdown->fetchAll();
            foreach ($toolsBreakdown as &$tool) {
                $tool['count'] = (int)($tool['count'] ?? 0);
                $tool['total_production_qty'] = (float)($tool['total_production_qty'] ?? 0);
                $tool['total_hours'] = (float)($tool['total_hours'] ?? 0);
            }
            unset($tool);
        }
        
        // 5. Employee-wise breakdown inside this machine
        $empBreakdownQuery = "
            SELECT 
                emp_id,
                operator_name,
                COUNT(*) as count,
                $qtyExpr as total_production_qty,
                $durationExpr as total_hours
            FROM production_logs
            WHERE `$machineField` = ? AND emp_id IS NOT NULL
            GROUP BY emp_id, operator_name
            ORDER BY total_production_qty DESC, count DESC
        ";
        $stmtEmpBreakdown = $pdo->prepare($empBreakdownQuery);
        $stmtEmpBreakdown->execute([$machineName]);
        $empBreakdown = $stmtEmpBreakdown->fetchAll();
        foreach ($empBreakdown as &$employee) {
            $employee['count'] = (int)($employee['count'] ?? 0);
            $employee['total_production_qty'] = (float)($employee['total_production_qty'] ?? 0);
            $employee['total_hours'] = (float)($employee['total_hours'] ?? 0);
            $employee['total_operations'] = $employee['count'];
        }
        unset($employee);
        
        // 6. Date-wise breakdown for chart/table
        $toolsCountSelect = $toolField ? "COUNT(DISTINCT `$toolField`) as tools_count" : "0 as tools_count";
        $dailyQuery = "
            SELECT 
                log_date,
                COUNT(*) as total_operations,
                COUNT(DISTINCT emp_id) as employee_count,
                $toolsCountSelect,
                $qtyExpr as total_production_qty,
                $durationExpr as total_hours,
                $acceptedExpr as total_accepted_qty,
                $ncExpr as total_nc_qty,
                CASE 
                    WHEN $qtyExpr > 0 THEN ROUND(($acceptedExpr / $qtyExpr) * 100, 2)
                    ELSE 0
                END as acceptance_rate
            FROM production_logs 
            WHERE `$machineField` = ?
            GROUP BY log_date
            ORDER BY log_date DESC
            LIMIT 30
        ";
        $stmtDaily = $pdo->prepare($dailyQuery);
        $stmtDaily->execute([$machineName]);
        $dailyData = array_reverse($stmtDaily->fetchAll()); // Reverse to show oldest first
        foreach ($dailyData as &$day) {
            $day['total_operations'] = (int)($day['total_operations'] ?? 0);
            $day['employee_count'] = (int)($day['employee_count'] ?? 0);
            $day['tools_count'] = (int)($day['tools_count'] ?? 0);
            $day['total_production_qty'] = (float)($day['total_production_qty'] ?? 0);
            $day['total_hours'] = (float)($day['total_hours'] ?? 0);
            $day['total_accepted_qty'] = (float)($day['total_accepted_qty'] ?? 0);
            $day['total_nc_qty'] = (float)($day['total_nc_qty'] ?? 0);
            $day['acceptance_rate'] = (float)($day['acceptance_rate'] ?? 0);
        }
        unset($day);

        $totalOps = (int)($summary['total_operations'] ?? 0);
        $empCount = (int)($summary['employee_count'] ?? 0);
        $totalProductionQty = (float)($summary['total_production_qty'] ?? 0);
        $totalHours = (float)($summary['total_hours'] ?? 0);
        $totalAcceptedQty = (float)($summary['total_accepted_qty'] ?? 0);
        $totalNcQty = (float)($summary['total_nc_qty'] ?? 0);
        $acceptanceRate = (float)($summary['acceptance_rate'] ?? 0);
        $avgDailyOperations = (float)($avgDaily['avg_daily_operations'] ?? 0);
        $avgDailyProduction = (float)($avgDaily['avg_daily_production'] ?? 0);
        $avgDailyHours = (float)($avgDaily['avg_daily_hours'] ?? 0);
        
        echo json_encode([
            'success' => true,
            'summary' => [
                'total_operations' => $totalOps,
                'total_production_qty' => $totalProductionQty,
                'total_hours' => $totalHours,
                'total_accepted_qty' => $totalAcceptedQty,
                'total_nc_qty' => $totalNcQty,
                'acceptance_rate' => $acceptanceRate,
                'employee_count' => $empCount,
                'tools_count' => (int)$toolsCount,
                'avg_daily_operations' => $avgDailyOperations,
                'avg_daily_production' => $avgDailyProduction,
                'avg_daily_hours' => $avgDailyHours
            ],
            'tools' => $toolsBreakdown,
            'employees' => $empBreakdown,
            'daily_data' => $dailyData
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET ALL FIELDS (FOR HIERARCHICAL SELECTION)
if ($action === 'get_all_fields') {
    requireLogin();
    
    try {
        $stmt = $pdo->query("SELECT id, field_name, field_label, field_type, parent_field_id FROM custom_fields 
                            WHERE is_active = 1 
                            ORDER BY display_order ASC");
        $fields = $stmt->fetchAll();
        
        // Organize into parent-child relationships
        $parentFields = [];
        $childFields = [];
        
        foreach ($fields as $field) {
            if ($field['parent_field_id'] === NULL || $field['parent_field_id'] == '') {
                $parentFields[] = $field;
            } else {
                if (!isset($childFields[$field['parent_field_id']])) {
                    $childFields[$field['parent_field_id']] = [];
                }
                $childFields[$field['parent_field_id']][] = $field;
            }
        }
        
        echo json_encode([
            'success' => true,
            'parent_fields' => $parentFields,
            'child_fields' => $childFields,
            'all_fields' => $fields
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET CHILD FIELDS FOR A PARENT (DEPENDENT FIELDS)
if ($action === 'get_dependent_fields') {
    requireLogin();
    
    $parentFieldId = $_GET['parent_field_id'] ?? '';
    
    if (!$parentFieldId) {
        echo json_encode(['success' => false, 'error' => 'Parent field ID required']);
        exit;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT id, field_name, field_label, field_type FROM custom_fields 
                             WHERE parent_field_id = ? AND is_active = 1 
                             ORDER BY display_order ASC");
        $stmt->execute([$parentFieldId]);
        $dependentFields = $stmt->fetchAll();
        
        echo json_encode([
            'success' => true,
            'dependent_fields' => $dependentFields
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET STATISTICS FOR DEPENDENT FIELDS FILTERED BY MACHINE VALUE
if ($action === 'get_machine_specific_stats') {
    requireLogin();
    
    $machineFieldName = $_GET['machine_field'] ?? '';
    $machineValue = $_GET['machine_value'] ?? '';
    $dependentFieldName = $_GET['dependent_field'] ?? '';
    
    if (!$machineFieldName || !$machineValue || !$dependentFieldName) {
        echo json_encode(['success' => false, 'error' => 'Required parameters missing']);
        exit;
    }
    
    try {
        // Get field info
        $stmt = $pdo->prepare("SELECT field_label, field_type FROM custom_fields WHERE field_name = ?");
        $stmt->execute([$dependentFieldName]);
        $field = $stmt->fetch();
        
        if (!$field) {
            if (hasProductionLogColumn($pdo, $dependentFieldName)) {
                $field = [
                    'field_label' => ucwords(str_replace('_', ' ', $dependentFieldName)),
                    'field_type' => 'text'
                ];
            } else {
                echo json_encode(['success' => false, 'error' => 'Field not found']);
                exit;
            }
        }

        $qtyColumn = resolveProductionQtyColumn($pdo);
        $durationColumn = resolveDurationColumn($pdo);
        $qtyExpr = numericSumExpression($qtyColumn);
        $durationExpr = numericSumExpression($durationColumn);
        
        // Get statistics for this field WHERE machine = specific value
        $statsQuery = "SELECT 
                        `$dependentFieldName` as value,
                        COUNT(*) as count,
                        $qtyExpr as total_production_qty,
                        $durationExpr as total_hours
                      FROM production_logs 
                      WHERE `$machineFieldName` = ? 
                      AND `$dependentFieldName` IS NOT NULL 
                      AND `$dependentFieldName` != '' 
                      GROUP BY `$dependentFieldName` 
                      ORDER BY total_production_qty DESC, count DESC";
        
        $stmtStats = $pdo->prepare($statsQuery);
        $stmtStats->execute([$machineValue]);
        $stats = $stmtStats->fetchAll();
        foreach ($stats as &$row) {
            $row['count'] = (int)($row['count'] ?? 0);
            $row['total_production_qty'] = (float)($row['total_production_qty'] ?? 0);
            $row['total_hours'] = (float)($row['total_hours'] ?? 0);
        }
        unset($row);
        
        // Also get totals for this machine
        $totalQuery = "SELECT 
                            COUNT(*) as total,
                            $qtyExpr as total_production_qty,
                            $durationExpr as total_hours
                       FROM production_logs 
                      WHERE `$machineFieldName` = ?";
        $stmtTotal = $pdo->prepare($totalQuery);
        $stmtTotal->execute([$machineValue]);
        $totalResult = $stmtTotal->fetch();
        
        // Get employee details for this machine
        $empQuery = "SELECT 
                            emp_id,
                            operator_name,
                            COUNT(*) as count,
                            COUNT(DISTINCT log_date) as working_days,
                            $qtyExpr as total_production_qty,
                            $durationExpr as total_hours
                    FROM production_logs 
                    WHERE `$machineFieldName` = ? 
                    AND emp_id IS NOT NULL 
                    GROUP BY emp_id, operator_name
                    ORDER BY total_production_qty DESC, count DESC";
        $stmtEmp = $pdo->prepare($empQuery);
        $stmtEmp->execute([$machineValue]);
        $employees = $stmtEmp->fetchAll();
        foreach ($employees as &$employee) {
            $employee['count'] = (int)($employee['count'] ?? 0);
            $employee['working_days'] = (int)($employee['working_days'] ?? 0);
            $employee['total_production_qty'] = (float)($employee['total_production_qty'] ?? 0);
            $employee['total_hours'] = (float)($employee['total_hours'] ?? 0);
            $employee['total_operations'] = $employee['count'];
        }
        unset($employee);
        
        echo json_encode([
            'success' => true,
            'field_label' => $field['field_label'],
            'field_type' => $field['field_type'],
            'machine_value' => $machineValue,
            'data' => $stats,
            'total_records' => (int)($totalResult['total'] ?? 0),
            'total_production_qty' => (float)($totalResult['total_production_qty'] ?? 0),
            'total_hours' => (float)($totalResult['total_hours'] ?? 0),
            'employees_working' => $employees,
            'employee_count' => count($employees)
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET ALL MACHINES WITH THEIR DATA SUMMARY
if ($action === 'get_machine_summary') {
    requireLogin();
    
    $machineFieldName = $_GET['machine_field'] ?? '';
    
    if (!$machineFieldName) {
        echo json_encode(['success' => false, 'error' => 'Machine field name required']);
        exit;
    }
    
    try {
        $qtyColumn = resolveProductionQtyColumn($pdo);
        $durationColumn = resolveDurationColumn($pdo);
        $qtyExpr = numericSumExpression($qtyColumn);
        $durationExpr = numericSumExpression($durationColumn);

        $query = "SELECT `$machineFieldName` as machine_name,
                        `$machineFieldName` as machine_value,
                        COUNT(*) as total_operations,
                        $qtyExpr as total_production_qty,
                        $durationExpr as total_hours,
                        COUNT(DISTINCT emp_id) as employee_count,
                        COUNT(DISTINCT log_date) as working_days,
                        MIN(log_date) as first_used,
                        MAX(log_date) as last_used,
                        ROUND(($qtyExpr) / NULLIF(COUNT(DISTINCT log_date), 0), 2) as avg_daily_production
                 FROM production_logs 
                 WHERE `$machineFieldName` IS NOT NULL AND `$machineFieldName` != ''
                 GROUP BY `$machineFieldName`
                 ORDER BY total_production_qty DESC, total_operations DESC";
        
        $stmt = $pdo->query($query);
        $machines = $stmt->fetchAll();
        foreach ($machines as &$machine) {
            $machine['total_operations'] = (int)($machine['total_operations'] ?? 0);
            $machine['total_production_qty'] = (float)($machine['total_production_qty'] ?? 0);
            $machine['total_hours'] = (float)($machine['total_hours'] ?? 0);
            $machine['employee_count'] = (int)($machine['employee_count'] ?? 0);
            $machine['working_days'] = (int)($machine['working_days'] ?? 0);
            $machine['avg_daily_production'] = (float)($machine['avg_daily_production'] ?? 0);
        }
        unset($machine);
        
        echo json_encode([
            'success' => true,
            'machines' => $machines
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET MACHINE EMPLOYEES WITH DETAILED BREAKDOWN
if ($action === 'get_machine_employees_detailed') {
    requireLogin();
    
    $machineFieldName = $_GET['machine_field'] ?? '';
    $machineValue = $_GET['machine_value'] ?? '';
    
    if (!$machineFieldName || !$machineValue) {
        echo json_encode(['success' => false, 'error' => 'Machine field and value required']);
        exit;
    }
    
    try {
        $qtyColumn = resolveProductionQtyColumn($pdo);
        $durationColumn = resolveDurationColumn($pdo);
        $qtyExpr = numericSumExpression($qtyColumn);
        $durationExpr = numericSumExpression($durationColumn);

        // Get employees with detailed statistics
        $query = "SELECT 
                    emp_id,
                    operator_name,
                    COUNT(*) as total_operations,
                    $qtyExpr as total_production_qty,
                    $durationExpr as total_hours,
                    COUNT(DISTINCT log_date) as working_days,
                    MIN(log_date) as first_date,
                    MAX(log_date) as last_date,
                    ROUND(COUNT(*) / NULLIF(COUNT(DISTINCT log_date), 0), 2) as avg_ops_per_day,
                    ROUND(($qtyExpr) / NULLIF(COUNT(DISTINCT log_date), 0), 2) as avg_production_per_day
                 FROM production_logs 
                 WHERE `$machineFieldName` = ? 
                 AND emp_id IS NOT NULL 
                 GROUP BY emp_id, operator_name
                 ORDER BY total_production_qty DESC, total_operations DESC";
        
        $stmt = $pdo->prepare($query);
        $stmt->execute([$machineValue]);
        $employees = $stmt->fetchAll();
        foreach ($employees as &$employee) {
            $employee['total_operations'] = (int)($employee['total_operations'] ?? 0);
            $employee['total_production_qty'] = (float)($employee['total_production_qty'] ?? 0);
            $employee['total_hours'] = (float)($employee['total_hours'] ?? 0);
            $employee['working_days'] = (int)($employee['working_days'] ?? 0);
            $employee['avg_ops_per_day'] = (float)($employee['avg_ops_per_day'] ?? 0);
            $employee['avg_production_per_day'] = (float)($employee['avg_production_per_day'] ?? 0);
        }
        unset($employee);
        
        echo json_encode([
            'success' => true,
            'machine_value' => $machineValue,
            'employees' => $employees,
            'total_employees' => count($employees)
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET PRODUCTION TREND BY DATE
if ($action === 'get_production_trend_by_date') {
    requireLogin();
    
    $machineFieldName = $_GET['machine_field'] ?? '';
    $machineValue = $_GET['machine_value'] ?? '';
    $requestedQuantityField = $_GET['quantity_field'] ?? '';
    
    if (!$machineFieldName || !$machineValue) {
        echo json_encode(['success' => false, 'error' => 'Machine field and value required']);
        exit;
    }
    
    try {
        $quantityField = hasProductionLogColumn($pdo, $requestedQuantityField)
            ? $requestedQuantityField
            : resolveProductionQtyColumn($pdo);
        $durationColumn = resolveDurationColumn($pdo);
        $qtyExpr = numericSumExpression($quantityField);
        $durationExpr = numericSumExpression($durationColumn);

        // Get daily production trend
        $query = "SELECT 
                    log_date,
                    log_date as date,
                    COUNT(*) as operations_count,
                    COUNT(DISTINCT emp_id) as employees_count,
                    $qtyExpr as total_quantity,
                    $durationExpr as total_hours
                 FROM production_logs 
                 WHERE `$machineFieldName` = ? 
                 AND log_date IS NOT NULL
                 GROUP BY log_date
                 ORDER BY log_date ASC";
        
        $stmt = $pdo->prepare($query);
        $stmt->execute([$machineValue]);
        $trends = $stmt->fetchAll();
        
        // Format data for chart
        $dates = [];
        $operations = [];
        $quantities = [];
        $hours = [];
        $employees = [];
        $totalQuantity = 0.0;
        $totalHours = 0.0;
        
        foreach ($trends as &$trend) {
            $trend['operations_count'] = (int)($trend['operations_count'] ?? 0);
            $trend['employees_count'] = (int)($trend['employees_count'] ?? 0);
            $trend['total_quantity'] = (float)($trend['total_quantity'] ?? 0);
            $trend['total_hours'] = (float)($trend['total_hours'] ?? 0);
            $trend['total_operations'] = $trend['operations_count'];

            $dates[] = $trend['log_date'];
            $operations[] = $trend['operations_count'];
            $quantities[] = $trend['total_quantity'];
            $hours[] = $trend['total_hours'];
            $employees[] = $trend['employees_count'];
            $totalQuantity += $trend['total_quantity'];
            $totalHours += $trend['total_hours'];
        }
        unset($trend);
        
        echo json_encode([
            'success' => true,
            'machine_value' => $machineValue,
            'quantity_field_used' => $quantityField,
            'dates' => $dates,
            'operations_count' => $operations,
            'total_quantity' => $quantities,
            'total_hours' => $hours,
            'employees_count' => $employees,
            'total_records' => count($trends),
            'total_operations' => array_sum($operations),
            'total_production_qty' => $totalQuantity,
            'total_hours_sum' => $totalHours,
            'data' => $trends
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// ===== ADVANCED ANALYTICS ENDPOINTS =====

// DAY-WISE PRODUCTION ANALYTICS
if ($action === 'get_daywise_analytics') {
    requireLogin();
    
    $sql = "SELECT 
        log_date as date,
        COUNT(*) as total_entries,
        SUM(CAST(plan_qty AS DECIMAL(10,2))) as total_plan_qty,
        SUM(CAST(produced_qty AS DECIMAL(10,2))) as total_produced_qty,
        SUM(CAST(accepted_qty AS DECIMAL(10,2))) as total_accepted_qty,
        SUM(CAST(produced_qty AS DECIMAL(10,2)) - CAST(accepted_qty AS DECIMAL(10,2))) as nc_qty
    FROM production_logs 
    WHERE log_date IS NOT NULL
    GROUP BY log_date 
    ORDER BY log_date DESC 
    LIMIT 30";
    
    $stmt = $pdo->query($sql);
    echo json_encode($stmt->fetchAll());
    exit;
}

// FIELD/TOOL-WISE PRODUCTION ANALYTICS
if ($action === 'get_fieldwise_analytics') {
    requireLogin();
    
    $sql = "SELECT 
        tool_no as field_name,
        COUNT(*) as total_entries,
        COUNT(DISTINCT operator_name) as worker_count,
        SUM(CAST(plan_qty AS DECIMAL(10,2))) as total_plan_qty,
        SUM(CAST(produced_qty AS DECIMAL(10,2))) as total_produced_qty,
        SUM(CAST(accepted_qty AS DECIMAL(10,2))) as total_accepted_qty,
        SUM(CAST(produced_qty AS DECIMAL(10,2)) - CAST(accepted_qty AS DECIMAL(10,2))) as nc_qty
    FROM production_logs 
    WHERE tool_no IS NOT NULL AND tool_no != ''
    GROUP BY tool_no 
    ORDER BY total_produced_qty DESC";
    
    $stmt = $pdo->query($sql);
    echo json_encode($stmt->fetchAll());
    exit;
}

// EMPLOYEE-WISE PRODUCTION ANALYTICS
if ($action === 'get_employeewise_analytics') {
    requireLogin();
    
    $sql = "SELECT 
        operator_name as employee_name,
        COUNT(*) as total_entries,
        COUNT(DISTINCT log_date) as working_days,
        COUNT(DISTINCT tool_no) as tools_handled,
        SUM(CAST(plan_qty AS DECIMAL(10,2))) as total_plan_qty,
        SUM(CAST(produced_qty AS DECIMAL(10,2))) as total_produced_qty,
        SUM(CAST(accepted_qty AS DECIMAL(10,2))) as total_accepted_qty,
        SUM(CAST(produced_qty AS DECIMAL(10,2)) - CAST(accepted_qty AS DECIMAL(10,2))) as nc_qty,
        ROUND(SUM(CAST(accepted_qty AS DECIMAL(10,2))) / NULLIF(SUM(CAST(produced_qty AS DECIMAL(10,2))), 0) * 100, 2) as quality_rate
    FROM production_logs 
    WHERE operator_name IS NOT NULL AND operator_name != ''
    GROUP BY operator_name 
    ORDER BY total_produced_qty DESC";
    
    $stmt = $pdo->query($sql);
    echo json_encode($stmt->fetchAll());
    exit;
}

// TOOL-WISE WORKER PRODUCTION (detailed breakdown)
if ($action === 'get_tool_worker_analytics') {
    requireLogin();
    
    $tool = trim((string)($_GET['tool'] ?? ''));
    if (!$tool) {
        echo json_encode([]);
        exit;
    }

    $toolColumn = resolveToolColumn($pdo);
    $planColumn = resolvePlanQtyColumn($pdo);
    $qtyColumn = resolveProductionQtyColumn($pdo);
    $acceptedColumn = resolveAcceptedQtyColumn($pdo);
    if (!$toolColumn) {
        echo json_encode([]);
        exit;
    }

    $planExpr = numericSumExpression($planColumn);
    $qtyExpr = numericSumExpression($qtyColumn);
    $acceptedExpr = numericSumExpression($acceptedColumn);
    $ncExpr = '0';
    if ($qtyColumn && $acceptedColumn) {
        $ncExpr = "COALESCE(SUM(GREATEST(CAST(NULLIF(`$qtyColumn`, '') AS DECIMAL(18,2)) - CAST(NULLIF(`$acceptedColumn`, '') AS DECIMAL(18,2)), 0)), 0)";
    }
    
    $sql = "SELECT 
        operator_name as worker_name,
        COUNT(*) as entries,
        COUNT(DISTINCT log_date) as days,
        $planExpr as plan_qty,
        $qtyExpr as produced_qty,
        $acceptedExpr as accepted_qty,
        $ncExpr as nc_qty
    FROM production_logs 
    WHERE `$toolColumn` = ? AND operator_name IS NOT NULL AND operator_name != ''
    GROUP BY operator_name
    ORDER BY produced_qty DESC, worker_name ASC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$tool]);
    echo json_encode($stmt->fetchAll());
    exit;
}

// EMPLOYEE DETAILED ANALYTICS (for search page)
if ($action === 'get_employee_full_analytics') {
    requireLogin();
    
    $employee = $_GET['employee'] ?? '';
    if (!$employee) {
        echo json_encode([]);
        exit;
    }
    
    $sql = "SELECT 
        log_date as date,
        tool_no,
        operation,
        part_name,
        SUM(CAST(plan_qty AS DECIMAL(10,2))) as plan_qty,
        SUM(CAST(produced_qty AS DECIMAL(10,2))) as produced_qty,
        SUM(CAST(accepted_qty AS DECIMAL(10,2))) as accepted_qty,
        SUM(CAST(produced_qty AS DECIMAL(10,2)) - CAST(accepted_qty AS DECIMAL(10,2))) as nc_qty,
        ROUND(SUM(CAST(accepted_qty AS DECIMAL(10,2))) / NULLIF(SUM(CAST(produced_qty AS DECIMAL(10,2))), 0) * 100, 2) as quality_rate
    FROM production_logs 
    WHERE operator_name = ?
    GROUP BY log_date, tool_no, operation, part_name
    ORDER BY log_date DESC, tool_no ASC";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$employee]);
    echo json_encode($stmt->fetchAll());
    exit;
}

// GET ALL TOOLS - For dropdown population
if ($action === 'get_all_tools') {
    requireLogin();

    $toolColumn = resolveToolColumn($pdo);
    if (!$toolColumn) {
        echo json_encode([]);
        exit;
    }
    
    $sql = "SELECT DISTINCT `$toolColumn` as tool_no, `$toolColumn` as name
            FROM production_logs 
            WHERE `$toolColumn` IS NOT NULL AND `$toolColumn` != ''
            ORDER BY `$toolColumn` ASC";
    
    $stmt = $pdo->query($sql);
    echo json_encode($stmt->fetchAll());
    exit;
}

// GET ALL TOOLS PRODUCTION ANALYTICS
if ($action === 'get_all_tool_production_analytics') {
    requireLogin();

    $toolColumn = resolveToolColumn($pdo);
    $planColumn = resolvePlanQtyColumn($pdo);
    $qtyColumn = resolveProductionQtyColumn($pdo);
    $acceptedColumn = resolveAcceptedQtyColumn($pdo);
    if (!$toolColumn) {
        echo json_encode([]);
        exit;
    }

    $planExpr = numericSumExpression($planColumn);
    $qtyExpr = numericSumExpression($qtyColumn);
    $acceptedExpr = numericSumExpression($acceptedColumn);
    $ncExpr = '0';
    if ($qtyColumn && $acceptedColumn) {
        $ncExpr = "COALESCE(SUM(GREATEST(CAST(NULLIF(`$qtyColumn`, '') AS DECIMAL(18,2)) - CAST(NULLIF(`$acceptedColumn`, '') AS DECIMAL(18,2)), 0)), 0)";
    }
    
    $sql = "SELECT 
        `$toolColumn` as tool_no,
        `$toolColumn` as tool_name,
        `$toolColumn` as field_name,
        COUNT(*) as total_entries,
        $planExpr as total_plan_qty,
        $qtyExpr as total_produced_qty,
        $acceptedExpr as total_accepted_qty,
        $ncExpr as nc_qty,
        CASE 
            WHEN $qtyExpr > 0 THEN ROUND(($acceptedExpr / $qtyExpr) * 100, 2)
            ELSE 0
        END as quality_percent,
        CASE 
            WHEN $planExpr > 0 THEN ROUND(($qtyExpr / $planExpr) * 100, 2)
            ELSE 0
        END as efficiency_percent
    FROM production_logs 
    WHERE `$toolColumn` IS NOT NULL AND `$toolColumn` != ''
    GROUP BY `$toolColumn`
    ORDER BY total_produced_qty DESC
    LIMIT 50";
    
    $stmt = $pdo->query($sql);
    echo json_encode($stmt->fetchAll());
    exit;
}

// GET SPECIFIC TOOL PRODUCTION ANALYTICS
if ($action === 'get_tool_production_analytics') {
    requireLogin();
    
    $tool_no = trim((string)($_GET['tool_no'] ?? ($_GET['tool'] ?? '')));
    if (!$tool_no) {
        echo json_encode([]);
        exit;
    }

    $toolColumn = resolveToolColumn($pdo);
    $planColumn = resolvePlanQtyColumn($pdo);
    $qtyColumn = resolveProductionQtyColumn($pdo);
    $acceptedColumn = resolveAcceptedQtyColumn($pdo);
    if (!$toolColumn) {
        echo json_encode([]);
        exit;
    }

    $planExpr = numericSumExpression($planColumn);
    $qtyExpr = numericSumExpression($qtyColumn);
    $acceptedExpr = numericSumExpression($acceptedColumn);
    $ncExpr = '0';
    if ($qtyColumn && $acceptedColumn) {
        $ncExpr = "COALESCE(SUM(GREATEST(CAST(NULLIF(`$qtyColumn`, '') AS DECIMAL(18,2)) - CAST(NULLIF(`$acceptedColumn`, '') AS DECIMAL(18,2)), 0)), 0)";
    }
    
    $sql = "SELECT 
        `$toolColumn` as tool_no,
        `$toolColumn` as tool_name,
        `$toolColumn` as field_name,
        COUNT(*) as total_entries,
        $planExpr as total_plan_qty,
        $qtyExpr as total_produced_qty,
        $acceptedExpr as total_accepted_qty,
        $ncExpr as nc_qty,
        CASE 
            WHEN $qtyExpr > 0 THEN ROUND(($acceptedExpr / $qtyExpr) * 100, 2)
            ELSE 0
        END as quality_percent,
        CASE 
            WHEN $planExpr > 0 THEN ROUND(($qtyExpr / $planExpr) * 100, 2)
            ELSE 0
        END as efficiency_percent
    FROM production_logs 
    WHERE `$toolColumn` = ?
    GROUP BY `$toolColumn`";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$tool_no]);
    echo json_encode($stmt->fetchAll());
    exit;
}
// ... existing code ...

// ADD THIS BLOCK TO HANDLE THE NEW ANALYTICS REQUEST
// ... inside api.php ...



// ... existing code ...
// --- INSIDE api.php switch($action) ---

// GET TOOL STROKE DATA (For dynamic updates in Maintenance page)
// GET TOOL CURRENT USAGE (For preview in entry form)
if ($action === 'get_tool_current_usage') {
    requireLogin();
    
    $toolId = intval($_GET['tool_id'] ?? 0);
    if ($toolId <= 0) {
        echo json_encode(['success' => false, 'error' => 'Invalid tool ID']);
        exit;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT id, current_usage, life_limit FROM field_options WHERE id = ?");
        $stmt->execute([$toolId]);
        $tool = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($tool) {
            echo json_encode([
                'success' => true,
                'current_usage' => $tool['current_usage'] ?: 0,
                'max_life' => $tool['life_limit'] ?: 100000
            ]);
        } else {
            echo json_encode(['success' => false, 'error' => 'Tool not found']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// --- RESET MAINTENANCE / TOOL LIFE ---
if ($action === 'reset_life_cycle') {
    requireAdmin();
    $id = intval($_POST['id']);
    $type = $_POST['type']; 
    
    try {
        if ($type == 'machine') {
            // 1. GET DETAILS FOR HISTORY
            $tech = isset($_POST['technician']) ? trim($_POST['technician']) : 'Admin';
            $notes = isset($_POST['notes']) ? trim($_POST['notes']) : 'Routine Maintenance';
            $date = date('Y-m-d');

            // 2. INSERT INTO HISTORY TABLE
            $log = $pdo->prepare("INSERT INTO maintenance_history (machine_option_id, service_date, technician_name, service_notes) VALUES (?, ?, ?, ?)");
            $historyResult = $log->execute([$id, $date, $tech, $notes]);
            
            if (!$historyResult) {
                echo json_encode(['success' => false, 'error' => 'Failed to insert maintenance history']);
                exit;
            }

            // 3. RESET MACHINE DATE TO TODAY
            $stmt = $pdo->prepare("UPDATE field_options SET last_service = ? WHERE id = ?");
            $updateResult = $stmt->execute([$date, $id]);
            
            if ($updateResult) {
                echo json_encode(['success' => true, 'message' => 'Service logged and cycle reset']);
            } else {
                echo json_encode(['success' => false, 'error' => 'Failed to update machine cycle']);
            }

        } else if ($type == 'tool') {
            // TOOL RESET - Reset current_usage to 0
            $stmt = $pdo->prepare("UPDATE field_options SET current_usage = 0 WHERE id = ?");
            $result = $stmt->execute([$id]);
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Tool reset successfully']);
            } else {
                echo json_encode(['success' => false, 'error' => 'Failed to reset tool']);
            }
        } else {
            echo json_encode(['success' => false, 'error' => 'Invalid type specified']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
    }
    exit;
}

// --- NEW: FETCH HISTORY ---
if ($action === 'get_machine_history') {
    requireAdmin();
    $id = $_POST['id'];
    $stmt = $pdo->prepare("SELECT * FROM maintenance_history WHERE machine_option_id = ? ORDER BY service_date DESC");
    $stmt->execute([$id]);
    echo json_encode($stmt->fetchAll());
    exit;
}

// --- FETCH ALL TOOL DATA FOR DYNAMIC UPDATES ---
if ($action === 'get_tool_data') {
    requireLogin();
    
    try {
        // Get the tool field ID from custom_fields
        $fieldStmt = $pdo->prepare("SELECT id FROM custom_fields WHERE field_label LIKE '%Tool%' LIMIT 1");
        $fieldStmt->execute();
        $field = $fieldStmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$field) {
            echo json_encode(['success' => false, 'error' => 'Tool field not found', 'tools' => []]);
            exit;
        }
        
        $toolFieldId = $field['id'];
        
        // Fetch all tools with their current usage
        $stmt = $pdo->prepare("
            SELECT 
                id, 
                label, 
                current_usage, 
                life_limit 
            FROM field_options 
            WHERE field_id = ? 
            ORDER BY label ASC
        ");
        $stmt->execute([$toolFieldId]);
        $tools = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Process each tool with calculations
        $toolsData = [];
        foreach ($tools as $tool) {
            $maxLife = ($tool['life_limit'] ?: 100000);
            $usedStrokes = ($tool['current_usage'] ?: 0);
            $remaining = $maxLife - $usedStrokes;
            $usagePercent = ($usedStrokes / $maxLife) * 100;
            
            $status = 'GOOD';
            if ($remaining <= 0) {
                $status = 'EXPIRED';
            } elseif ($remaining < 1000) {
                $status = 'CRITICAL';
            }
            
            $toolsData[] = [
                'id' => $tool['id'],
                'label' => $tool['label'],
                'current_usage' => $usedStrokes,
                'max_life' => $maxLife,
                'remaining' => max(0, $remaining),
                'usage_percent' => round($usagePercent, 1),
                'status' => $status
            ];
        }
        
        echo json_encode(['success' => true, 'tools' => $toolsData]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage(), 'tools' => []]);
    }
    exit;
}

// =========================================================
// USER MANAGEMENT ENDPOINTS (Admin Dashboard)
// =========================================================

// GET ALL USERS
if ($action === 'get_all_users') {
    requireAdmin();
    try {
        $stmt = $pdo->query("SELECT id, username, role, created_by, created_at FROM users ORDER BY created_at DESC");
        $users = $stmt->fetchAll();
        
        // Get creator names
        foreach ($users as &$user) {
            if ($user['created_by']) {
                $creatorStmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
                $creatorStmt->execute([$user['created_by']]);
                $creator = $creatorStmt->fetch();
                $user['created_by'] = $creator['username'] ?? 'Unknown';
            }
        }
        
        echo json_encode(['success' => true, 'users' => $users]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// CREATE NEW USER
if ($action === 'create_user' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    try {
        $username = $_POST['username'] ?? '';
        $password = $_POST['password'] ?? '';
        $role = $_POST['role'] ?? 'User';
        $adminId = $_SESSION['user_id'];
        
        // Validate input
        if (empty($username) || empty($password)) {
            echo json_encode(['success' => false, 'error' => 'Username and password are required']);
            exit;
        }
        
        // Check if username already exists
        $checkStmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
        $checkStmt->execute([$username]);
        if ($checkStmt->fetch()) {
            echo json_encode(['success' => false, 'error' => 'Username already exists']);
            exit;
        }
        
        // Get admin's ID from username
        $adminStmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
        $adminStmt->execute([$adminId]);
        $adminData = $adminStmt->fetch();
        $adminUserId = $adminData['id'] ?? null;
        
        // Insert new user
        $stmt = $pdo->prepare("INSERT INTO users (username, password, role, created_by) VALUES (?, ?, ?, ?)");
        $stmt->execute([$username, $password, $role, $adminUserId]);
        
        // Log activity
        $logStmt = $pdo->prepare("INSERT INTO activity_log (user_id, username, action_type, details) VALUES (?, ?, ?, ?)");
        $logStmt->execute([$adminUserId, $_SESSION['user_id'], 'USER_CREATED', "Created user: $username with role: $role"]);
        
        echo json_encode(['success' => true, 'message' => 'User created successfully']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// DELETE USER
if ($action === 'delete_user' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    try {
        $userId = $_POST['user_id'] ?? '';
        
        if (empty($userId)) {
            echo json_encode(['success' => false, 'error' => 'User ID is required']);
            exit;
        }
        
        // Don't allow deleting Administrator accounts
        $userStmt = $pdo->prepare("SELECT id, username, role FROM users WHERE id = ?");
        $userStmt->execute([$userId]);
        $user = $userStmt->fetch();
        
        if (!$user) {
            echo json_encode(['success' => false, 'error' => 'User not found']);
            exit;
        }
        
        if ($user['role'] === 'Administrator') {
            echo json_encode(['success' => false, 'error' => 'Cannot delete Administrator accounts']);
            exit;
        }
        
        // Delete user
        $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        
        // Log activity
        $adminStmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
        $adminStmt->execute([$_SESSION['user_id']]);
        $adminData = $adminStmt->fetch();
        $adminUserId = $adminData['id'] ?? null;
        
        $logStmt = $pdo->prepare("INSERT INTO activity_log (user_id, username, action_type, details) VALUES (?, ?, ?, ?)");
        $logStmt->execute([$adminUserId, $_SESSION['user_id'], 'USER_DELETED', "Deleted user: {$user['username']}"]);
        
        echo json_encode(['success' => true, 'message' => 'User deleted successfully']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// CHANGE USER PASSWORD (by admin)
if ($action === 'change_user_password' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    try {
        $userId = $_POST['user_id'] ?? '';
        $newPassword = $_POST['new_password'] ?? '';
        
        if (empty($userId) || empty($newPassword)) {
            echo json_encode(['success' => false, 'error' => 'User ID and new password are required']);
            exit;
        }
        
        // Get user info
        $userStmt = $pdo->prepare("SELECT username FROM users WHERE id = ?");
        $userStmt->execute([$userId]);
        $user = $userStmt->fetch();
        
        if (!$user) {
            echo json_encode(['success' => false, 'error' => 'User not found']);
            exit;
        }
        
        // Update password
        $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $stmt->execute([$newPassword, $userId]);
        
        // Log activity
        $adminStmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
        $adminStmt->execute([$_SESSION['user_id']]);
        $adminData = $adminStmt->fetch();
        $adminUserId = $adminData['id'] ?? null;
        
        $logStmt = $pdo->prepare("INSERT INTO activity_log (user_id, username, action_type, details) VALUES (?, ?, ?, ?)");
        $logStmt->execute([$adminUserId, $_SESSION['user_id'], 'PASSWORD_CHANGED', "Changed password for user: {$user['username']}"]);
        
        echo json_encode(['success' => true, 'message' => 'Password updated successfully']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// CHANGE OWN PASSWORD
if ($action === 'change_own_password' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireLogin();
    try {
        $currentPassword = $_POST['current_password'] ?? '';
        $newPassword = $_POST['new_password'] ?? '';
        
        if (empty($currentPassword) || empty($newPassword)) {
            echo json_encode(['success' => false, 'error' => 'Current and new passwords are required']);
            exit;
        }
        
        // Get current user
        $userStmt = $pdo->prepare("SELECT id, password FROM users WHERE username = ?");
        $userStmt->execute([$_SESSION['user_id']]);
        $user = $userStmt->fetch();
        
        // Verify current password
        if (!$user || $user['password'] !== $currentPassword) {
            echo json_encode(['success' => false, 'error' => 'Current password is incorrect']);
            exit;
        }
        
        // Update password
        $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $stmt->execute([$newPassword, $user['id']]);
        
        // Log activity
        $logStmt = $pdo->prepare("INSERT INTO activity_log (user_id, username, action_type, details) VALUES (?, ?, ?, ?)");
        $logStmt->execute([$user['id'], $_SESSION['user_id'], 'PASSWORD_CHANGED', 'Changed own password']);
        
        echo json_encode(['success' => true, 'message' => 'Password changed successfully']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET ACTIVITY LOG
if ($action === 'get_activity_log') {
    requireLogin();
    try {
        // Show data-entry actions with employee data and production date details
        $query = "SELECT entry_id, user_id, username, employee_name, record_count, entry_time, action_type, details
                  FROM activity_log
                  WHERE action_type = 'DATA_ENTRY'
                  ORDER BY entry_time DESC
                  LIMIT 100";
        $stmt = $pdo->query($query);
        $logs = $stmt->fetchAll();
        
        echo json_encode(['success' => true, 'logs' => $logs]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// LOG USER ACTIVITY (called when entering data)
if ($action === 'log_activity' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireLogin();
    try {
        $employeeName = trim($_POST['employee_name'] ?? 'Unknown');
        $empId = trim($_POST['emp_id'] ?? $_POST['employee_id'] ?? '');
        $logDate = trim($_POST['log_date'] ?? $_POST['date'] ?? '');
        $employeeData = $empId !== '' ? "{$empId} - {$employeeName}" : $employeeName;
        $recordCount = $_POST['record_count'] ?? 1;
        $details = $logDate !== '' ? "Log Date: {$logDate}" : '';
        
        // Get current user ID
        $userStmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
        $userStmt->execute([$_SESSION['user_id']]);
        $user = $userStmt->fetch();
        
        if (!$user) {
            echo json_encode(['success' => false, 'error' => 'User not found']);
            exit;
        }
        
        // Log the activity
        $stmt = $pdo->prepare("INSERT INTO activity_log (user_id, username, employee_name, record_count, action_type, details) VALUES (?, ?, ?, ?, 'DATA_ENTRY', ?)");
        $stmt->execute([$user['id'], $_SESSION['user_id'], $employeeData, $recordCount, $details]);
        
        echo json_encode(['success' => true, 'message' => 'Activity logged']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// =========================================================
// ROLE PERMISSIONS MANAGEMENT (Admin Dashboard)
// =========================================================

// GET CURRENT USER PAGE PERMISSIONS (for dynamic sidebar + client guards)
if ($action === 'get_my_page_permissions') {
    requireLogin();
    try {
        $sessionRole = trim((string)($_SESSION['role'] ?? ''));
        $roleNormalized = strtolower($sessionRole);

        $defaultPageKeys = [
            'admin_dashboard',
            'admin_fields',
            'all_employees',
            'analytics',
            'entry_log',
            'maintenance',
            'register_emp',
            'search_emp',
            'tools_dashboard',
            'view_log'
        ];

        // Admin has full access
        if (in_array($roleNormalized, ['administrator', 'admin'], true)) {
            echo json_encode([
                'success' => true,
                'role' => $sessionRole,
                'permissions' => array_fill_keys($defaultPageKeys, true)
            ]);
            exit;
        }

        $permissions = array_fill_keys($defaultPageKeys, true);
        $tableStmt = $pdo->query("SHOW TABLES LIKE 'role_permissions'");
        $hasPermissionTable = $tableStmt && $tableStmt->fetchColumn();

        if ($hasPermissionTable) {
            // Start from strict false and allow only configured pages for non-admins
            $permissions = array_fill_keys($defaultPageKeys, false);

            $stmt = $pdo->prepare("SELECT page_name, is_allowed FROM role_permissions WHERE role = ?");
            $stmt->execute([$sessionRole]);
            $rows = $stmt->fetchAll();

            if (!empty($rows)) {
                foreach ($rows as $row) {
                    $pageName = (string)($row['page_name'] ?? '');
                    if ($pageName !== '') {
                        $permissions[$pageName] = intval($row['is_allowed']) === 1;
                    }
                }
            } else {
                // Fallback legacy behavior if role exists but has no rows yet
                $permissions = array_fill_keys($defaultPageKeys, true);
                if ($roleNormalized === 'controller') {
                    $permissions['admin_dashboard'] = false;
                    $permissions['admin_fields'] = false;
                    $permissions['register_emp'] = false;
                } elseif ($roleNormalized === 'supervisor') {
                    $permissions['admin_dashboard'] = false;
                }
            }
        } else {
            // Fallback legacy behavior when permission table is not deployed
            if ($roleNormalized === 'controller') {
                $permissions['admin_dashboard'] = false;
                $permissions['admin_fields'] = false;
                $permissions['register_emp'] = false;
            } elseif ($roleNormalized === 'supervisor') {
                $permissions['admin_dashboard'] = false;
            }
        }

        echo json_encode([
            'success' => true,
            'role' => $sessionRole,
            'permissions' => $permissions
        ]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// GET ALL ROLE PERMISSIONS
if ($action === 'get_role_permissions') {
    requireAdmin();
    try {
        // Get all roles and their page permissions
        $query = "SELECT role, page_name, page_label, is_allowed 
                  FROM role_permissions 
                  WHERE page_name <> 'machines_tools_admin'
                  ORDER BY role, page_name";
        $stmt = $pdo->query($query);
        $permissions = $stmt->fetchAll();
        
        // Group by role
        $grouped = [];
        foreach ($permissions as $perm) {
            if (!isset($grouped[$perm['role']])) {
                $grouped[$perm['role']] = [];
            }
            $grouped[$perm['role']][] = $perm;
        }
        
        echo json_encode(['success' => true, 'permissions' => $grouped]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// UPDATE ROLE PERMISSION (Turn page access ON/OFF)
if ($action === 'update_role_permission' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requireAdmin();
    try {
        $role = $_POST['role'] ?? '';
        $pageName = $_POST['page_name'] ?? '';
        $isAllowed = $_POST['is_allowed'] ?? 0;
        
        if (empty($role) || empty($pageName)) {
            echo json_encode(['success' => false, 'error' => 'Role and page name required']);
            exit;
        }
        
        // Update permission
        $stmt = $pdo->prepare("UPDATE role_permissions SET is_allowed = ? WHERE role = ? AND page_name = ?");
        $stmt->execute([$isAllowed, $role, $pageName]);
        
        // Log activity
        $adminStmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
        $adminStmt->execute([$_SESSION['user_id']]);
        $adminData = $adminStmt->fetch();
        $adminUserId = $adminData['id'] ?? null;
        
        $logStmt = $pdo->prepare("INSERT INTO activity_log (user_id, username, action_type, details) VALUES (?, ?, ?, ?)");
        $action_detail = "Updated {$role} permission for page: {$pageName} to " . ($isAllowed ? 'ALLOWED' : 'BLOCKED');
        $logStmt->execute([$adminUserId, $_SESSION['user_id'], 'PERMISSION_CHANGED', $action_detail]);
        
        echo json_encode(['success' => true, 'message' => 'Permission updated']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// DEFAULT: If no action specified or unknown action
http_response_code(400);
echo json_encode([
    'success' => false,
    'error' => empty($action) ? 'No action specified' : "Unknown action: '$action'",
    'available_actions' => [
        'login' => 'POST - User authentication',
        'check_session' => 'GET - Check if user is logged in',
        'logout' => 'GET - End user session',
        'get_all_users' => 'GET - Get all users (admin only)',
        'create_user' => 'POST - Create new user (admin only)',
        'delete_user' => 'POST - Delete user (admin only)',
        'change_user_password' => 'POST - Change user password (admin only)',
        'change_own_password' => 'POST - Change own password',
        'get_activity_log' => 'GET - Get activity log',
        'log_activity' => 'POST - Log user activity',
        'get_my_page_permissions' => 'GET - Get page permissions for current logged-in user',
        'get_role_permissions' => 'GET - Get all role page permissions (admin only)',
        'update_role_permission' => 'POST - Update role page access (admin only)',
        'get_custom_fields' => 'GET - Get form fields',
        'add_custom_field' => 'POST - Create new field',
        'get_options' => 'GET - Get field dropdown options',
        'save_log' => 'POST - Save production entry and update tool strokes',
        'get_logs' => 'GET - Get all production logs',
        'get_report_fields' => 'GET - Get fields for report view',
        'get_tool_data' => 'GET - Get current tool status and strokes',
        'reset_life_cycle' => 'POST - Reset machine service or tool strokes',
        'get_machine_history' => 'POST - Get maintenance history'
    ]
]);
?>
