<?php
// Protect this page with session check
include 'session_check.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mektron - Data Analytics Dashboard</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <link rel="stylesheet" href="style.css">
    
    <style>
        /* ===== ANALYTICS DASHBOARD STYLES ===== */
        body {
            background: #f4f6f9;
            font-family: 'Segoe UI', sans-serif;
            margin: 0;
            overflow-x: hidden;
        }

        .main-content {
            margin-left: 0;
            padding: 25px;
            width: 100%;
            box-sizing: border-box;
            min-height: 100vh;
            transition: margin-left 0.3s ease;
        }

        .sidebar {
            position: fixed;
            top: 0;
            left: -260px;
            width: 260px;
            height: 100%;
            background: #013B3B;
            transition: left 0.3s ease;
            z-index: 1000;
            overflow-y: auto;
        }

        .sidebar.active { left: 0; }
        .main-content.shifted { margin-left: 260px; }

        .analytics-container {
            background: white;
            padding: 30px;
            border-radius: 12px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
        }

        .analytics-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #eee;
            padding-bottom: 20px;
        }

        .analytics-header h1 {
            margin: 0;
            color: #2c3e50;
            font-size: 28px;
        }

        .header-controls {
            display: flex;
            gap: 15px;
            align-items: center;
        }

        .btn-toggle-sidebar {
            background: #013B3B;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 16px;
            transition: 0.3s;
        }

        .btn-toggle-sidebar:hover { background: #00564f; }

        /* ===== HAMBURGER MENU BUTTON ===== */
        .menu-toggle-btn {
            background: none;
            border: none;
            font-size: 24px;
            color: #013B3B;
            cursor: pointer;
            margin-right: 15px;
            padding: 5px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
            transition: 0.2s;
        }
        .menu-toggle-btn:hover { background: #e0e0e0; }

        /* ===== OVERLAY FOR SIDEBAR ===== */
        .overlay {
            display: none;
            position: fixed;
            top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 999;
        }
        .overlay.active { display: block; }

        /* ===== CONTROLS SECTION ===== */
        .controls-section {
            background: #f9f9f9;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 30px;
            border-left: 5px solid #013B3B;
        }

        .control-group {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 15px;
        }

        .control-item label {
            display: block;
            font-weight: 600;
            color: #555;
            margin-bottom: 8px;
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .control-item select,
        .control-item input {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 14px;
            box-sizing: border-box;
            transition: 0.3s;
        }

        .control-item select:focus,
        .control-item input:focus {
            border-color: #013B3B;
            outline: none;
            box-shadow: 0 0 0 3px rgba(1, 59, 59, 0.1);
        }

        .btn-load {
            background: #00a65a;
            color: white;
            border: none;
            padding: 12px 25px;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            transition: 0.3s;
            align-self: flex-end;
        }

        .btn-load:hover { background: #008d4c; }
        .btn-load:active { transform: scale(0.98); }

        /* ===== STATS CARDS ===== */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
            transition: transform 0.2s;
        }

        .stat-card:hover { transform: translateY(-5px); }
        .stat-card h3 { margin: 0 0 10px 0; font-size: 13px; opacity: 0.9; text-transform: uppercase; }
        .stat-card .value { font-size: 32px; font-weight: 800; }
        .stat-card .icon { float: right; font-size: 40px; opacity: 0.2; }

        .stat-card.total { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .stat-card.employees { background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); }
        .stat-card.records { background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); }

        /* ===== CHARTS SECTION ===== */
        .charts-section {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 30px;
            margin-bottom: 30px;
        }

        .chart-container {
            background: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
            border-top: 4px solid #013B3B;
        }

        .chart-title {
            font-size: 16px;
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .chart-title i { color: #013B3B; }

        /* ===== DETAILED STATS TABLE ===== */
        .details-section {
            background: white;
            padding: 25px;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
            border-top: 4px solid #013B3B;
        }

        .details-title {
            font-size: 16px;
            font-weight: 700;
            color: #2c3e50;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }

        .data-table thead {
            background: #f0f0f0;
            border-bottom: 2px solid #ddd;
        }

        .data-table th {
            padding: 12px;
            text-align: left;
            font-weight: 600;
            color: #555;
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .data-table td {
            padding: 12px;
            border-bottom: 1px solid #eee;
            font-size: 14px;
            color: #666;
        }

        .data-table tbody tr:hover {
            background: #f9f9f9;
        }

        .badge {
            display: inline-block;
            padding: 5px 12px;
            background: #e3f2fd;
            color: #1976d2;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }

        /* ===== LOADING & EMPTY STATES ===== */
        .loading-spinner {
            text-align: center;
            padding: 40px;
            color: #999;
        }

        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #013B3B;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto 20px;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .empty-state {
            text-align: center;
            padding: 40px;
            color: #999;
        }

        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            opacity: 0.3;
        }

        /* ===== RESPONSIVE ===== */
        @media (max-width: 1024px) {
            .charts-section { grid-template-columns: 1fr; }
        }

        @media (max-width: 768px) {
            .main-content { padding: 15px; }
            .analytics-container { padding: 20px; }
            .control-group { grid-template-columns: 1fr; }
            .header-controls { flex-direction: column; width: 100%; }
            .btn-load { width: 100%; }
        }

        .no-data { color: #999; font-style: italic; text-align: center; padding: 20px; }
    </style>
</head>

<body>

<div id="overlay" class="overlay" onclick="toggleSidebar()"></div>

<script src="sidebar.js"></script>

<div class="main-content">
    <div class="analytics-container">
        
        <div class="analytics-header">
            <div style="display: flex; align-items: center;">
                <button class="menu-toggle-btn" onclick="toggleSidebar()">
                    <i class="fas fa-bars"></i>
                </button>
                <h1><i class="fas fa-chart-bar"></i> Data Analytics Dashboard</h1>
            </div>
        </div>

        <div class="controls-section">
            <div class="control-group">
                <div class="control-item">
                    <label><i class="fas fa-layer-group"></i> Select Field to Analyze:</label>
                    <select id="fieldSelect" onchange="loadFieldStatistics()">
                        <option value="">-- Choose a field --</option>
                    </select>
                </div>
                <div class="control-item">
                    <label><i class="fas fa-tools"></i> Machine/Tool Field:</label>
                    <select id="machineFieldSelect" onchange="loadMachineData()">
                        <option value="">-- Choose machine field --</option>
                    </select>
                </div>
            </div>

            <div style="border-top: 2px solid #ddd; padding-top: 20px; margin-top: 20px;">
                <div style="font-weight: 600; color: #2c3e50; margin-bottom: 15px; display: flex; align-items: center; gap: 8px;">
                    <i class="fas fa-exchange-alt"></i> Compare Two Fields
                </div>
                <div class="control-group">
                    <div class="control-item">
                        <label><i class="fas fa-chart-bar"></i> Field 1:</label>
                        <select id="compareField1">
                            <option value="">-- Choose first field --</option>
                        </select>
                    </div>
                    <div class="control-item">
                        <label><i class="fas fa-chart-pie"></i> Field 2:</label>
                        <select id="compareField2">
                            <option value="">-- Choose second field --</option>
                        </select>
                    </div>
                </div>
                <button class="btn-load" style="background: #17a2b8;" onclick="loadFieldComparison()">
                    <i class="fas fa-balance-scale"></i> Compare Fields
                </button>
            </div>

            <div style="display: flex; gap: 10px; margin-top: 15px;">
                <button class="btn-load" onclick="loadAllAnalytics()">
                    <i class="fas fa-sync-alt"></i> Load All Analytics
                </button>
                <button class="btn-load" style="background: #1abc9c;" onclick="loadDateWiseReport()">
                    <i class="fas fa-calendar-alt"></i> Load Date-Wise Report
                </button>
                <button class="btn-load" style="background: #6c757d;" onclick="location.reload()">
                    <i class="fas fa-redo"></i> Refresh
                </button>
            </div>
        </div>

        <div class="stats-grid">
            <div class="stat-card total">
                <div class="icon"><i class="fas fa-database"></i></div>
                <h3>Total Records</h3>
                <div class="value" id="totalRecords">-</div>
            </div>
            <div class="stat-card employees">
                <div class="icon"><i class="fas fa-users"></i></div>
                <h3>Active Employees</h3>
                <div class="value" id="employeeCount">-</div>
            </div>
            <div class="stat-card records">
                <div class="icon"><i class="fas fa-chart-line"></i></div>
                <h3>Data Points</h3>
                <div class="value" id="dataPoints">-</div>
            </div>
        </div>

        <div class="charts-section">
            <div class="chart-container">
                <div class="chart-title">
                    <i class="fas fa-chart-bar"></i> 30-Day Production Trend
                </div>
                <canvas id="dailyProductionChart"></canvas>
            </div>

            <div class="chart-container">
                <div class="chart-title">
                    <i class="fas fa-pie-chart"></i> Field Distribution
                </div>
                <canvas id="fieldDistributionChart"></canvas>
            </div>
        </div>

        <!-- ===== MACHINE-WISE ANALYTICS (ENHANCED) ===== -->
        <div class="details-section" id="machineSection" style="display:none;">
            <div class="details-title">
                <i class="fas fa-tools"></i> Machine/Tool Statistics
            </div>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Machine/Tool</th>
                        <th>Employees</th>
                        <th>Total Operations</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody id="machineTableBody">
                    <tr><td colspan="4" class="no-data">No data available</td></tr>
                </tbody>
            </table>
        </div>

        <!-- ===== MACHINE DETAIL ANALYSIS (NEW) ===== -->
        <div class="details-section" id="machineDetailSection" style="display:none;">
            <div class="details-title">
                <i class="fas fa-microscope"></i> <span id="machineDetailTitle">Machine Detail Analysis</span>
            </div>

            <!-- Machine Summary Cards -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-bottom: 20px;">
                <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 10px;">
                    <div style="font-size: 12px; opacity: 0.9; text-transform: uppercase;">Total Operations</div>
                    <div style="font-size: 32px; font-weight: bold; margin: 10px 0;" id="machineDetailOpsCount">0</div>
                </div>
                <div style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); color: white; padding: 20px; border-radius: 10px;">
                    <div style="font-size: 12px; opacity: 0.9; text-transform: uppercase;">Assigned Employees</div>
                    <div style="font-size: 32px; font-weight: bold; margin: 10px 0;" id="machineDetailEmpCount">0</div>
                </div>
                <div style="background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); color: white; padding: 20px; border-radius: 10px;">
                    <div style="font-size: 12px; opacity: 0.9; text-transform: uppercase;">Tools Used</div>
                    <div style="font-size: 32px; font-weight: bold; margin: 10px 0;" id="machineDetailToolsCount">0</div>
                </div>
                <div style="background: linear-gradient(135deg, #fa709a 0%, #fee140 100%); color: white; padding: 20px; border-radius: 10px;">
                    <div style="font-size: 12px; opacity: 0.9; text-transform: uppercase;">Avg Daily Plans</div>
                    <div style="font-size: 32px; font-weight: bold; margin: 10px 0;" id="machineDetailDailyPlans">0</div>
                </div>
            </div>

            <!-- Tools Used in Machine -->
            <div style="background: white; padding: 20px; border-radius: 10px; box-shadow: 0 4px 15px rgba(0,0,0,0.05); border-top: 4px solid #667eea; margin-bottom: 20px;">
                <div style="font-weight: 600; color: #2c3e50; margin-bottom: 15px; display: flex; align-items: center; gap: 8px;">
                    <i class="fas fa-hammer" style="color: #667eea;"></i> Tools Used in This Machine
                </div>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Tool Name</th>
                            <th>Usage Count</th>
                            <th>Percentage</th>
                        </tr>
                    </thead>
                    <tbody id="machineToolsBody">
                        <tr><td colspan="3" class="no-data">No tools data</td></tr>
                    </tbody>
                </table>
            </div>

            <!-- Employees Working on Machine -->
            <div style="background: white; padding: 20px; border-radius: 10px; box-shadow: 0 4px 15px rgba(0,0,0,0.05); border-top: 4px solid #f093fb; margin-bottom: 20px;">
                <div style="font-weight: 600; color: #2c3e50; margin-bottom: 15px; display: flex; align-items: center; gap: 8px;">
                    <i class="fas fa-users" style="color: #f093fb;"></i> Employees Working on This Machine
                </div>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Employee ID</th>
                            <th>Employee Name</th>
                            <th>Total Operations</th>
                            <th>% of Total</th>
                        </tr>
                    </thead>
                    <tbody id="machineEmployeesBody">
                        <tr><td colspan="4" class="no-data">No employees data</td></tr>
                    </tbody>
                </table>
            </div>

            <!-- Date-Wise Operations for Machine -->
            <div style="background: white; padding: 20px; border-radius: 10px; box-shadow: 0 4px 15px rgba(0,0,0,0.05); border-top: 4px solid #4facfe; margin-bottom: 20px;">
                <div style="font-weight: 600; color: #2c3e50; margin-bottom: 20px; display: flex; align-items: center; justify-content: space-between;">
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-calendar-alt" style="color: #4facfe; font-size: 18px;"></i> Date-Wise Operations Breakdown
                    </div>
                    <div style="display: flex; gap: 8px;">
                        <button onclick="switchMachineChart('bar')" style="padding: 6px 12px; background: #4facfe; color: white; border: none; border-radius: 6px; cursor: pointer; font-weight: 600;" id="btnMachineBar">
                            <i class="fas fa-chart-bar"></i> Bar
                        </button>
                        <button onclick="switchMachineChart('line')" style="padding: 6px 12px; background: #f0f0f0; color: #555; border: none; border-radius: 6px; cursor: pointer; font-weight: 600;" id="btnMachineLine">
                            <i class="fas fa-chart-line"></i> Line
                        </button>
                        <button onclick="switchMachineChart('pie')" style="padding: 6px 12px; background: #f0f0f0; color: #555; border: none; border-radius: 6px; cursor: pointer; font-weight: 600;" id="btnMachinePie">
                            <i class="fas fa-chart-pie"></i> Pie
                        </button>
                    </div>
                </div>
                <div style="position: relative; height: 400px; width: 100%;">
                    <canvas id="machineOperationsChart"></canvas>
                </div>
            </div>

            <!-- Daily Plans Detail Table -->
            <div style="background: white; padding: 20px; border-radius: 10px; box-shadow: 0 4px 15px rgba(0,0,0,0.05); border-top: 4px solid #fa709a;">
                <div style="font-weight: 600; color: #2c3e50; margin-bottom: 15px; display: flex; align-items: center; gap: 8px;">
                    <i class="fas fa-list-check" style="color: #fa709a;"></i> Daily Plans & Operations
                </div>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Day</th>
                            <th>Daily Operations</th>
                            <th>Employees Worked</th>
                            <th>Tools Used</th>
                            <th>Avg Per Employee</th>
                        </tr>
                    </thead>
                    <tbody id="machineDailyPlansBody">
                        <tr><td colspan="6" class="no-data">No daily data</td></tr>
                    </tbody>
                </table>
            </div>

            <button onclick="closeMachineDetail()" class="btn-load" style="background: #6c757d; margin-top: 20px; width: 100%;">
                <i class="fas fa-times"></i> Close Detail View
            </button>
        </div>

        <div class="details-section" id="fieldDistributionSection" style="display:none;">
            <div class="details-title">
                <i class="fas fa-list"></i> <span id="fieldDistributionTitle">Field Data Distribution</span>
            </div>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Value</th>
                        <th>Count</th>
                        <th>Percentage</th>
                    </tr>
                </thead>
                <tbody id="fieldDistributionBody">
                    <tr><td colspan="3" class="no-data">No data available</td></tr>
                </tbody>
            </table>
        </div>

        <div class="details-section">
            <div class="details-title">
                <i class="fas fa-user-tie"></i> Top Employees by Activity
            </div>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Employee ID</th>
                        <th>Employee Name</th>
                        <th>Total Records</th>
                    </tr>
                </thead>
                <tbody id="topEmployeesBody">
                    <tr><td colspan="3" class="no-data">Loading...</td></tr>
                </tbody>
            </table>
        </div>



        <div class="details-section" id="employeeMachineSection" style="display:none;">
            <div class="details-title">
                <i class="fas fa-link"></i> Employee-Machine Assignment
            </div>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Employee ID</th>
                        <th>Employee Name</th>
                        <th>Assigned Machines/Tools</th>
                    </tr>
                </thead>
                <tbody id="employeeMachineBody">
                    <tr><td colspan="3" class="no-data">No data available</td></tr>
                </tbody>
            </table>
        </div>

        <!-- ===== DATE-WISE STATISTICS REPORT ===== -->
        <div class="details-section" id="dateWiseSection" style="display:none;">
            <div class="details-title">
                <i class="fas fa-calendar-alt"></i> Date-Wise Statistics Report
            </div>
            
            <div style="margin-bottom: 20px; display: flex; gap: 10px; flex-wrap: wrap;">
                <input type="date" id="dateRangeStart" style="padding: 10px; border: 1px solid #ddd; border-radius: 6px;">
                <input type="date" id="dateRangeEnd" style="padding: 10px; border: 1px solid #ddd; border-radius: 6px;">
                <button onclick="loadDateWiseReport()" class="btn-load" style="background: #1abc9c;">
                    <i class="fas fa-filter"></i> Filter Report
                </button>
            </div>

            <div style="background: white; padding: 25px; border-radius: 10px; box-shadow: 0 4px 15px rgba(0,0,0,0.05); border-top: 4px solid #1abc9c; margin-bottom: 20px;">
                <div style="font-weight: 600; color: #2c3e50; margin-bottom: 15px;">📊 Daily Activity Heatmap</div>
                <div style="position: relative; height: 300px; width: 100%;">
                    <canvas id="dateHeatmapChart"></canvas>
                </div>
            </div>

            <table class="data-table">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Day</th>
                        <th>Total Records</th>
                        <th>Employees Active</th>
                        <th>Avg Records/Employee</th>
                        <th>Peak Hour</th>
                    </tr>
                </thead>
                <tbody id="dateWiseBody">
                    <tr><td colspan="6" class="no-data">No data available</td></tr>
                </tbody>
            </table>
        </div>

        <!-- ===== ENHANCED COMPARISON SECTION ===== -->
        <div class="details-section" id="enhancedComparisonSection" style="display:none;">
            <div class="details-title">
                <i class="fas fa-chart-area"></i> Enhanced Field Comparison
            </div>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 20px;">
                <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 10px;">
                    <div style="font-size: 12px; opacity: 0.9; text-transform: uppercase; letter-spacing: 1px;">Field 1 Insights</div>
                    <div style="font-size: 24px; font-weight: bold; margin: 10px 0;" id="compField1Insights">-</div>
                    <div style="font-size: 13px; opacity: 0.85;" id="compField1Stats">-</div>
                </div>
                <div style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); color: white; padding: 20px; border-radius: 10px;">
                    <div style="font-size: 12px; opacity: 0.9; text-transform: uppercase; letter-spacing: 1px;">Field 2 Insights</div>
                    <div style="font-size: 24px; font-weight: bold; margin: 10px 0;" id="compField2Insights">-</div>
                    <div style="font-size: 13px; opacity: 0.85;" id="compField2Stats">-</div>
                </div>
            </div>

            <div style="background: white; padding: 25px; border-radius: 10px; box-shadow: 0 4px 15px rgba(0,0,0,0.05); border-top: 4px solid #1abc9c; margin-bottom: 20px;">
                <div style="font-weight: 600; color: #2c3e50; margin-bottom: 20px; display: flex; align-items: center; justify-content: space-between;">
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <i class="fas fa-chart-area" style="color: #1abc9c; font-size: 18px;"></i> Comparative Distribution
                    </div>
                    <div style="display: flex; gap: 8px;">
                        <button onclick="switchComparisonVisualization('bar')" style="padding: 6px 12px; background: #1abc9c; color: white; border: none; border-radius: 6px; cursor: pointer; font-weight: 600;" id="btnCompBar">
                            <i class="fas fa-chart-bar"></i> Bar
                        </button>
                        <button onclick="switchComparisonVisualization('radar')" style="padding: 6px 12px; background: #f0f0f0; color: #555; border: none; border-radius: 6px; cursor: pointer; font-weight: 600;" id="btnCompRadar">
                            <i class="fas fa-chart-area"></i> Radar
                        </button>
                        <button onclick="switchComparisonVisualization('pie')" style="padding: 6px 12px; background: #f0f0f0; color: #555; border: none; border-radius: 6px; cursor: pointer; font-weight: 600;" id="btnCompPie">
                            <i class="fas fa-chart-pie"></i> Pie
                        </button>
                    </div>
                </div>
                <div style="position: relative; height: 400px; width: 100%;">
                    <canvas id="enhancedComparisonChart"></canvas>
                </div>
            </div>

            <div style="background: white; padding: 25px; border-radius: 10px; box-shadow: 0 4px 15px rgba(0,0,0,0.05); border-top: 4px solid #1abc9c;">
                <div style="font-weight: 600; color: #2c3e50; margin-bottom: 15px;">📊 Statistical Comparison</div>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Metric</th>
                            <th id="statField1NameEnhanced">Field 1</th>
                            <th id="statField2NameEnhanced">Field 2</th>
                            <th>Difference</th>
                            <th>% Variance</th>
                            <th>Winner</th>
                        </tr>
                    </thead>
                    <tbody id="enhancedComparisonBody">
                        <tr><td colspan="6" class="no-data">Select two fields to compare...</td></tr>
                    </tbody>
                </table>
            </div>
        </div>

    </div>
</div>

<script>
    // ===== CHART INSTANCES =====
    let dailyProductionChart = null;
    let fieldDistributionChart = null;

    // ===== TOGGLE SIDEBAR =====
    function toggleSidebar() {
        const sidebar = document.querySelector('.sidebar');
        const mainContent = document.querySelector('.main-content');
        
        if (sidebar.classList.contains('active')) {
            sidebar.classList.remove('active');
            mainContent.classList.remove('shifted');
        } else {
            sidebar.classList.add('active');
            mainContent.classList.add('shifted');
        }
    }

    // ===== LOAD ANALYTICS FIELDS DROPDOWN =====
    async function loadFieldsDropdown() {
        try {
            const response = await fetch('api.php?action=get_analytics_fields');
            const result = await response.json();
            
            if (result.success) {
                const fieldSelect = document.getElementById('fieldSelect');
                const machineSelect = document.getElementById('machineFieldSelect');
                const compareField1 = document.getElementById('compareField1');
                const compareField2 = document.getElementById('compareField2');
                
                fieldSelect.innerHTML = '<option value="">-- Choose a field --</option>';
                machineSelect.innerHTML = '<option value="">-- Choose machine field --</option>';
                compareField1.innerHTML = '<option value="">-- Choose first field --</option>';
                compareField2.innerHTML = '<option value="">-- Choose second field --</option>';
                
                result.fields.forEach(field => {
                    const option = document.createElement('option');
                    option.value = field.field_name;
                    option.textContent = field.field_label;
                    fieldSelect.appendChild(option);
                    
                    // Add to machine dropdown
                    const option2 = option.cloneNode(true);
                    machineSelect.appendChild(option2);
                    
                    // Add to comparison dropdowns
                    const option3 = option.cloneNode(true);
                    compareField1.appendChild(option3);
                    
                    const option4 = option.cloneNode(true);
                    compareField2.appendChild(option4);
                });
            }
        } catch (error) {
            console.error('Error loading fields:', error);
        }
    }

    // ===== LOAD DASHBOARD STATISTICS =====
    async function loadDashboardStats() {
        try {
            const response = await fetch('api.php?action=get_analytics_dashboard');
            const result = await response.json();
            
            if (result.success) {
                document.getElementById('totalRecords').textContent = result.total_records;
                document.getElementById('employeeCount').textContent = result.employee_count;
                document.getElementById('dataPoints').textContent = result.date_stats.length;
                
                // Build Daily Production Chart
                buildDailyProductionChart(result.date_stats);
                
                // Build Top Employees Table
                buildTopEmployeesTable(result.top_employees);
            }
        } catch (error) {
            console.error('Error loading dashboard stats:', error);
        }
    }

    // ===== BUILD DAILY PRODUCTION CHART =====
    function buildDailyProductionChart(dateStats) {
        const ctx = document.getElementById('dailyProductionChart');
        if (!ctx) return;
        
        const labels = dateStats.map(d => d.log_date);
        const data = dateStats.map(d => d.count);
        
        if (dailyProductionChart) {
            dailyProductionChart.destroy();
        }
        
        dailyProductionChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Production Records',
                    data: data,
                    backgroundColor: 'rgba(102, 126, 234, 0.6)',
                    borderColor: 'rgba(102, 126, 234, 1)',
                    borderWidth: 1,
                    borderRadius: 6
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                plugins: {
                    legend: { display: false }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: { stepSize: 1 }
                    }
                }
            }
        });
    }

    // ===== BUILD TOP EMPLOYEES TABLE =====
    function buildTopEmployeesTable(topEmployees) {
        const tbody = document.getElementById('topEmployeesBody');
        
        if (!topEmployees || topEmployees.length === 0) {
            tbody.innerHTML = '<tr><td colspan="3" class="no-data">No employee data</td></tr>';
            return;
        }
        
        tbody.innerHTML = topEmployees.map((emp, idx) => `
            <tr>
                <td><span class="badge">${emp.emp_id}</span></td>
                <td><strong>${emp.operator_name}</strong></td>
                <td>${emp.count}</td>
            </tr>
        `).join('');
    }

    // ===== LOAD FIELD STATISTICS =====
    async function loadFieldStatistics() {
        const fieldName = document.getElementById('fieldSelect').value;
        
        if (!fieldName) {
            document.getElementById('fieldDistributionSection').style.display = 'none';
            return;
        }
        
        try {
            const response = await fetch(`api.php?action=get_field_statistics&field_name=${fieldName}`);
            const result = await response.json();
            
            if (result.success) {
                document.getElementById('fieldDistributionTitle').textContent = 
                    `${result.field_label} - Distribution`;
                
                buildFieldDistributionTable(result.data, result.total_records);
                buildFieldDistributionChart(result.data, result.field_label);
                
                document.getElementById('fieldDistributionSection').style.display = 'block';
            }
        } catch (error) {
            console.error('Error loading field statistics:', error);
        }
    }

    // ===== BUILD FIELD DISTRIBUTION TABLE =====
    function buildFieldDistributionTable(data, total) {
        const tbody = document.getElementById('fieldDistributionBody');
        
        if (!data || data.length === 0) {
            tbody.innerHTML = '<tr><td colspan="3" class="no-data">No data available</td></tr>';
            return;
        }
        
        tbody.innerHTML = data.map(item => {
            const percentage = ((item.count / total) * 100).toFixed(1);
            return `
                <tr>
                    <td><strong>${item.value || 'N/A'}</strong></td>
                    <td>${item.count}</td>
                    <td>
                        <div style="display: flex; align-items: center; gap: 8px;">
                            <div style="width: 100px; height: 6px; background: #ddd; border-radius: 3px; overflow: hidden;">
                                <div style="width: ${percentage}%; height: 100%; background: #00a65a;"></div>
                            </div>
                            ${percentage}%
                        </div>
                    </td>
                </tr>
            `;
        }).join('');
    }

    // ===== BUILD FIELD DISTRIBUTION CHART =====
    function buildFieldDistributionChart(data, label) {
        const ctx = document.getElementById('fieldDistributionChart');
        if (!ctx) return;
        
        const labels = data.slice(0, 10).map(d => d.value || 'N/A');
        const values = data.slice(0, 10).map(d => d.count);
        
        const colors = [
            '#667eea', '#764ba2', '#f093fb', '#f5576c', '#4facfe',
            '#00f2fe', '#43e97b', '#38f9d7', '#fa709a', '#fee140'
        ];
        
        if (fieldDistributionChart) {
            fieldDistributionChart.destroy();
        }
        
        fieldDistributionChart = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [{
                    data: values,
                    backgroundColor: colors.slice(0, labels.length),
                    borderColor: 'white',
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: { padding: 15, font: { size: 12 } }
                    }
                }
            }
        });
    }

    // ===== LOAD MACHINE DATA =====
    async function loadMachineData() {
        const machineField = document.getElementById('machineFieldSelect').value;
        
        if (!machineField) {
            document.getElementById('machineSection').style.display = 'none';
            document.getElementById('employeeMachineSection').style.display = 'none';
            return;
        }
        
        try {
            // Load machine statistics
            const machineResponse = await fetch(`api.php?action=get_machine_tool_list&machine_field=${machineField}`);
            const machineResult = await machineResponse.json();
            
            if (machineResult.success) {
                buildMachineTable(machineResult.machines);
                document.getElementById('machineSection').style.display = 'block';
            }
            
            // Load employee-machine mapping
            const mappingResponse = await fetch(`api.php?action=get_employee_machine_mapping&machine_field=${machineField}`);
            const mappingResult = await mappingResponse.json();
            
            if (mappingResult.success) {
                buildEmployeeMachineTable(mappingResult.mappings);
                document.getElementById('employeeMachineSection').style.display = 'block';
            }
        } catch (error) {
            console.error('Error loading machine data:', error);
        }
    }

    // ===== BUILD MACHINE TABLE WITH ACTION BUTTON =====
    function buildMachineTable(machines) {
        const tbody = document.getElementById('machineTableBody');
        
        if (!machines || machines.length === 0) {
            tbody.innerHTML = '<tr><td colspan="4" class="no-data">No machine data</td></tr>';
            return;
        }
        
        tbody.innerHTML = machines.map(machine => `
            <tr>
                <td><strong>${machine.machine || 'N/A'}</strong></td>
                <td><span class="badge">${machine.employee_count}</span></td>
                <td>${machine.total_operations}</td>
                <td>
                    <button onclick="loadMachineDetail('${machine.machine}')" class="btn-load" style="padding: 6px 12px; background: #17a2b8; font-size: 12px;">
                        <i class="fas fa-search"></i> View Detail
                    </button>
                </td>
            </tr>
        `).join('');
    }

    // ===== BUILD EMPLOYEE-MACHINE TABLE =====
    function buildEmployeeMachineTable(mappings) {
        const tbody = document.getElementById('employeeMachineBody');
        
        if (!mappings || mappings.length === 0) {
            tbody.innerHTML = '<tr><td colspan="3" class="no-data">No assignment data</td></tr>';
            return;
        }
        
        tbody.innerHTML = mappings.map(mapping => `
            <tr>
                <td><span class="badge">${mapping.emp_id}</span></td>
                <td><strong>${mapping.operator_name}</strong></td>
                <td>
                    <div style="display: flex; flex-wrap: wrap; gap: 5px;">
                        ${mapping.machines.slice(0, 3).map(m => 
                            `<span class="badge" style="background: #fff3cd; color: #856404;">${m}</span>`
                        ).join('')}
                        ${mapping.machines.length > 3 ? `<span class="badge">+${mapping.machines.length - 3}</span>` : ''}
                    </div>
                </td>
            </tr>
        `).join('');
    }

    // ===== LOAD FIELD COMPARISON =====
    async function loadFieldComparison() {
        const field1 = document.getElementById('compareField1').value;
        const field2 = document.getElementById('compareField2').value;

        if (!field1 || !field2) {
            alert('Please select both fields to compare');
            return;
        }

        if (field1 === field2) {
            alert('Please select different fields');
            return;
        }

        console.log('Loading comparison for:', field1, field2);

        try {
            // Fetch data for both fields
            const response1 = await fetch(`api.php?action=get_field_statistics&field_name=${field1}`);
            const response2 = await fetch(`api.php?action=get_field_statistics&field_name=${field2}`);
            
            const data1 = await response1.json();
            const data2 = await response2.json();

            if (!data1.success || !data2.success) {
                throw new Error(`API Error`);
            }

            // Update headers
            const field1Label = document.getElementById('compareField1').options[document.getElementById('compareField1').selectedIndex].text;
            const field2Label = document.getElementById('compareField2').options[document.getElementById('compareField2').selectedIndex].text;

            document.getElementById('compField1Name').textContent = field1Label;
            document.getElementById('compField2Name').textContent = field2Label;
            document.getElementById('compField1Count').textContent = data1.data.length || 0;
            document.getElementById('compField2Count').textContent = data2.data.length || 0;
            
            document.getElementById('statField1Name').textContent = field1Label;
            document.getElementById('statField2Name').textContent = field2Label;

            // Build comparison charts
            try {
                buildComparisonCharts(data1, data2, field1Label, field2Label);
                buildComparisonStats(data1, data2, field1Label, field2Label);
            } catch (chartError) {
                console.error('Chart building error:', chartError);
            }

            // Show comparison section
            document.getElementById('comparisonSection').style.display = 'block';
            document.getElementById('comparisonSection').scrollIntoView({ behavior: 'smooth' });

        } catch (error) {
            console.error('Error loading field comparison:', error);
            alert('Error: ' + error.message);
        }
    }



    // ===== CALCULATE STATISTICS =====
    function calculateStats(fieldData) {
        const data = fieldData.data || [];
        
        if (data.length === 0) {
            return {
                uniqueValues: 0,
                totalCount: 0,
                maxFreq: 0,
                avgFreq: 0
            };
        }

        const uniqueValues = data.length;
        const totalCount = data.reduce((sum, item) => sum + (item.count || 0), 0);
        const maxFreq = Math.max(...data.map(item => item.count || 0));
        const avgFreq = totalCount > 0 ? totalCount / uniqueValues : 0;

        return {
            uniqueValues,
            totalCount,
            maxFreq,
            avgFreq
        };
    }



    // ===== LOAD ALL ANALYTICS =====
    async function loadAllAnalytics() {
        await loadDashboardStats();
        await loadFieldStatistics();
        await loadMachineData();
    }

    // ===== LOAD DATE-WISE REPORT =====
    async function loadDateWiseReport() {
        try {
            const startDate = document.getElementById('dateRangeStart').value || null;
            const endDate = document.getElementById('dateRangeEnd').value || null;
            
            let url = 'api.php?action=get_date_wise_statistics';
            if (startDate) url += `&start_date=${startDate}`;
            if (endDate) url += `&end_date=${endDate}`;
            
            const response = await fetch(url);
            const result = await response.json();
            
            if (result.success) {
                buildDateWiseTable(result.data);
                buildDateHeatmap(result.data);
                document.getElementById('dateWiseSection').style.display = 'block';
                document.getElementById('dateWiseSection').scrollIntoView({ behavior: 'smooth' });
            }
        } catch (error) {
            console.error('Error loading date-wise report:', error);
            alert('Error: ' + error.message);
        }
    }

    // ===== BUILD DATE-WISE TABLE =====
    function buildDateWiseTable(dateData) {
        const tbody = document.getElementById('dateWiseBody');
        
        if (!dateData || dateData.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6" class="no-data">No data available</td></tr>';
            return;
        }
        
        tbody.innerHTML = dateData.map(row => {
            const days = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
            const date = new Date(row.log_date);
            const dayName = days[date.getDay()];
            const avgRecords = (row.total_records / (row.employee_count || 1)).toFixed(2);
            
            return `
                <tr>
                    <td><strong>${row.log_date}</strong></td>
                    <td>${dayName}</td>
                    <td style="font-weight: 600; color: #667eea;">${row.total_records}</td>
                    <td><span class="badge">${row.employee_count}</span></td>
                    <td>${avgRecords}</td>
                    <td>${row.peak_hour || 'N/A'}</td>
                </tr>
            `;
        }).join('');
    }

    // ===== BUILD DATE HEATMAP CHART =====
    function buildDateHeatmap(dateData) {
        const ctx = document.getElementById('dateHeatmapChart');
        if (!ctx) return;
        
        const labels = dateData.map(d => d.log_date);
        const recordCounts = dateData.map(d => d.total_records);
        const employeeCounts = dateData.map(d => d.employee_count);
        
        if (window.dateHeatmapChart) {
            window.dateHeatmapChart.destroy();
        }
        
        window.dateHeatmapChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: 'Total Records',
                        data: recordCounts,
                        backgroundColor: 'rgba(102, 126, 234, 0.8)',
                        borderColor: 'rgba(102, 126, 234, 1)',
                        borderWidth: 1,
                        yAxisID: 'y'
                    },
                    {
                        label: 'Employees Active',
                        data: employeeCounts,
                        backgroundColor: 'rgba(240, 147, 251, 0.6)',
                        borderColor: 'rgba(240, 147, 251, 1)',
                        borderWidth: 1,
                        yAxisID: 'y1'
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                interaction: { mode: 'index', intersect: false },
                scales: {
                    y: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                        title: { display: true, text: 'Records' }
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        title: { display: true, text: 'Employees' },
                        grid: { drawOnChartArea: false }
                    }
                },
                plugins: {
                    legend: { position: 'top' }
                }
            }
        });
    }

    // ===== SWITCH COMPARISON VISUALIZATION =====
    function switchComparisonVisualization(chartType) {
        if (!window.comparisonChartData || !window.comparisonChartData.data1) {
            return;
        }

        // Update button states
        document.getElementById('btnCompBar').style.background = chartType === 'bar' ? '#1abc9c' : '#f0f0f0';
        document.getElementById('btnCompBar').style.color = chartType === 'bar' ? 'white' : '#555';
        document.getElementById('btnCompRadar').style.background = chartType === 'radar' ? '#1abc9c' : '#f0f0f0';
        document.getElementById('btnCompRadar').style.color = chartType === 'radar' ? 'white' : '#555';
        document.getElementById('btnCompPie').style.background = chartType === 'pie' ? '#1abc9c' : '#f0f0f0';
        document.getElementById('btnCompPie').style.color = chartType === 'pie' ? 'white' : '#555';

        window.comparisonChartData.currentType = chartType;

        if (chartType === 'bar') {
            buildEnhancedBarComparison(window.comparisonChartData.data1, window.comparisonChartData.data2, window.comparisonChartData.label1, window.comparisonChartData.label2);
        } else if (chartType === 'radar') {
            buildRadarComparison(window.comparisonChartData.data1, window.comparisonChartData.data2, window.comparisonChartData.label1, window.comparisonChartData.label2);
        } else if (chartType === 'pie') {
            buildPieComparison(window.comparisonChartData.data1, window.comparisonChartData.data2, window.comparisonChartData.label1, window.comparisonChartData.label2);
        }
    }

    // ===== BUILD ENHANCED BAR COMPARISON =====
    function buildEnhancedBarComparison(data1, data2, label1, label2) {
        try {
            const canvas = document.getElementById('enhancedComparisonChart');
            if (!canvas) return;

            if (window.enhancedComparisonChart) {
                window.enhancedComparisonChart.destroy();
            }

            const values1 = (data1.data || []).slice(0, 8);
            const values2 = (data2.data || []).slice(0, 8);
            
            const labels1 = values1.map(d => String(d.value || 'N/A').substring(0, 12));
            const labels2 = values2.map(d => String(d.value || 'N/A').substring(0, 12));
            
            const allLabels = [...labels1, ...labels2];
            const data1Values = [...values1.map(d => d.count), ...new Array(values2.length).fill(null)];
            const data2Values = [...new Array(values1.length).fill(null), ...values2.map(d => d.count)];

            window.enhancedComparisonChart = new Chart(canvas, {
                type: 'bar',
                data: {
                    labels: allLabels,
                    datasets: [
                        {
                            label: label1,
                            data: data1Values,
                            backgroundColor: '#667eea',
                            borderColor: '#667eea',
                            borderWidth: 2,
                            borderRadius: 6
                        },
                        {
                            label: label2,
                            data: data2Values,
                            backgroundColor: '#f093fb',
                            borderColor: '#f093fb',
                            borderWidth: 2,
                            borderRadius: 6
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    indexAxis: 'x',
                    scales: {
                        x: {
                            ticks: { autoSkip: false, maxRotation: 45, minRotation: 0 },
                            grid: { display: false }
                        },
                        y: { beginAtZero: true }
                    },
                    plugins: {
                        legend: { position: 'top' },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    if (context.raw === null) return null;
                                    return context.dataset.label + ': ' + context.raw;
                                }
                            }
                        }
                    }
                }
            });
        } catch (e) {
            console.error('Error in buildEnhancedBarComparison:', e);
        }
    }

    // ===== BUILD RADAR COMPARISON =====
    function buildRadarComparison(data1, data2, label1, label2) {
        try {
            const canvas = document.getElementById('enhancedComparisonChart');
            if (!canvas) return;

            if (window.enhancedComparisonChart) {
                window.enhancedComparisonChart.destroy();
            }

            const values1 = (data1.data || []).slice(0, 6);
            const values2 = (data2.data || []).slice(0, 6);
            
            const maxLen = Math.max(values1.length, values2.length);
            
            // Pad to same length
            while (values1.length < maxLen) values1.push({value: 'N/A', count: 0});
            while (values2.length < maxLen) values2.push({value: 'N/A', count: 0});
            
            const labels = values1.map(d => String(d.value || 'N/A').substring(0, 12));

            window.enhancedComparisonChart = new Chart(canvas, {
                type: 'radar',
                data: {
                    labels: labels,
                    datasets: [
                        {
                            label: label1,
                            data: values1.map(d => d.count),
                            borderColor: '#667eea',
                            backgroundColor: 'rgba(102, 126, 234, 0.2)',
                            borderWidth: 2
                        },
                        {
                            label: label2,
                            data: values2.map(d => d.count),
                            borderColor: '#f093fb',
                            backgroundColor: 'rgba(240, 147, 251, 0.2)',
                            borderWidth: 2
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        r: { beginAtZero: true }
                    },
                    plugins: {
                        legend: { position: 'top' }
                    }
                }
            });
        } catch (e) {
            console.error('Error in buildRadarComparison:', e);
        }
    }

    // ===== BUILD PIE COMPARISON =====
    function buildPieComparison(data1, data2, label1, label2) {
        try {
            const canvas = document.getElementById('enhancedComparisonChart');
            if (!canvas) return;

            if (window.enhancedComparisonChart) {
                window.enhancedComparisonChart.destroy();
            }

            const values1 = (data1.data || []).slice(0, 6);
            const values2 = (data2.data || []).slice(0, 6);
            
            const labels1 = values1.map(d => String(d.value || 'N/A').substring(0, 10));
            const labels2 = values2.map(d => String(d.value || 'N/A').substring(0, 10));
            
            // Create side-by-side pie representation (nested)
            const allLabels = [...labels1, ...labels2];
            const allCounts = [...values1.map(d => d.count), ...values2.map(d => d.count)];

            const colors = [
                '#667eea', '#764ba2', '#f093fb', '#f5576c', '#4facfe', '#00f2fe',
                '#667eea', '#764ba2', '#f093fb', '#f5576c', '#4facfe', '#00f2fe'
            ];

            window.enhancedComparisonChart = new Chart(canvas, {
                type: 'doughnut',
                data: {
                    labels: allLabels,
                    datasets: [
                        {
                            label: [label1, label2],
                            data: allCounts,
                            backgroundColor: colors.slice(0, allLabels.length),
                            borderColor: 'white',
                            borderWidth: 2
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'right',
                            labels: { padding: 15, font: { size: 11 } }
                        },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    const percentage = ((context.raw / total) * 100).toFixed(1);
                                    return context.label + ': ' + context.raw + ' (' + percentage + '%)';
                                }
                            }
                        }
                    }
                }
            });
        } catch (e) {
            console.error('Error in buildPieComparison:', e);
        }
    }

    // ===== ENHANCED BUILD COMPARISON STATS =====
    function buildEnhancedComparisonStats(data1, data2, label1, label2) {
        const tbody = document.getElementById('enhancedComparisonBody');
        if (!tbody) return;

        try {
            const stats1 = calculateStats(data1);
            const stats2 = calculateStats(data2);

            const metrics = [
                { name: 'Unique Values', v1: stats1.uniqueValues, v2: stats2.uniqueValues },
                { name: 'Total Records', v1: stats1.totalCount, v2: stats2.totalCount },
                { name: 'Max Frequency', v1: stats1.maxFreq, v2: stats2.maxFreq },
                { name: 'Avg Frequency', v1: stats1.avgFreq.toFixed(2), v2: stats2.avgFreq.toFixed(2) },
                { name: 'Diversity Index', v1: (stats1.uniqueValues / (stats1.totalCount || 1) * 100).toFixed(2), v2: (stats2.uniqueValues / (stats2.totalCount || 1) * 100).toFixed(2) }
            ];

            tbody.innerHTML = metrics.map(m => {
                const v1 = parseFloat(m.v1);
                const v2 = parseFloat(m.v2);
                const diff = (v1 - v2).toFixed(2);
                const pctChange = v2 ? ((v1 - v2) / v2 * 100).toFixed(1) : 0;
                const winner = v1 > v2 ? '🟦 Field 1' : v1 < v2 ? '🟪 Field 2' : '⚖️ Equal';
                
                return `
                    <tr>
                        <td><strong>${m.name}</strong></td>
                        <td>${m.v1}</td>
                        <td>${m.v2}</td>
                        <td style="color: ${diff > 0 ? '#43e97b' : '#f5576c'}; font-weight: 600;">${diff > 0 ? '+' : ''}${diff}</td>
                        <td style="color: ${pctChange > 0 ? '#43e97b' : '#f5576c'}; font-weight: 600;">${pctChange > 0 ? '+' : ''}${pctChange}%</td>
                        <td>${winner}</td>
                    </tr>
                `;
            }).join('');

            // Update insights cards
            document.getElementById('compField1Insights').textContent = stats1.uniqueValues;
            document.getElementById('compField1Stats').innerHTML = `<strong>${stats1.totalCount}</strong> total records<br><strong>${stats1.maxFreq}</strong> max frequency`;
            
            document.getElementById('compField2Insights').textContent = stats2.uniqueValues;
            document.getElementById('compField2Stats').innerHTML = `<strong>${stats2.totalCount}</strong> total records<br><strong>${stats2.maxFreq}</strong> max frequency`;

            document.getElementById('statField1NameEnhanced').textContent = label1;
            document.getElementById('statField2NameEnhanced').textContent = label2;
        } catch (e) {
            console.error('Error in buildEnhancedComparisonStats:', e);
        }
    }

    // ===== UPDATE LOAD FIELD COMPARISON =====
    async function loadFieldComparison() {
        const field1 = document.getElementById('compareField1').value;
        const field2 = document.getElementById('compareField2').value;

        if (!field1 || !field2) {
            alert('Please select both fields to compare');
            return;
        }

        if (field1 === field2) {
            alert('Please select different fields');
            return;
        }

        try {
            const response1 = await fetch(`api.php?action=get_field_statistics&field_name=${field1}`);
            const response2 = await fetch(`api.php?action=get_field_statistics&field_name=${field2}`);
            
            const data1 = await response1.json();
            const data2 = await response2.json();

            if (!data1.success || !data2.success) {
                throw new Error('Failed to load field data');
            }

            const field1Label = document.getElementById('compareField1').options[document.getElementById('compareField1').selectedIndex].text;
            const field2Label = document.getElementById('compareField2').options[document.getElementById('compareField2').selectedIndex].text;

            window.comparisonChartData = {
                data1: data1,
                data2: data2,
                label1: field1Label,
                label2: field2Label,
                currentType: 'bar'
            };

            buildEnhancedBarComparison(data1, data2, field1Label, field2Label);
            buildEnhancedComparisonStats(data1, data2, field1Label, field2Label);

            document.getElementById('enhancedComparisonSection').style.display = 'block';
            document.getElementById('enhancedComparisonSection').scrollIntoView({ behavior: 'smooth' });

        } catch (error) {
            console.error('Error:', error);
            alert('Error: ' + error.message);
        }
    }

    // ===== LOAD MACHINE DETAIL ANALYSIS (NEW) =====
    async function loadMachineDetail(machineName) {
        try {
            const machineField = document.getElementById('machineFieldSelect').value;
            if (!machineField) {
                alert('Please select a machine field first');
                return;
            }

            const response = await fetch(`api.php?action=get_machine_detail_analysis&machine_field=${machineField}&machine_name=${encodeURIComponent(machineName)}`);
            const result = await response.json();

            if (result.success) {
                // Update title
                document.getElementById('machineDetailTitle').textContent = `Machine: ${machineName} - Detailed Analysis`;

                // Update summary cards
                document.getElementById('machineDetailOpsCount').textContent = result.summary.total_operations;
                document.getElementById('machineDetailEmpCount').textContent = result.summary.employee_count;
                document.getElementById('machineDetailToolsCount').textContent = result.summary.tools_count;
                const avgDaily = parseFloat(result.summary.avg_daily_operations) || 0;
                document.getElementById('machineDetailDailyPlans').textContent = avgDaily.toFixed(1);

                // Build tables
                buildMachineToolsTable(result.tools, result.summary.total_operations);
                buildMachineEmployeesTable(result.employees, result.summary.total_operations);
                buildMachineDailyPlansTable(result.daily_data);

                // Build chart
                buildMachineOperationsChart(result.daily_data, 'bar');

                // Store data for chart switching
                window.machineChartData = {
                    daily_data: result.daily_data,
                    machine_name: machineName,
                    currentType: 'bar'
                };

                // Show section
                document.getElementById('machineDetailSection').style.display = 'block';
                document.getElementById('machineDetailSection').scrollIntoView({ behavior: 'smooth' });
            } else {
                alert('Error: ' + result.error);
            }
        } catch (error) {
            console.error('Error loading machine detail:', error);
            alert('Error: ' + error.message);
        }
    }

    // ===== BUILD MACHINE TOOLS TABLE =====
    function buildMachineToolsTable(tools, totalOps) {
        const tbody = document.getElementById('machineToolsBody');
        
        if (!tools || tools.length === 0) {
            tbody.innerHTML = '<tr><td colspan="3" class="no-data">No tools data</td></tr>';
            return;
        }
        
        tbody.innerHTML = tools.map(tool => {
            const percentage = ((tool.count / totalOps) * 100).toFixed(1);
            return `
                <tr>
                    <td><strong>${tool.tool_name || 'N/A'}</strong></td>
                    <td>${tool.count}</td>
                    <td>
                        <div style="display: flex; align-items: center; gap: 8px;">
                            <div style="width: 100px; height: 6px; background: #ddd; border-radius: 3px; overflow: hidden;">
                                <div style="width: ${percentage}%; height: 100%; background: #667eea;"></div>
                            </div>
                            ${percentage}%
                        </div>
                    </td>
                </tr>
            `;
        }).join('');
    }

    // ===== BUILD MACHINE EMPLOYEES TABLE =====
    function buildMachineEmployeesTable(employees, totalOps) {
        const tbody = document.getElementById('machineEmployeesBody');
        
        if (!employees || employees.length === 0) {
            tbody.innerHTML = '<tr><td colspan="4" class="no-data">No employees data</td></tr>';
            return;
        }
        
        tbody.innerHTML = employees.map(emp => {
            const percentage = ((emp.count / totalOps) * 100).toFixed(1);
            return `
                <tr>
                    <td><span class="badge">${emp.emp_id}</span></td>
                    <td><strong>${emp.operator_name}</strong></td>
                    <td>${emp.count}</td>
                    <td>
                        <div style="display: flex; align-items: center; gap: 8px;">
                            <div style="width: 100px; height: 6px; background: #ddd; border-radius: 3px; overflow: hidden;">
                                <div style="width: ${percentage}%; height: 100%; background: #f093fb;"></div>
                            </div>
                            ${percentage}%
                        </div>
                    </td>
                </tr>
            `;
        }).join('');
    }

    // ===== BUILD MACHINE DAILY PLANS TABLE =====
    function buildMachineDailyPlansTable(dailyData) {
        const tbody = document.getElementById('machineDailyPlansBody');
        
        if (!dailyData || dailyData.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6" class="no-data">No daily data</td></tr>';
            return;
        }
        
        tbody.innerHTML = dailyData.map(row => {
            const days = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
            const date = new Date(row.log_date);
            const dayName = days[date.getDay()];
            const avgPerEmp = (row.total_operations / (row.employee_count || 1)).toFixed(1);
            
            return `
                <tr>
                    <td><strong>${row.log_date}</strong></td>
                    <td>${dayName}</td>
                    <td style="font-weight: 600; color: #4facfe;">${row.total_operations}</td>
                    <td><span class="badge">${row.employee_count}</span></td>
                    <td><span class="badge" style="background: #fff3cd; color: #856404;">${row.tools_count}</span></td>
                    <td>${avgPerEmp}</td>
                </tr>
            `;
        }).join('');
    }

    // ===== BUILD MACHINE OPERATIONS CHART (BAR) =====
    function buildMachineOperationsChart(dailyData, chartType) {
        const ctx = document.getElementById('machineOperationsChart');
        if (!ctx) return;

        // Safely destroy existing chart
        if (window.machineOperationsChart && typeof window.machineOperationsChart.destroy === 'function') {
            window.machineOperationsChart.destroy();
        }

        const labels = dailyData.map(d => d.log_date);
        const operations = dailyData.map(d => d.total_operations);
        const employees = dailyData.map(d => d.employee_count);
        const tools = dailyData.map(d => d.tools_count);

        if (chartType === 'bar') {
            window.machineOperationsChart = new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [
                        {
                            label: 'Total Operations',
                            data: operations,
                            backgroundColor: '#4facfe',
                            borderColor: '#4facfe',
                            borderWidth: 1,
                            borderRadius: 6
                        },
                        {
                            label: 'Employees',
                            data: employees,
                            backgroundColor: '#f093fb',
                            borderColor: '#f093fb',
                            borderWidth: 1,
                            borderRadius: 6
                        },
                        {
                            label: 'Tools Used',
                            data: tools,
                            backgroundColor: '#43e97b',
                            borderColor: '#43e97b',
                            borderWidth: 1,
                            borderRadius: 6
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        x: { grid: { display: false } },
                        y: { beginAtZero: true }
                    },
                    plugins: {
                        legend: { position: 'top' }
                    }
                }
            });
        } else if (chartType === 'line') {
            window.machineOperationsChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: labels,
                    datasets: [
                        {
                            label: 'Total Operations',
                            data: operations,
                            borderColor: '#4facfe',
                            backgroundColor: 'rgba(79, 172, 254, 0.1)',
                            borderWidth: 3,
                            tension: 0.4,
                            fill: true
                        },
                        {
                            label: 'Employees',
                            data: employees,
                            borderColor: '#f093fb',
                            backgroundColor: 'rgba(240, 147, 251, 0.1)',
                            borderWidth: 3,
                            tension: 0.4,
                            fill: true
                        },
                        {
                            label: 'Tools Used',
                            data: tools,
                            borderColor: '#43e97b',
                            backgroundColor: 'rgba(67, 233, 123, 0.1)',
                            borderWidth: 3,
                            tension: 0.4,
                            fill: true
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: {
                        y: { beginAtZero: true }
                    },
                    plugins: {
                        legend: { position: 'top' }
                    }
                }
            });
        } else if (chartType === 'pie') {
            const totalOps = operations.reduce((a, b) => a + b, 0);
            const labels2 = dailyData.map(d => d.log_date);
            
            window.machineOperationsChart = new Chart(ctx, {
                type: 'doughnut',
                data: {
                    labels: labels2,
                    datasets: [{
                        label: 'Operations per Day',
                        data: operations,
                        backgroundColor: [
                            '#4facfe', '#667eea', '#f093fb', '#43e97b', '#fa709a',
                            '#fee140', '#00f2fe', '#764ba2', '#f5576c', '#38f9d7'
                        ],
                        borderColor: 'white',
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { position: 'right' },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    const percentage = ((context.raw / totalOps) * 100).toFixed(1);
                                    return context.label + ': ' + context.raw + ' (' + percentage + '%)';
                                }
                            }
                        }
                    }
                }
            });
        }
    }

    // ===== SWITCH MACHINE CHART TYPE =====
    function switchMachineChart(chartType) {
        if (!window.machineChartData) return;

        document.getElementById('btnMachineBar').style.background = chartType === 'bar' ? '#4facfe' : '#f0f0f0';
        document.getElementById('btnMachineBar').style.color = chartType === 'bar' ? 'white' : '#555';
        document.getElementById('btnMachineLine').style.background = chartType === 'line' ? '#4facfe' : '#f0f0f0';
        document.getElementById('btnMachineLine').style.color = chartType === 'line' ? 'white' : '#555';
        document.getElementById('btnMachinePie').style.background = chartType === 'pie' ? '#4facfe' : '#f0f0f0';
        document.getElementById('btnMachinePie').style.color = chartType === 'pie' ? 'white' : '#555';

        window.machineChartData.currentType = chartType;
        buildMachineOperationsChart(window.machineChartData.daily_data, chartType);
    }

    // ===== CLOSE MACHINE DETAIL =====
    function closeMachineDetail() {
        document.getElementById('machineDetailSection').style.display = 'none';
        if (window.machineOperationsChart && typeof window.machineOperationsChart.destroy === 'function') {
            window.machineOperationsChart.destroy();
            window.machineOperationsChart = null;
        }
    }

    // ===== INITIALIZE ON PAGE LOAD =====
    document.addEventListener('DOMContentLoaded', function() {
        console.log('✓ Analytics Dashboard Initialized');
        
        // Set default date range (last 30 days)
        const today = new Date();
        const thirtyDaysAgo = new Date(today.getTime() - (30 * 24 * 60 * 60 * 1000));
        
        document.getElementById('dateRangeStart').valueAsDate = thirtyDaysAgo;
        document.getElementById('dateRangeEnd').valueAsDate = today;
        
        loadFieldsDropdown();
        loadDashboardStats();
    });
    
</script>

</body>
</html>