<?php
// Protect this page with session check
include 'session_check.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mektron - Interactive Analytics</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <link rel="stylesheet" href="style.css">
    
    <style>
        /* ===== HEADER (UNCHANGED) ===== */
        body {
            background: #f4f6f9;
            font-family: 'Segoe UI', sans-serif;
            margin: 0;
            overflow-x: hidden;
        }

        .main-content {
            margin-left: 0;
            padding: 0;
            width: 100%;
            box-sizing: border-box;
            min-height: 100vh;
            transition: margin-left 0.3s ease;
        }

        .sidebar {
            position: fixed;
            top: 0;
            left: -260px;
            width: 260px;
            height: 100%;
            background: #013B3B;
            transition: left 0.3s ease;
            z-index: 1000;
            overflow-y: auto;
        }

        .sidebar.active { left: 0; }
        .main-content.shifted { margin-left: 260px; }

        .overlay {
            display: none;
            position: fixed;
            top: 0; left: 0; width: 100%; height: 100%;
            background: rgba(0,0,0,0.5);
            z-index: 999;
        }
        .overlay.active { display: block; }

        /* ===== HEADER SECTION (KEEP ORIGINAL) ===== */
        .analytics-header {
            background: white;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 20px 30px;
            border-bottom: 2px solid #eee;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .analytics-header h1 {
            margin: 0;
            color: #2c3e50;
            font-size: 28px;
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .menu-toggle-btn {
            background: none;
            border: none;
            font-size: 24px;
            color: #013B3B;
            cursor: pointer;
            padding: 5px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            border-radius: 4px;
            transition: 0.2s;
        }
        .menu-toggle-btn:hover { background: #e0e0e0; }

        /* ===== MAIN TAB NAVIGATION ===== */
        .tab-navigation {
            background: white;
            padding: 0;
            border-bottom: 3px solid #eee;
            display: flex;
            gap: 0;
            overflow-x: auto;
        }

        .tab-btn {
            flex: 1;
            padding: 18px 25px;
            border: none;
            background: white;
            cursor: pointer;
            font-size: 15px;
            font-weight: 600;
            color: #666;
            border-bottom: 4px solid transparent;
            transition: all 0.3s ease;
            white-space: nowrap;
            display: flex;
            align-items: center;
            gap: 10px;
            justify-content: center;
        }

        .tab-btn:hover {
            background: #f9f9f9;
            color: #013B3B;
        }

        .tab-btn.active {
            background: #f0f7f7;
            color: #013B3B;
            border-bottom-color: #00a65a;
        }

        /* ===== TAB CONTENT AREA ===== */
        .tab-content-wrapper {
            padding: 30px;
            min-height: calc(100vh - 200px);
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
            animation: fadeIn 0.3s ease;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        /* ===== INTERACTIVE SELECTION CARDS ===== */
        .selection-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .selection-card {
            background: white;
            border: 2px solid #eee;
            border-radius: 12px;
            padding: 25px;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(0,0,0,0.04);
        }

        .selection-card:hover {
            border-color: #00a65a;
            box-shadow: 0 8px 20px rgba(0,0,0,0.1);
            transform: translateY(-5px);
        }

        .selection-card.active {
            background: linear-gradient(135deg, #00a65a 0%, #008d4c 100%);
            border-color: #008d4c;
            color: white;
        }

        .selection-card-icon {
            font-size: 32px;
            margin-bottom: 15px;
            opacity: 0.7;
        }

        .selection-card.active .selection-card-icon {
            opacity: 1;
        }

        .selection-card h3 {
            margin: 0 0 10px 0;
            font-size: 18px;
            font-weight: 700;
            color: #2c3e50;
        }

        .selection-card.active h3 {
            color: white;
        }

        .selection-card p {
            margin: 0;
            font-size: 13px;
            color: #777;
            line-height: 1.5;
        }

        .selection-card.active p {
            color: rgba(255,255,255,0.9);
        }

        /* ===== ANALYTICS SECTION ===== */
        .analytics-section {
            background: white;
            border-radius: 12px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 2px 12px rgba(0,0,0,0.05);
            display: none;
        }

        .analytics-section.show {
            display: block;
            animation: slideIn 0.4s ease;
        }

        @keyframes slideIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        .section-title {
            font-size: 22px;
            font-weight: 700;
            color: #2c3e50;
            margin: 0 0 25px 0;
            display: flex;
            align-items: center;
            gap: 15px;
            border-bottom: 3px solid #f0f0f0;
            padding-bottom: 15px;
        }

        .section-title i {
            font-size: 28px;
            color: #00a65a;
        }

        /* ===== STATS CARDS (MODERN) ===== */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 25px;
            border-radius: 12px;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
            transition: all 0.3s ease;
            border: none;
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            right: -50px;
            width: 100px;
            height: 100px;
            background: rgba(255,255,255,0.1);
            border-radius: 50%;
        }

        .stat-card:hover {
            transform: translateY(-8px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
        }

        .stat-card.blue { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); }
        .stat-card.pink { background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); }
        .stat-card.cyan { background: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%); }
        .stat-card.green { background: linear-gradient(135deg, #43e97b 0%, #38f9d7 100%); }

        .stat-card-label {
            font-size: 12px;
            opacity: 0.9;
            text-transform: uppercase;
            letter-spacing: 1px;
            margin-bottom: 12px;
            font-weight: 600;
        }

        .stat-card-value {
            font-size: 36px;
            font-weight: 800;
            margin: 0;
        }

        .stat-card-detail {
            font-size: 12px;
            opacity: 0.8;
            margin-top: 10px;
        }

        /* ===== CHART CONTAINERS ===== */
        .chart-section {
            background: linear-gradient(160deg, #ffffff 0%, #f8fbff 60%, #f5fff9 100%);
            border: 1px solid rgba(102, 126, 234, 0.12);
            border-radius: 12px;
            padding: 25px;
            margin-bottom: 25px;
            box-shadow: 0 8px 24px rgba(15, 23, 42, 0.06);
        }

        .chart-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }

        .chart-title {
            font-size: 18px;
            font-weight: 700;
            color: #2c3e50;
            display: flex;
            align-items: center;
            gap: 10px;
            margin: 0;
        }

        .chart-controls {
            display: flex;
            gap: 10px;
        }

        .chart-btn {
            padding: 8px 16px;
            border: 2px solid #eee;
            background: #f9f9f9;
            color: #555;
            border-radius: 6px;
            cursor: pointer;
            font-size: 12px;
            font-weight: 600;
            transition: all 0.2s ease;
        }

        .chart-btn:hover {
            border-color: #00a65a;
            color: #00a65a;
        }

        .chart-btn.active {
            background: #00a65a;
            color: white;
            border-color: #00a65a;
        }

        .chart-canvas {
            position: relative;
            height: 350px;
            margin-bottom: 10px;
            padding: 10px;
            border-radius: 12px;
            border: 1px solid #e8edff;
            background: linear-gradient(180deg, #ffffff 0%, #f7faff 100%);
            box-shadow: inset 0 1px 0 rgba(255,255,255,0.8);
        }

        /* ===== DATA TABLE ===== */
        .data-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
            font-size: 14px;
        }

        .data-table thead {
            background: linear-gradient(135deg, #f9f9f9 0%, #f0f0f0 100%);
            border-bottom: 3px solid #eee;
        }

        .data-table th {
            padding: 15px;
            text-align: left;
            font-weight: 700;
            color: #2c3e50;
            font-size: 12px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .data-table td {
            padding: 14px 15px;
            border-bottom: 1px solid #eee;
            color: #555;
        }

        .data-table tbody tr {
            transition: all 0.2s ease;
        }

        .data-table tbody tr:hover {
            background: #f9f9f9;
            transform: scale(1.01);
        }

        .progress-bar {
            background: #eee;
            height: 8px;
            border-radius: 4px;
            overflow: hidden;
            margin-top: 5px;
        }

        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #00a65a, #43e97b);
            border-radius: 4px;
        }

        /* ===== BADGE ===== */
        .badge {
            display: inline-block;
            padding: 6px 14px;
            background: #e3f2fd;
            color: #1976d2;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }

        /* ===== EMPTY STATE ===== */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #999;
        }

        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.2;
        }

        .empty-state p {
            font-size: 16px;
            margin: 10px 0;
        }

        /* ===== RESPONSIVE ===== */
        @media (max-width: 768px) {
            .tab-content-wrapper { padding: 15px; }
            .selection-grid { grid-template-columns: 1fr; }
            .stats-grid { grid-template-columns: 1fr 1fr; }
            .chart-canvas { height: 250px; }
            .analytics-header h1 { font-size: 20px; }
            .tab-btn { padding: 15px 15px; font-size: 13px; }
        }

        .no-data { color: #999; font-style: italic; text-align: center; padding: 20px; }
    </style>
</head>

<body>

<div id="overlay" class="overlay" onclick="toggleSidebar()"></div>

<script src="sidebar.js"></script>

<div class="main-content">
    <!-- HEADER (UNCHANGED) -->
    <div class="analytics-header">
        <div style="display: flex; align-items: center;">
            <button class="menu-toggle-btn" onclick="toggleSidebar()">
                <i class="fas fa-bars"></i>
            </button>
            <h1><i class="fas fa-chart-bar"></i> Interactive Analytics</h1>
        </div>
    </div>

    <!-- TAB NAVIGATION -->
    <div class="tab-navigation">
        <button class="tab-btn active" onclick="switchTab('dashboard')">
            <i class="fas fa-home"></i> Dashboard
        </button>
        <button class="tab-btn" onclick="switchTab('fields')">
            <i class="fas fa-layer-group"></i> Field Analytics
        </button>
        <button class="tab-btn" onclick="switchTab('machines')">
            <i class="fas fa-cogs"></i> Machine Analytics
        </button>
        <button class="tab-btn" onclick="switchTab('comparison')">
            <i class="fas fa-balance-scale"></i> Comparison
        </button>
        <button class="tab-btn" onclick="switchTab('timeline')">
            <i class="fas fa-calendar-alt"></i> Timeline
        </button>
    </div>

    <!-- TAB CONTENT -->
    <div class="tab-content-wrapper">

        <!-- ===== DASHBOARD TAB ===== -->
        <div class="tab-content active" id="dashboard">
            <div class="section-title">
                <i class="fas fa-gauge-high"></i> Overall Dashboard
            </div>

            <div class="stats-grid">
                <div class="stat-card blue">
                    <div class="stat-card-label">Total Production</div>
                    <div class="stat-card-value" id="dashTotalRecords">-</div>
                    <div class="stat-card-detail">Produced quantity sum</div>
                </div>
                <div class="stat-card pink">
                    <div class="stat-card-label">Active Employees</div>
                    <div class="stat-card-value" id="dashEmployees">-</div>
                    <div class="stat-card-detail">Team members</div>
                </div>
                <div class="stat-card cyan">
                    <div class="stat-card-label">Total Hours</div>
                    <div class="stat-card-value" id="dashDataPoints">-</div>
                    <div class="stat-card-detail">Duration sum</div>
                </div>
                <div class="stat-card green">
                    <div class="stat-card-label">Avg Daily Prod.</div>
                    <div class="stat-card-value" id="dashAvgDaily">-</div>
                    <div class="stat-card-detail">Units/day (30 days)</div>
                </div>
            </div>

            <div class="chart-section">
                <div class="chart-title"><i class="fas fa-chart-line"></i> 30-Day Production Quantity Trend</div>
                <div class="chart-canvas">
                    <canvas id="dashboardChart"></canvas>
                </div>
            </div>

            <div class="chart-section">
                <div class="chart-title"><i class="fas fa-user-tie"></i> Top Employees</div>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Employee ID</th>
                            <th>Name</th>
                            <th>Production Qty</th>
                            <th>Performance</th>
                        </tr>
                    </thead>
                    <tbody id="topEmployeesBody">
                        <tr><td colspan="4" class="no-data">Loading...</td></tr>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- ===== FIELD ANALYTICS TAB ===== -->
        <div class="tab-content" id="fields">
            <div class="section-title">
                <i class="fas fa-layer-group"></i> Field Analysis
            </div>

            <div style="margin-bottom: 20px;">
                <label style="display: block; margin-bottom: 10px; font-weight: 600; color: #555;">Select a field to analyze:</label>
                <select id="fieldSelect" onchange="analyzeField()" style="width: 100%; padding: 12px; border: 2px solid #eee; border-radius: 8px; font-size: 14px;">
                    <option value="">-- Choose a field --</option>
                </select>
            </div>

            <div id="fieldAnalyticsArea" style="display: none;">
                <div class="stats-grid">
                    <div class="stat-card blue">
                        <div class="stat-card-label">Unique Values</div>
                        <div class="stat-card-value" id="fieldUnique">0</div>
                    </div>
                    <div class="stat-card pink">
                        <div class="stat-card-label">Total Records</div>
                        <div class="stat-card-value" id="fieldTotal">0</div>
                    </div>
                    <div class="stat-card cyan">
                        <div class="stat-card-label">Most Frequent</div>
                        <div class="stat-card-value" id="fieldMax">0</div>
                    </div>
                    <div class="stat-card green">
                        <div class="stat-card-label">Coverage</div>
                        <div class="stat-card-value" id="fieldCoverage">0%</div>
                    </div>
                </div>

                <div class="chart-section">
                    <div class="chart-title"><i class="fas fa-pie-chart"></i> Distribution</div>
                    <div class="chart-canvas">
                        <canvas id="fieldChart"></canvas>
                    </div>
                </div>

                <div class="chart-section">
                    <div class="chart-title"><i class="fas fa-list"></i> Top Values</div>
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Value</th>
                                <th>Count</th>
                                <th>Percentage</th>
                                <th>Visualization</th>
                            </tr>
                        </thead>
                        <tbody id="fieldDataBody">
                            <tr><td colspan="4" class="no-data">No data</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <div id="fieldEmptyState" style="text-align: center; padding: 80px 20px; color: #999;">
                <i class="fas fa-inbox" style="font-size: 64px; opacity: 0.2; display: block; margin-bottom: 20px;"></i>
                <p>Select a field to view its analysis</p>
            </div>
        </div>

        <!-- ===== MACHINE ANALYTICS TAB ===== -->
        <div class="tab-content" id="machines">
            <div class="section-title">
                <i class="fas fa-cogs"></i> Machine Analytics
            </div>

            <div style="margin-bottom: 20px;">
                <label style="display: block; margin-bottom: 10px; font-weight: 600; color: #555;">Select machine field:</label>
                <select id="machineFieldSelect" onchange="analyzeMachines()" style="width: 100%; padding: 12px; border: 2px solid #eee; border-radius: 8px; font-size: 14px;">
                    <option value="">-- Choose machine field --</option>
                </select>
            </div>

            <div id="machineListArea" style="display: none;">
                <div class="selection-grid" id="machineSelectionGrid">
                    <div class="selection-card" onclick="this.classList.toggle('active');">
                        <div class="selection-card-icon"><i class="fas fa-spinner"></i></div>
                        <h3>Loading...</h3>
                    </div>
                </div>

                <div id="machineDetailArea" style="display: none;">
                    <div class="chart-section">
                            <div class="chart-title"><i class="fas fa-chart-bar"></i> <span id="machineDetailTitle">Machine Details</span></div>
                            <div class="stats-grid">
                                <div class="stat-card blue">
                                    <div class="stat-card-label">Total Production</div>
                                    <div class="stat-card-value" id="machOps">0</div>
                                </div>
                                <div class="stat-card pink">
                                    <div class="stat-card-label">Employees</div>
                                    <div class="stat-card-value" id="machEmps">0</div>
                                </div>
                                <div class="stat-card cyan">
                                    <div class="stat-card-label">Tools</div>
                                    <div class="stat-card-value" id="machTools">0</div>
                                </div>
                                <div class="stat-card green">
                                    <div class="stat-card-label">Total Hours</div>
                                    <div class="stat-card-value" id="machAvg">0</div>
                                </div>
                            </div>
                        </div>

                    <div class="chart-section">
                        <div class="chart-header">
                            <div class="chart-title"><i class="fas fa-chart-area"></i> Production Timeline</div>
                            <div class="chart-controls">
                                <button class="chart-btn active" onclick="switchMachineChart('bar')">Bar</button>
                                <button class="chart-btn" onclick="switchMachineChart('line')">Line</button>
                                <button class="chart-btn" onclick="switchMachineChart('pie')">Pie</button>
                            </div>
                        </div>
                        <div class="chart-canvas">
                            <canvas id="machineChart"></canvas>
                        </div>
                    </div>

                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                        <div class="chart-section">
                            <div class="chart-title"><i class="fas fa-hammer"></i> Tools Used</div>
                            <table class="data-table">
                                <thead>
                                    <tr><th>Tool</th><th>Production Qty</th><th>%</th></tr>
                                </thead>
                                <tbody id="machineToolsBody">
                                    <tr><td colspan="3" class="no-data">No data</td></tr>
                                </tbody>
                            </table>
                        </div>

                        <div class="chart-section">
                            <div class="chart-title"><i class="fas fa-users"></i> Employees</div>
                            <table class="data-table">
                                <thead>
                                    <tr><th>Employee</th><th>Production Qty</th><th>%</th></tr>
                                </thead>
                                <tbody id="machineEmployeesBody">
                                    <tr><td colspan="3" class="no-data">No data</td></tr>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <div class="chart-section">
                        <div class="chart-title"><i class="fas fa-calendar"></i> Daily Breakdown</div>
                        <table class="data-table">
                            <thead>
                                <tr>
                                    <th>Date</th><th>Day</th><th>Production</th><th>Hours</th><th>Employees</th><th>Avg/Emp</th>
                                </tr>
                            </thead>
                            <tbody id="machineDailyBody">
                                <tr><td colspan="6" class="no-data">No data</td></tr>
                            </tbody>
                        </table>
                    </div>

                    <button onclick="closeMachineDetail()" style="width: 100%; padding: 14px; background: #6c757d; color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer; margin-top: 20px;">
                        <i class="fas fa-arrow-left"></i> Back to Machines
                    </button>
                </div>
            </div>

            <div id="machineEmptyState" style="text-align: center; padding: 80px 20px; color: #999;">
                <i class="fas fa-inbox" style="font-size: 64px; opacity: 0.2; display: block; margin-bottom: 20px;"></i>
                <p>Select a machine field to view machines</p>
            </div>
        </div>

        <!-- ===== COMPARISON TAB ===== -->
        <div class="tab-content" id="comparison">
            <div class="section-title">
                <i class="fas fa-balance-scale"></i> Field Comparison
            </div>

            <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 20px; margin-bottom: 20px;">
                <div>
                    <label style="display: block; margin-bottom: 10px; font-weight: 600; color: #555;">Field 1:</label>
                    <select id="compareField1" style="width: 100%; padding: 12px; border: 2px solid #eee; border-radius: 8px;">
                        <option value="">-- Choose field --</option>
                    </select>
                </div>
                <div>
                    <label style="display: block; margin-bottom: 10px; font-weight: 600; color: #555;">Field 2:</label>
                    <select id="compareField2" style="width: 100%; padding: 12px; border: 2px solid #eee; border-radius: 8px;">
                        <option value="">-- Choose field --</option>
                    </select>
                </div>
                <div>
                    <label style="display: block; margin-bottom: 10px; font-weight: 600; color: #555;">Compare Using:</label>
                    <select id="compareMetric" style="width: 100%; padding: 12px; border: 2px solid #eee; border-radius: 8px;">
                        <option value="total_production_qty">Total Production Qty</option>
                        <option value="total_hours">Total Hours</option>
                        <option value="count">Record Count</option>
                    </select>
                </div>
            </div>

            <div style="display: flex; gap: 10px; margin-bottom: 20px;">
                <button onclick="compareFields()" style="flex: 1; padding: 14px; background: #00a65a; color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer;">
                    <i class="fas fa-chart-bar"></i> Compare Fields
                </button>
                <button onclick="swapComparisonFields()" style="padding: 14px 18px; background: #34495e; color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer;">
                    <i class="fas fa-right-left"></i> Swap
                </button>
            </div>

            <div id="comparisonArea" style="display: none;">
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 20px;">
                    <div class="stat-card blue">
                        <div class="stat-card-label">Field 1 Insights</div>
                        <div class="stat-card-value" id="comp1Insights">-</div>
                        <div style="font-size: 12px; margin-top: 10px;" id="comp1Stats">-</div>
                    </div>
                    <div class="stat-card pink">
                        <div class="stat-card-label">Field 2 Insights</div>
                        <div class="stat-card-value" id="comp2Insights">-</div>
                        <div style="font-size: 12px; margin-top: 10px;" id="comp2Stats">-</div>
                    </div>
                </div>

                <div class="chart-section">
                    <div class="chart-header">
                        <div class="chart-title"><i class="fas fa-chart-area"></i> <span id="comparisonChartTitle">Comparison Chart</span></div>
                        <div class="chart-controls">
                            <button class="chart-btn active" onclick="switchCompChart('bar')">Bar</button>
                            <button class="chart-btn" onclick="switchCompChart('radar')">Radar</button>
                            <button class="chart-btn" onclick="switchCompChart('pie')">Pie</button>
                        </div>
                    </div>
                    <div class="chart-canvas">
                        <canvas id="comparisonChart"></canvas>
                    </div>
                </div>

                <div class="chart-section">
                    <div class="chart-title"><i class="fas fa-table"></i> Statistical Comparison</div>
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Metric</th>
                                <th>Field 1</th>
                                <th>Field 2</th>
                                <th>Difference</th>
                                <th>Winner</th>
                            </tr>
                        </thead>
                        <tbody id="comparisonStatsBody">
                            <tr><td colspan="5" class="no-data">No comparison data</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <div id="comparisonEmptyState" style="text-align: center; padding: 80px 20px; color: #999;">
                <i class="fas fa-inbox" style="font-size: 64px; opacity: 0.2; display: block; margin-bottom: 20px;"></i>
                <p>Select two fields and click Compare to see results</p>
            </div>
        </div>

        <!-- ===== TIMELINE TAB ===== -->
        <div class="tab-content" id="timeline">
            <div class="section-title">
                <i class="fas fa-calendar-alt"></i> Daily Analytics - All Days
            </div>

            <button onclick="loadAllDaysTimeline()" style="width: 100%; padding: 14px; background: #00a65a; color: white; border: none; border-radius: 8px; font-weight: 600; cursor: pointer; margin-bottom: 20px;">
                <i class="fas fa-sync-alt"></i> Load All Days Analytics
            </button>

            <div id="timelineArea" style="display: none;">
                <div class="stats-grid">
                    <div class="stat-card blue">
                        <div class="stat-card-label">Total Days</div>
                        <div class="stat-card-value" id="timelineDays">0</div>
                    </div>
                    <div class="stat-card pink">
                        <div class="stat-card-label">Total Operations</div>
                        <div class="stat-card-value" id="timelineOps">0</div>
                    </div>
                    <div class="stat-card cyan">
                        <div class="stat-card-label">Total Employees</div>
                        <div class="stat-card-value" id="timelineEmps">0</div>
                    </div>
                    <div class="stat-card green">
                        <div class="stat-card-label">Avg Daily Ops</div>
                        <div class="stat-card-value" id="timelineAvg">0</div>
                    </div>
                </div>

                <div class="chart-section">
                    <div class="chart-title"><i class="fas fa-chart-line"></i> Operations Trend</div>
                    <div class="chart-canvas">
                        <canvas id="timelineChart"></canvas>
                    </div>
                </div>

                <div id="dailyDetailsContainer" style="margin-top: 30px;">
                    <!-- Daily cards will be inserted here -->
                </div>
            </div>

            <div id="timelineEmptyState" style="text-align: center; padding: 80px 20px; color: #999;">
                <i class="fas fa-inbox" style="font-size: 64px; opacity: 0.2; display: block; margin-bottom: 20px;"></i>
                <p>Click "Load All Days Analytics" to view detailed breakdown for each day</p>
            </div>
        </div>

    </div>

</div>

<script>
    // ===== CHART INSTANCES =====
    let dashboardChart = null;
    let fieldChart = null;
    let machineChart = null;
    let comparisonChart = null;
    let timelineChart = null;

    // ===== INTERACTIVE BAR THEME =====
    const barColorPairs = [
        ['#22d3ee', '#0ea5e9'],
        ['#34d399', '#22c55e'],
        ['#60a5fa', '#3b82f6'],
        ['#f472b6', '#ec4899'],
        ['#f59e0b', '#ef4444'],
        ['#a78bfa', '#7c3aed'],
        ['#fb7185', '#e11d48'],
        ['#2dd4bf', '#14b8a6']
    ];

    function hexToRgba(hex, alpha = 1) {
        if (typeof hex !== 'string' || !hex.startsWith('#')) return `rgba(59,130,246,${alpha})`;
        const sanitized = hex.replace('#', '');
        const full = sanitized.length === 3
            ? sanitized.split('').map(ch => ch + ch).join('')
            : sanitized;
        const r = parseInt(full.substring(0, 2), 16);
        const g = parseInt(full.substring(2, 4), 16);
        const b = parseInt(full.substring(4, 6), 16);
        return `rgba(${r},${g},${b},${alpha})`;
    }

    function getBarColorPair(datasetIndex, valueIndex) {
        return barColorPairs[(datasetIndex + valueIndex) % barColorPairs.length];
    }

    const interactiveBarThemePlugin = {
        id: 'interactiveBarTheme',
        beforeUpdate(chart) {
            if (chart.config.type !== 'bar') return;
            const datasets = chart.data?.datasets || [];

            datasets.forEach((dataset, datasetIndex) => {
                const values = Array.isArray(dataset.data) ? dataset.data : [];
                const backgrounds = [];
                const borders = [];
                const hoverBackgrounds = [];

                values.forEach((_, valueIndex) => {
                    const [base, accent] = getBarColorPair(datasetIndex, valueIndex);
                    backgrounds.push(hexToRgba(base, 0.78));
                    borders.push(accent);
                    hoverBackgrounds.push(hexToRgba(accent, 0.95));
                });

                dataset.backgroundColor = backgrounds;
                dataset.borderColor = borders;
                dataset.hoverBackgroundColor = hoverBackgrounds;
                dataset.borderWidth = 1.5;
                dataset.borderRadius = 10;
                dataset.borderSkipped = false;
                dataset.hoverBorderWidth = 2;
                dataset.maxBarThickness = dataset.maxBarThickness || 34;
                dataset.barPercentage = dataset.barPercentage ?? 0.72;
                dataset.categoryPercentage = dataset.categoryPercentage ?? 0.66;
            });
        },
        beforeDatasetDraw(chart, args) {
            const meta = chart.getDatasetMeta(args.index);
            if (meta.type !== 'bar') return;
            const ds = chart.data.datasets[args.index] || {};
            const color = Array.isArray(ds.borderColor) ? ds.borderColor[0] : ds.borderColor;
            const glow = typeof color === 'string' ? hexToRgba(color, 0.45) : 'rgba(14,165,233,0.45)';

            chart.ctx.save();
            chart.ctx.shadowColor = glow;
            chart.ctx.shadowBlur = 12;
            chart.ctx.shadowOffsetX = 0;
            chart.ctx.shadowOffsetY = 2;
        },
        afterDatasetDraw(chart, args) {
            const meta = chart.getDatasetMeta(args.index);
            if (meta.type !== 'bar') return;
            chart.ctx.restore();
        }
    };

    Chart.register(interactiveBarThemePlugin);
    Chart.defaults.animation.duration = 900;
    Chart.defaults.animation.easing = 'easeOutQuart';
    Chart.defaults.plugins.legend.labels.usePointStyle = true;
    Chart.defaults.plugins.legend.labels.boxWidth = 10;
    Chart.defaults.plugins.legend.labels.padding = 14;

    // ===== TOGGLE SIDEBAR =====
    function toggleSidebar() {
        const sidebar = document.querySelector('.sidebar');
        const mainContent = document.querySelector('.main-content');
        
        if (sidebar.classList.contains('active')) {
            sidebar.classList.remove('active');
            mainContent.classList.remove('shifted');
        } else {
            sidebar.classList.add('active');
            mainContent.classList.add('shifted');
        }
    }

    // ===== TAB SWITCHING =====
    function switchTab(tabName) {
        // Hide all tabs
        document.querySelectorAll('.tab-content').forEach(tab => tab.classList.remove('active'));
        document.querySelectorAll('.tab-btn').forEach(btn => btn.classList.remove('active'));

        // Show selected tab
        document.getElementById(tabName).classList.add('active');
        event.target.closest('.tab-btn').classList.add('active');

        // Load data if first time
        if (tabName === 'dashboard' && !dashboardChart) {
            loadDashboard();
        }
    }

    // ===== LOAD DROPDOWN FIELDS =====
    async function loadFields() {
        try {
            const response = await fetch('api.php?action=get_analytics_fields');
            const result = await response.json();

            if (result.success) {
                ['fieldSelect', 'machineFieldSelect', 'compareField1', 'compareField2'].forEach(id => {
                    const select = document.getElementById(id);
                    select.innerHTML = '<option value="">-- Choose --</option>';
                    result.fields.forEach(field => {
                        const option = document.createElement('option');
                        option.value = field.field_name;
                        option.textContent = field.field_label;
                        select.appendChild(option);
                    });
                });
            }
        } catch (error) {
            console.error('Error loading fields:', error);
        }
    }

    // ===== LOAD DASHBOARD =====
    async function loadDashboard() {
        try {
            const response = await fetch('api.php?action=get_analytics_dashboard');
            const result = await response.json();

            if (result.success) {
                const totalProduction = Number(result.total_production_qty ?? result.total_records ?? 0);
                const totalHours = Number(result.total_hours ?? 0);
                const days = Array.isArray(result.date_stats) ? result.date_stats.length : 0;
                const avgDailyProduction = totalProduction / (days || 1);

                document.getElementById('dashTotalRecords').textContent = totalProduction.toLocaleString(undefined, { maximumFractionDigits: 2 });
                document.getElementById('dashEmployees').textContent = result.employee_count;
                document.getElementById('dashDataPoints').textContent = totalHours.toLocaleString(undefined, { maximumFractionDigits: 2 });
                document.getElementById('dashAvgDaily').textContent = avgDailyProduction.toLocaleString(undefined, { maximumFractionDigits: 2 });

                buildDashboardChart(result.date_stats);
                buildTopEmployeesTable(result.top_employees);
            }
        } catch (error) {
            console.error('Error loading dashboard:', error);
        }
    }

    // ===== BUILD DASHBOARD CHART =====
    function buildDashboardChart(stats) {
        const ctx = document.getElementById('dashboardChart');
        if (!ctx) return;

        if (dashboardChart) dashboardChart.destroy();

        dashboardChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: stats.map(s => s.log_date),
                datasets: [{
                    label: 'Production Qty/Day',
                    data: stats.map(s => Number(s.total_quantity ?? s.count ?? 0)),
                    backgroundColor: 'rgba(0, 166, 90, 0.7)',
                    borderColor: '#00a65a',
                    borderWidth: 2,
                    borderRadius: 6
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: { legend: { display: true } },
                scales: { y: { beginAtZero: true } }
            }
        });
    }

    // ===== BUILD TOP EMPLOYEES TABLE =====
    function buildTopEmployeesTable(employees) {
        const tbody = document.getElementById('topEmployeesBody');
        if (!employees || employees.length === 0) {
            tbody.innerHTML = '<tr><td colspan="4" class="no-data">No data</td></tr>';
            return;
        }

        const ranked = employees.slice(0, 5).map(emp => ({
            ...emp,
            metric: Number(emp.total_qty ?? emp.count ?? 0)
        }));
        const topMetric = Number(ranked[0]?.metric ?? 1) || 1;

        tbody.innerHTML = ranked.map(emp => `
            <tr>
                <td><span class="badge">${emp.emp_id}</span></td>
                <td><strong>${emp.operator_name}</strong></td>
                <td>${emp.metric.toLocaleString(undefined, { maximumFractionDigits: 2 })}</td>
                <td>
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: ${(emp.metric / topMetric * 100)}%"></div>
                    </div>
                </td>
            </tr>
        `).join('');
    }

    // ===== ANALYZE FIELD =====
    async function analyzeField() {
        const field = document.getElementById('fieldSelect').value;
        if (!field) {
            document.getElementById('fieldAnalyticsArea').style.display = 'none';
            document.getElementById('fieldEmptyState').style.display = 'block';
            return;
        }

        try {
            const response = await fetch(`api.php?action=get_field_statistics&field_name=${field}`);
            const result = await response.json();

            if (result.success) {
                const data = result.data;
                const total = result.total_records;

                document.getElementById('fieldUnique').textContent = data.length;
                document.getElementById('fieldTotal').textContent = total;
                document.getElementById('fieldMax').textContent = Math.max(...data.map(d => d.count));
                document.getElementById('fieldCoverage').textContent = '100%';

                buildFieldChart(data);
                buildFieldTable(data, total);

                document.getElementById('fieldAnalyticsArea').style.display = 'block';
                document.getElementById('fieldEmptyState').style.display = 'none';
            }
        } catch (error) {
            console.error('Error analyzing field:', error);
        }
    }

    // ===== BUILD FIELD CHART =====
    function buildFieldChart(data) {
        const ctx = document.getElementById('fieldChart');
        if (!ctx) return;

        if (fieldChart) fieldChart.destroy();

        const top10 = data.slice(0, 10);
        const colors = ['#667eea', '#764ba2', '#f093fb', '#f5576c', '#4facfe', '#00f2fe', '#43e97b', '#38f9d7', '#fa709a', '#fee140'];

        fieldChart = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: top10.map(d => String(d.value).substring(0, 15)),
                datasets: [{
                    data: top10.map(d => d.count),
                    backgroundColor: colors,
                    borderColor: 'white',
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: { legend: { position: 'right' } }
            }
        });
    }

    // ===== BUILD FIELD TABLE =====
    function buildFieldTable(data, total) {
        const tbody = document.getElementById('fieldDataBody');
        tbody.innerHTML = data.slice(0, 10).map(item => {
            const pct = ((item.count / total) * 100).toFixed(1);
            return `
                <tr>
                    <td><strong>${item.value || 'N/A'}</strong></td>
                    <td>${item.count}</td>
                    <td>${pct}%</td>
                    <td>
                        <div class="progress-bar">
                            <div class="progress-fill" style="width: ${pct}%"></div>
                        </div>
                    </td>
                </tr>
            `;
        }).join('');
    }

    // ===== ANALYZE MACHINES =====
    async function analyzeMachines() {
        const field = document.getElementById('machineFieldSelect').value;
        if (!field) {
            document.getElementById('machineListArea').style.display = 'none';
            document.getElementById('machineEmptyState').style.display = 'block';
            return;
        }

        try {
            const response = await fetch(`api.php?action=get_machine_tool_list&machine_field=${field}`);
            const result = await response.json();

            if (result.success) {
                const grid = document.getElementById('machineSelectionGrid');
                grid.innerHTML = result.machines.map(mach => `
                    <div class="selection-card" onclick="selectMachine('${mach.machine}', '${field}', event)">
                        <div class="selection-card-icon"><i class="fas fa-cogs"></i></div>
                        <h3>${mach.machine}</h3>
                        <p><strong>${Number(mach.total_production_qty ?? mach.total_operations ?? 0).toLocaleString(undefined, { maximumFractionDigits: 2 })}</strong> qty<br><strong>${Number(mach.total_hours ?? 0).toLocaleString(undefined, { maximumFractionDigits: 2 })}</strong> hours</p>
                    </div>
                `).join('');

                document.getElementById('machineListArea').style.display = 'block';
                document.getElementById('machineEmptyState').style.display = 'none';
            }
        } catch (error) {
            console.error('Error analyzing machines:', error);
        }
    }

    // ===== SELECT MACHINE =====
    async function selectMachine(machineName, field, event) {
        event.currentTarget.classList.add('active');

        try {
            const response = await fetch(`api.php?action=get_machine_detail_analysis&machine_field=${field}&machine_name=${encodeURIComponent(machineName)}`);
            const result = await response.json();

            if (result.success) {
                document.getElementById('machineDetailTitle').textContent = `${machineName} - Detailed Analysis`;
                const totalProduction = Number(result.summary.total_production_qty ?? result.summary.total_operations ?? 0);
                const totalHours = Number(result.summary.total_hours ?? 0);
                const totalForShare = totalProduction > 0 ? totalProduction : Number(result.summary.total_operations ?? 0);

                document.getElementById('machOps').textContent = totalProduction.toLocaleString(undefined, { maximumFractionDigits: 2 });
                document.getElementById('machEmps').textContent = result.summary.employee_count;
                document.getElementById('machTools').textContent = result.summary.tools_count;
                document.getElementById('machAvg').textContent = totalHours.toLocaleString(undefined, { maximumFractionDigits: 2 });

                buildMachineChart(result.daily_data, 'bar');
                buildToolsTable(result.tools, totalForShare);
                buildEmployeesTable(result.employees, totalForShare);
                buildDailyTable(result.daily_data);

                window.machineData = {
                    daily_data: result.daily_data,
                    currentType: 'bar'
                };

                document.getElementById('machineDetailArea').style.display = 'block';
                document.getElementById('machineDetailArea').scrollIntoView({ behavior: 'smooth' });
            }
        } catch (error) {
            console.error('Error:', error);
        }
    }

    // ===== BUILD MACHINE CHART =====
    function buildMachineChart(data, type) {
        const ctx = document.getElementById('machineChart');
        if (!ctx) return;

        if (machineChart && typeof machineChart.destroy === 'function') {
            machineChart.destroy();
        }

        const labels = data.map(d => d.log_date);
        const production = data.map(d => Number(d.total_production_qty ?? d.total_operations ?? 0));
        const hours = data.map(d => Number(d.total_hours ?? 0));
        const emps = data.map(d => d.employee_count);

        if (type === 'bar') {
            machineChart = new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [
                        {
                            label: 'Production Qty',
                            data: production,
                            backgroundColor: 'rgba(0, 166, 90, 0.7)',
                            borderColor: '#00a65a',
                            borderWidth: 1
                        },
                        {
                            label: 'Hours',
                            data: hours,
                            backgroundColor: 'rgba(240, 147, 251, 0.7)',
                            borderColor: '#f093fb',
                            borderWidth: 1
                        },
                        {
                            label: 'Employees',
                            data: emps,
                            backgroundColor: 'rgba(79, 172, 254, 0.7)',
                            borderColor: '#4facfe',
                            borderWidth: 1
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: { y: { beginAtZero: true } }
                }
            });
        } else if (type === 'line') {
            machineChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: labels,
                    datasets: [
                        {
                            label: 'Production Qty',
                            data: production,
                            borderColor: '#00a65a',
                            borderWidth: 3,
                            tension: 0.4,
                            fill: false
                        },
                        {
                            label: 'Hours',
                            data: hours,
                            borderColor: '#f093fb',
                            borderWidth: 3,
                            tension: 0.4,
                            fill: false
                        },
                        {
                            label: 'Employees',
                            data: emps,
                            borderColor: '#4facfe',
                            borderWidth: 3,
                            tension: 0.4,
                            fill: false
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: { y: { beginAtZero: true } }
                }
            });
        } else if (type === 'pie') {
            machineChart = new Chart(ctx, {
                type: 'doughnut',
                data: {
                    labels: labels,
                    datasets: [{
                        data: production,
                        backgroundColor: ['#667eea', '#764ba2', '#f093fb', '#f5576c', '#4facfe', '#00f2fe', '#43e97b', '#38f9d7', '#fa709a', '#fee140']
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: { legend: { position: 'right' } }
                }
            });
        }
    }

    // ===== SWITCH MACHINE CHART =====
    function switchMachineChart(type) {
        if (!window.machineData) return;
        document.querySelectorAll('[onclick*="switchMachineChart"]').forEach(btn => btn.classList.remove('active'));
        event.target.classList.add('active');
        buildMachineChart(window.machineData.daily_data, type);
    }

    // ===== BUILD TABLES =====
    function buildToolsTable(tools, total) {
        const tbody = document.getElementById('machineToolsBody');
        if (!tools || tools.length === 0) {
            tbody.innerHTML = '<tr><td colspan="3">No data</td></tr>';
            return;
        }
        tbody.innerHTML = tools.map(tool => {
            const metric = Number(tool.total_production_qty ?? tool.count ?? 0);
            const pct = ((metric / (total || 1)) * 100).toFixed(1);
            return `<tr><td>${tool.tool_name || 'N/A'}</td><td>${metric.toLocaleString(undefined, { maximumFractionDigits: 2 })}</td><td>${pct}%</td></tr>`;
        }).join('');
    }

    function buildEmployeesTable(emps, total) {
        const tbody = document.getElementById('machineEmployeesBody');
        if (!emps || emps.length === 0) {
            tbody.innerHTML = '<tr><td colspan="3">No data</td></tr>';
            return;
        }
        tbody.innerHTML = emps.map(emp => {
            const metric = Number(emp.total_production_qty ?? emp.count ?? 0);
            const pct = ((metric / (total || 1)) * 100).toFixed(1);
            return `<tr><td>${emp.operator_name}</td><td>${metric.toLocaleString(undefined, { maximumFractionDigits: 2 })}</td><td>${pct}%</td></tr>`;
        }).join('');
    }

    function buildDailyTable(data) {
        const tbody = document.getElementById('machineDailyBody');
        if (!data || data.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6">No data</td></tr>';
            return;
        }
        const days = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
        tbody.innerHTML = data.map(row => {
            const date = new Date(row.log_date);
            const day = days[date.getDay()];
            const dailyProduction = Number(row.total_production_qty ?? row.total_operations ?? 0);
            const dailyHours = Number(row.total_hours ?? 0);
            const avg = (dailyProduction / (Number(row.employee_count) || 1)).toFixed(1);
            return `<tr>
                <td>${row.log_date}</td>
                <td>${day}</td>
                <td>${dailyProduction.toLocaleString(undefined, { maximumFractionDigits: 2 })}</td>
                <td>${dailyHours.toLocaleString(undefined, { maximumFractionDigits: 2 })}</td>
                <td>${row.employee_count}</td>
                <td>${avg}</td>
            </tr>`;
        }).join('');
    }

    function closeMachineDetail() {
        document.getElementById('machineDetailArea').style.display = 'none';
        if (machineChart && typeof machineChart.destroy === 'function') {
            machineChart.destroy();
            machineChart = null;
        }
    }

    function getComparisonMetricConfig() {
        const metric = document.getElementById('compareMetric')?.value || 'total_production_qty';
        if (metric === 'count') {
            return { key: 'count', totalKey: 'total_records', label: 'Record Count', unit: 'records', decimals: 0 };
        }
        if (metric === 'total_hours') {
            return { key: 'total_hours', totalKey: 'total_hours', label: 'Total Hours', unit: 'hours', decimals: 2 };
        }
        return { key: 'total_production_qty', totalKey: 'total_production_qty', label: 'Total Production Qty', unit: 'qty', decimals: 2 };
    }

    function formatComparisonValue(value, metricCfg) {
        return Number(value || 0).toLocaleString(undefined, {
            minimumFractionDigits: 0,
            maximumFractionDigits: metricCfg.decimals
        });
    }

    function getFieldCompareSummary(result, metricCfg) {
        const rows = Array.isArray(result.data) ? result.data : [];
        const metricValues = rows.map(row => Number(row[metricCfg.key] ?? row.count ?? 0));
        const metricTotalFromRows = metricValues.reduce((sum, value) => sum + value, 0);
        const metricTotal = Number(result[metricCfg.totalKey] ?? metricTotalFromRows);
        const totalRecords = Number(result.total_records ?? 0);
        const uniqueValues = rows.length;

        let topValueName = 'N/A';
        let topMetric = 0;
        if (rows.length > 0) {
            const sorted = [...rows].sort((a, b) => Number(b[metricCfg.key] ?? b.count ?? 0) - Number(a[metricCfg.key] ?? a.count ?? 0));
            topValueName = String(sorted[0]?.value ?? 'N/A');
            topMetric = Number(sorted[0]?.[metricCfg.key] ?? sorted[0]?.count ?? 0);
        }

        return {
            uniqueValues,
            totalRecords,
            metricTotal,
            topMetric,
            topValueName,
            avgMetric: uniqueValues > 0 ? metricTotal / uniqueValues : 0
        };
    }

    function swapComparisonFields() {
        const f1 = document.getElementById('compareField1');
        const f2 = document.getElementById('compareField2');
        const old = f1.value;
        f1.value = f2.value;
        f2.value = old;
        if (f1.value && f2.value && f1.value !== f2.value) {
            compareFields();
        }
    }

    // ===== COMPARE FIELDS =====
    async function compareFields() {
        const f1 = document.getElementById('compareField1').value;
        const f2 = document.getElementById('compareField2').value;

        if (!f1 || !f2 || f1 === f2) {
            alert('Select two different fields');
            return;
        }

        const metricCfg = getComparisonMetricConfig();

        try {
            const [r1, r2] = await Promise.all([
                fetch(`api.php?action=get_field_statistics&field_name=${f1}`).then(r => r.json()),
                fetch(`api.php?action=get_field_statistics&field_name=${f2}`).then(r => r.json())
            ]);

            if (r1.success && r2.success) {
                const label1 = document.getElementById('compareField1').options[document.getElementById('compareField1').selectedIndex].text;
                const label2 = document.getElementById('compareField2').options[document.getElementById('compareField2').selectedIndex].text;
                const s1 = getFieldCompareSummary(r1, metricCfg);
                const s2 = getFieldCompareSummary(r2, metricCfg);

                window.comparisonData = {
                    r1,
                    r2,
                    s1,
                    s2,
                    label1,
                    label2,
                    metricCfg
                };

                document.getElementById('comp1Insights').textContent = formatComparisonValue(s1.metricTotal, metricCfg);
                document.getElementById('comp1Stats').innerHTML = `<strong>${s1.uniqueValues}</strong> unique values<br><strong>${s1.topValueName}</strong> top (${formatComparisonValue(s1.topMetric, metricCfg)} ${metricCfg.unit})`;
                document.getElementById('comp2Insights').textContent = formatComparisonValue(s2.metricTotal, metricCfg);
                document.getElementById('comp2Stats').innerHTML = `<strong>${s2.uniqueValues}</strong> unique values<br><strong>${s2.topValueName}</strong> top (${formatComparisonValue(s2.topMetric, metricCfg)} ${metricCfg.unit})`;
                document.getElementById('comparisonChartTitle').textContent = `Comparison by ${metricCfg.label}`;
                document.querySelectorAll('[onclick*="switchCompChart"]').forEach(btn => btn.classList.remove('active'));
                const defaultChartBtn = document.querySelector('[onclick="switchCompChart(\'bar\')"]');
                if (defaultChartBtn) defaultChartBtn.classList.add('active');

                buildComparisonChart('bar');
                buildComparisonStats();

                document.getElementById('comparisonArea').style.display = 'block';
                document.getElementById('comparisonEmptyState').style.display = 'none';
            }
        } catch (error) {
            console.error('Error:', error);
        }
    }

    // ===== BUILD COMPARISON CHART =====
    function buildComparisonChart(type) {
        if (!window.comparisonData) return;

        const ctx = document.getElementById('comparisonChart');
        if (!ctx) return;
        if (comparisonChart && typeof comparisonChart.destroy === 'function') {
            comparisonChart.destroy();
        }

        const { s1, s2, label1, label2, metricCfg } = window.comparisonData;
        const labels = ['Total', 'Top Value', 'Average/Value'];
        const d1 = [s1.metricTotal, s1.topMetric, s1.avgMetric];
        const d2 = [s2.metricTotal, s2.topMetric, s2.avgMetric];

        if (type === 'bar') {
            comparisonChart = new Chart(ctx, {
                type: 'bar',
                data: {
                    labels,
                    datasets: [
                        {
                            label: `${label1} (${metricCfg.unit})`,
                            data: d1,
                            backgroundColor: 'rgba(102, 126, 234, 0.7)'
                        },
                        {
                            label: `${label2} (${metricCfg.unit})`,
                            data: d2,
                            backgroundColor: 'rgba(240, 147, 251, 0.7)'
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: { y: { beginAtZero: true } }
                }
            });
        } else if (type === 'radar') {
            comparisonChart = new Chart(ctx, {
                type: 'radar',
                data: {
                    labels,
                    datasets: [
                        {
                            label: `${label1} (${metricCfg.unit})`,
                            data: d1,
                            borderColor: '#667eea',
                            backgroundColor: 'rgba(102, 126, 234, 0.2)'
                        },
                        {
                            label: `${label2} (${metricCfg.unit})`,
                            data: d2,
                            borderColor: '#f093fb',
                            backgroundColor: 'rgba(240, 147, 251, 0.2)'
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    scales: { r: { beginAtZero: true } }
                }
            });
        } else if (type === 'pie') {
            comparisonChart = new Chart(ctx, {
                type: 'doughnut',
                data: {
                    labels: [label1, label2],
                    datasets: [{
                        data: [s1.metricTotal, s2.metricTotal],
                        backgroundColor: ['#667eea', '#f093fb']
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: { legend: { position: 'right' } }
                }
            });
        }
    }

    // ===== SWITCH COMPARISON CHART =====
    function switchCompChart(type) {
        if (!window.comparisonData) return;
        document.querySelectorAll('[onclick*="switchCompChart"]').forEach(btn => btn.classList.remove('active'));
        event.target.classList.add('active');
        buildComparisonChart(type);
    }

    // ===== BUILD COMPARISON STATS =====
    function buildComparisonStats() {
        if (!window.comparisonData) return;

        const { s1, s2, label1, label2, metricCfg } = window.comparisonData;
        const tbody = document.getElementById('comparisonStatsBody');

        const metricRows = [
            { name: 'Unique Values', v1: s1.uniqueValues, v2: s2.uniqueValues, decimals: 0 },
            { name: 'Total Records', v1: s1.totalRecords, v2: s2.totalRecords, decimals: 0 },
            { name: metricCfg.label, v1: s1.metricTotal, v2: s2.metricTotal, decimals: metricCfg.decimals },
            { name: `Top ${metricCfg.label}`, v1: s1.topMetric, v2: s2.topMetric, decimals: metricCfg.decimals },
            { name: `Avg ${metricCfg.label}/Value`, v1: s1.avgMetric, v2: s2.avgMetric, decimals: metricCfg.decimals }
        ];

        const rowsHtml = metricRows.map(row => {
            const v1 = Number(row.v1 || 0);
            const v2 = Number(row.v2 || 0);
            const diff = v1 - v2;
            const winner = v1 > v2 ? label1 : v1 < v2 ? label2 : 'Equal';
            const formattedV1 = v1.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: row.decimals });
            const formattedV2 = v2.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: row.decimals });
            const formattedDiff = diff.toLocaleString(undefined, { minimumFractionDigits: 0, maximumFractionDigits: row.decimals });
            return `<tr>
                <td><strong>${row.name}</strong></td>
                <td>${formattedV1}</td>
                <td>${formattedV2}</td>
                <td>${formattedDiff}</td>
                <td>${winner}</td>
            </tr>`;
        }).join('');

        const topValueNameRow = `<tr>
            <td><strong>Top Value Name</strong></td>
            <td>${s1.topValueName}</td>
            <td>${s2.topValueName}</td>
            <td>-</td>
            <td>-</td>
        </tr>`;

        tbody.innerHTML = rowsHtml + topValueNameRow;
    }

    // ===== LOAD ALL DAYS TIMELINE =====
    async function loadAllDaysTimeline() {
        try {
            const response = await fetch(`api.php?action=get_all_dates_statistics`);
            const result = await response.json();

            if (result.success) {
                const data = result.data;
                const totalOps = data.reduce((sum, d) => sum + d.total_records, 0);
                const totalEmps = data.reduce((sum, d) => sum + d.employee_count, 0);

                document.getElementById('timelineDays').textContent = data.length;
                document.getElementById('timelineOps').textContent = totalOps;
                document.getElementById('timelineEmps').textContent = totalEmps;
                document.getElementById('timelineAvg').textContent = (totalOps / data.length).toFixed(0);

                buildTimelineChart(data);
                buildAllDailyDetails(data);

                document.getElementById('timelineArea').style.display = 'block';
                document.getElementById('timelineEmptyState').style.display = 'none';
            }
        } catch (error) {
            console.error('Error:', error);
        }
    }

    // ===== BUILD ALL DAILY DETAILS =====
    async function buildAllDailyDetails(dates) {
        const container = document.getElementById('dailyDetailsContainer');
        container.innerHTML = '';
        const days = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];

        for (const dateObj of dates) {
            const date = new Date(dateObj.log_date);
            const dayName = days[date.getDay()];
            const dayCard = document.createElement('div');
            dayCard.className = 'chart-section';
            dayCard.innerHTML = `
                <div class="chart-title" style="margin-bottom: 20px;">
                    <i class="fas fa-calendar-day"></i> ${dateObj.log_date} (${dayName})
                </div>

                <div class="stats-grid" style="margin-bottom: 20px;">
                    <div class="stat-card blue">
                        <div class="stat-card-label">Operations</div>
                        <div class="stat-card-value">${dateObj.total_records}</div>
                        <div style="font-size: 12px; margin-top: 8px;">Production Records</div>
                    </div>
                    <div class="stat-card pink">
                        <div class="stat-card-label">Employees</div>
                        <div class="stat-card-value">${dateObj.employee_count}</div>
                        <div style="font-size: 12px; margin-top: 8px;">Active Workers</div>
                    </div>
                    <div class="stat-card cyan">
                        <div class="stat-card-label">Avg per Emp</div>
                        <div class="stat-card-value">${(dateObj.total_records / (dateObj.employee_count || 1)).toFixed(1)}</div>
                        <div style="font-size: 12px; margin-top: 8px;">Operations</div>
                    </div>
                    <div class="stat-card green">
                        <div class="stat-card-label">Tools</div>
                        <div class="stat-card-value">${dateObj.tools_count || '0'}</div>
                        <div style="font-size: 12px; margin-top: 8px;">Equipment Used</div>
                    </div>
                </div>

                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-top: 15px;" id="dailyTables-${dateObj.log_date}">
                    <!-- Tables will be inserted here -->
                </div>
            `;
            container.appendChild(dayCard);

            // Load detailed daily data
            loadDailyDetails(dateObj.log_date);
        }
    }

    // ===== LOAD DAILY DETAILS (Machines, Employees, Tools) =====
    async function loadDailyDetails(logDate) {
        try {
            const response = await fetch(`api.php?action=get_daily_production_details&log_date=${logDate}`);
            const result = await response.json();

            if (result.success) {
                const container = document.getElementById(`dailyTables-${logDate}`);
                if (!container) return;

                // Build Machines Table
                const machinesHtml = `
                    <div>
                        <h4 style="margin: 0 0 12px 0; color: #2c3e50; font-size: 14px; text-transform: uppercase; letter-spacing: 0.5px;"><i class="fas fa-cogs"></i> Machines/Tools</h4>
                        <table class="data-table" style="font-size: 13px;">
                            <thead>
                                <tr>
                                    <th>Machine/Tool</th>
                                    <th>Operations</th>
                                    <th>%</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${(result.machines && result.machines.length > 0) ? result.machines.map(m => `
                                    <tr>
                                        <td><strong>${m.machine_name || m.tool_name || 'N/A'}</strong></td>
                                        <td>${m.count}</td>
                                        <td>${((m.count / result.total_operations) * 100).toFixed(1)}%</td>
                                    </tr>
                                `).join('') : '<tr><td colspan="3" class="no-data">No data</td></tr>'}
                            </tbody>
                        </table>
                    </div>
                `;

                // Build Employees Table
                const employeesHtml = `
                    <div>
                        <h4 style="margin: 0 0 12px 0; color: #2c3e50; font-size: 14px; text-transform: uppercase; letter-spacing: 0.5px;"><i class="fas fa-users"></i> Employees</h4>
                        <table class="data-table" style="font-size: 13px;">
                            <thead>
                                <tr>
                                    <th>Employee</th>
                                    <th>Operations</th>
                                    <th>%</th>
                                </tr>
                            </thead>
                            <tbody>
                                ${(result.employees && result.employees.length > 0) ? result.employees.map(e => `
                                    <tr>
                                        <td><strong>${e.operator_name || 'Unknown'}</strong></td>
                                        <td>${e.count}</td>
                                        <td>${((e.count / result.total_operations) * 100).toFixed(1)}%</td>
                                    </tr>
                                `).join('') : '<tr><td colspan="3" class="no-data">No data</td></tr>'}
                            </tbody>
                        </table>
                    </div>
                `;

                container.innerHTML = machinesHtml + employeesHtml;
            }
        } catch (error) {
            console.error('Error loading daily details:', error);
        }
    }

    // ===== BUILD TIMELINE CHART =====
    function buildTimelineChart(data) {
        const ctx = document.getElementById('timelineChart');
        if (!ctx) return;

        if (timelineChart && typeof timelineChart.destroy === 'function') {
            timelineChart.destroy();
        }

        timelineChart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: data.map(d => d.log_date),
                datasets: [
                    {
                        label: 'Records',
                        data: data.map(d => d.total_records),
                        borderColor: '#00a65a',
                        backgroundColor: 'rgba(0, 166, 90, 0.1)',
                        borderWidth: 3,
                        tension: 0.4,
                        fill: true
                    },
                    {
                        label: 'Employees',
                        data: data.map(d => d.employee_count),
                        borderColor: '#f093fb',
                        backgroundColor: 'rgba(240, 147, 251, 0.1)',
                        borderWidth: 3,
                        tension: 0.4,
                        fill: true
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: { y: { beginAtZero: true } },
                plugins: { legend: { display: true } }
            }
        });
    }

    // ===== INITIALIZE =====
    document.addEventListener('DOMContentLoaded', function() {
        loadFields();
        const compareMetric = document.getElementById('compareMetric');
        if (compareMetric) {
            compareMetric.addEventListener('change', () => {
                if (window.comparisonData) compareFields();
            });
        }
    });

</script>

</body>
</html>
