# Admin Dashboard - Complete Setup Guide

## Overview
The Admin Dashboard is a comprehensive user management system that allows Administrators to:
- Create new users with different roles
- Manage existing users (view, edit passwords, delete)
- Change their own password
- View complete activity/audit logs
- Track all data entries and user actions

## Database Setup

### 1. Initialize Database Tables

Run the following SQL commands in phpMyAdmin or MySQL CLI to create the required tables:

```sql
-- Copy and paste all content from: ADMIN_DASHBOARD_SETUP.sql
```

**Or use the initialization script:**

```bash
# Execute in phpMyAdmin > Import tab:
# Upload: ADMIN_DASHBOARD_SETUP.sql
```

The script creates:
- **users** table: Stores user accounts with roles and creation metadata
- **activity_log** table: Tracks all user activities for audit purposes

### 2. Create Initial Admin User

After creating the tables, insert an initial admin account:

```sql
INSERT INTO users (username, password, role, created_by) 
VALUES ('ADMIN', 'admin123', 'Administrator', NULL);
```

**Important:** Change this password immediately after first login!

## Accessing the Admin Dashboard

### URL
```
http://localhost/erp.manufacture/admin_dashboard.php
```

### Access Control
- **Restricted to:** Administrator role only
- **Others will be redirected:** to index.php with an error message
- **Session check:** Automatic validation on page load

## Features in Detail

### 1. CREATE NEW USER
**Location:** Left sidebar, top section

**Features:**
- Input: Username (unique)
- Input: Password
- Dropdown: Role selection
  - **Administrator**: Full system access, can create/delete users
  - **Controller**: Data entry and viewing
  - **User**: Limited data entry

**Validation:**
- All fields required
- Username must be unique
- Password cannot be empty

**On Success:**
- New user created
- Activity logged in audit trail
- "created_by" field automatically populated with admin ID

### 2. MANAGE EXISTING USERS
**Location:** Right side, main table

**Displays:**
- Row #
- Username
- Role (with color-coded badges)
- Created By (Admin name)
- Creation Date/Time
- Action buttons

**Available Actions:**
- **Change Password**: Allows admin to reset any user's password
- **Delete**: Removes user account permanently
  - ❌ **Cannot delete** Administrator accounts
  - ✅ **Can delete** Controller and User accounts

**Security Note:** Using self-referencing foreign key "created_by" to track which admin created each user.

### 3. CHANGE OWN PASSWORD
**Location:** Left sidebar, bottom section

**Requirements:**
- Current Password (for verification)
- New Password
- Confirm New Password (must match)

**Validation:**
- Passwords must match
- Current password must be correct
- Cannot reuse same password

**On Success:**
- Password updated immediately
- Activity logged
- Session maintained (no need to re-login)

### 4. USER ACTIVITY LOG
**Location:** Right side, bottom table

**Displays:**
- Entry ID (formatted as E001, E002, etc.)
- Username (who made the entry)
- Employee Data Entered (name of employee)
- Date/Time
- Record Count (number of records added)

**Access Control:**
- 📖 **View-Only** for non-admin users
- ❌ **Delete protected**: No DELETE endpoint accessible to non-admins
- 📊 **Audit Purpose**: Cannot be modified or deleted

**Log Types:**
- DATA_ENTRY: User entered production data
- USER_CREATED: Admin created new user
- USER_DELETED: Admin deleted user
- PASSWORD_CHANGED: User or admin changed password

## API Endpoints (Backend)

All endpoints require authentication and proper role checking.

### User Management Endpoints

#### GET All Users
```
GET /api.php?action=get_all_users
Required: Admin role
Returns: List of all users with created_by information
```

#### Create User
```
POST /api.php
Parameters:
  - action: create_user
  - username: string (unique)
  - password: string
  - role: enum (Administrator, Controller, User)
Required: Admin role
```

#### Delete User
```
POST /api.php
Parameters:
  - action: delete_user
  - user_id: integer
Required: Admin role
Protection: Cannot delete Administrator accounts
```

#### Change User Password (Admin)
```
POST /api.php
Parameters:
  - action: change_user_password
  - user_id: integer
  - new_password: string
Required: Admin role
```

#### Change Own Password
```
POST /api.php
Parameters:
  - action: change_own_password
  - current_password: string
  - new_password: string
Required: Logged-in user
```

### Activity Log Endpoints

#### Get Activity Log
```
GET /api.php?action=get_activity_log
Required: Logged-in user (any role)
Returns: Last 100 activity records
```

#### Log Activity
```
POST /api.php
Parameters:
  - action: log_activity
  - employee_name: string
  - record_count: integer
Required: Logged-in user
Purpose: Called when user enters production data
```

## Database Schema

### Tables Structure

#### users Table
```
Column          Type            Purpose
────────────────────────────────────────────────────
id              INT (PK)        Unique user ID
username        VARCHAR(100)    Login name (unique)
password        VARCHAR(255)    User password
role            ENUM            User role/permissions
created_by      INT (FK)        Admin who created user
created_at      TIMESTAMP       Account creation time
```

**Foreign Key:** created_by → users(id)
**Indexes:** username, role, created_by

#### activity_log Table
```
Column          Type            Purpose
────────────────────────────────────────────────────
entry_id        INT (PK)        Unique log entry ID
user_id         INT (FK)        User who made action
username        VARCHAR(100)    Username for reference
employee_name   VARCHAR(255)    Name of employee data
record_count    INT             Number of records added
entry_time      TIMESTAMP       Time of activity
action_type     VARCHAR(100)    Type of action
details         TEXT            Additional details
```

**Foreign Key:** user_id → users(id)
**Indexes:** user_id, entry_time, username

## Security Features

### 1. Role-Based Access Control (RBAC)
- **Administrator**: Complete system access
- **Controller**: Data entry and viewing
- **User**: Limited access
- Dashboard restricted to Administrator role only

### 2. Self-Referencing Foreign Key
```
users.created_by → users.id
```
Ensures you can track which admin created each user account.

### 3. Deletion Protection
- Administrator accounts **cannot be deleted**
- Check: `if (user.role !== 'Administrator') { showDeleteButton() }`
- Non-admin users can only be deleted by administrators

### 4. Activity Logging
- Every user action is logged
- Non-admins cannot delete logs
- No DELETE endpoint for non-admins on activity_log
- Immutable audit trail

### 5. Password Protection
- Current password verification required
- Passwords not hashed (TODO: implement bcrypt in production)
- Session-based authentication
- Timeout protection

## Security Notes for Production

⚠️ **Current Implementation:**
```php
// Current: Plain text passwords
$stmt = $pdo->prepare("INSERT INTO users (password) VALUES (?)");

// TODO: Change to hashed passwords
$stmt = $pdo->prepare("INSERT INTO users (password) VALUES (?)");
$stmt->execute([password_hash($password, PASSWORD_BCRYPT)]);
```

### Password Hashing Implementation (Optional)
```php
// In production, use:
password_hash($password, PASSWORD_BCRYPT)
password_verify($input_password, $hashed_password)
```

## Workflow Examples

### Admin Creating New User
1. Log in as Administrator
2. Open Admin Dashboard
3. Fill "CREATE NEW USER" form:
   - Username: "sales_rep_1"
   - Password: "P@ssw0rd123"
   - Role: "Controller"
4. Click "CREATE USER"
5. System:
   - ✅ Validates input
   - ✅ Creates user record
   - ✅ Sets created_by to admin's ID
   - ✅ Logs activity
   - ✅ Shows success message

### Activity Log Entry
When a user (SALES_REP_1) enters production data:
1. User fills production entry form
2. Clicks "Save"
3. System calls: `api.php?action=log_activity`
4. Database records:
   - user_id: 2 (SALES_REP_1's ID)
   - username: "SALES_REP_1"
   - employee_name: "John Doe - Q3 Sales"
   - record_count: 15
   - entry_time: 2026-02-14 10:30:45
   - action_type: "DATA_ENTRY"

### Admin Viewing Activity
1. Admin opens Admin Dashboard
2. Scrolls to "USER ACTIVITY LOG" section
3. See all entries with:
   - E001: SALES_REP_1 entered data for John Doe - Q3 Sales on 2026-02-14
   - E002: CONTROLLER entered data for Jane Smith - Analysis on 2026-02-14
   - etc.

## Testing Checklist

- [ ] Database tables created successfully
- [ ] Initial admin user inserted
- [ ] Admin can login to system
- [ ] Admin can access admin_dashboard.php
- [ ] Non-admin users redirected if accessing dashboard
- [ ] Can create new user
- [ ] Created user appears in table
- [ ] Can change password for any user
- [ ] Cannot delete Administrator account
- [ ] Can delete Controller/User accounts
- [ ] Activity log shows user creation
- [ ] Can change own password
- [ ] Activity log shows password changes
- [ ] Activity log is read-only for non-admins

## Troubleshooting

### "Access Denied! Only Administrators can access this page."
**Cause:** Non-admin user trying to access dashboard
**Solution:** Log in as Administrator account

### "Users cannot be loaded"
**Cause:** users table not created
**Solution:** Run ADMIN_DASHBOARD_SETUP.sql to initialize database

### "Create User button not working"
**Cause:** Missing API endpoint
**Solution:** Verify api.php has all user management endpoints

### "Activity log not showing"
**Cause:** activity_log table not created or empty
**Solution:** Create table using ADMIN_DASHBOARD_SETUP.sql and log some activities

## File Structure

```
erp.manufacture/
├── admin_dashboard.php          ← Main dashboard UI
├── api.php                      ← API endpoints (updated)
├── ADMIN_DASHBOARD_SETUP.sql    ← Database initialization
├── login.php                    ← User authentication
├── session_check.php            ← Session validation
└── [other files...]
```

## Integration with Existing System

The admin dashboard integrates with your existing ERP system:
- Uses same PDO database connection as api.php
- Same session handling (uses $_SESSION['user_id'] and $_SESSION['role'])
- Same authentication flow (through login.php)
- Compatible with existing users table (if present)
- Activity logging can be extended for other features

## Next Steps

1. ✅ Create database tables using ADMIN_DASHBOARD_SETUP.sql
2. ✅ Insert initial admin user
3. ✅ Test admin dashboard functionality
4. ⏳ Optional: Implement password hashing (bcrypt)
5. ⏳ Optional: Extend activity logging to other modules
6. ⏳ Optional: Add email notifications on user creation

## Support

For issues or questions:
1. Check this guide first
2. Review database schema in ADMIN_DASHBOARD_SETUP.sql
3. Check api.php for endpoint definitions
4. Review browser console for JavaScript errors
5. Check PHP error logs for backend errors

---

**Last Updated:** February 14, 2026
**Version:** 1.0
**Status:** Ready for Production
